//===========================================================================
// @(#) $Name$
// @(#) $Id: DwmMcBlockDbEntry.cc 9240 2017-04-25 17:21:20Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2015
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcBlockDbEntry.cc
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

#include <iomanip>
#include <fstream>
#include <mutex>
#include <regex>
#include <sstream>
#include <vector>
#include <utility>

#include <json/json.h>

#include "DwmCvsTag.hh"
#include "DwmDateTime.hh"
#include "DwmSysLogger.hh"

#include "DwmMcBlockDbEntry.hh"

static const Dwm::CvsTag cvstag("@(#) $Name$ $Id: DwmMcBlockDbEntry.cc 9240 2017-04-25 17:21:20Z dwm $");

using namespace std;

namespace Dwm {

  namespace McBlock {

    Ipv4Routes<string>  DbEntry::_defaultRegistryRoutes;
    CountryCodes        DbEntry::_countryCodes;
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool DbEntry::LoadDefaultRegistryRoutes()
    {
      static mutex  mtx;
      
      static const vector<pair<regex,string>>  nicMap = {
        { regex(".+afrinic.net.+", regex::ECMAScript|regex::optimize),
          "AFRINIC" },
        { regex(".+apnic.net.+", regex::ECMAScript|regex::optimize),
          "APNIC" },
        { regex(".+arin.net.+", regex::ECMAScript|regex::optimize),
          "ARIN" },
        { regex(".+ripe.net.+", regex::ECMAScript|regex::optimize),
          "RIPE" },
        { regex(".+lacnic.net.+", regex::ECMAScript|regex::optimize),
          "LACNIC" },
        { regex("multicast", regex::ECMAScript|regex::optimize),
          "mcast" },
        { regex("future", regex::ECMAScript|regex::optimize),
          "future" }
      };

      //  Load from /usr/local/etc/rdap_ipv4.json, which is part of
      //  libDwmRDAP.  This is easier (but still painful) to maintain than
      //  a compiled-in structure.  IANA has done a poor job of updating
      //  their 'bootstrap' file, so I maintain my own that I correct when
      //  I see a mismapping.
      Json::Value   jv;
      Json::Reader  jr;
      ifstream  is("/usr/local/etc/rdap_ipv4.json");
      if (is) {
        lock_guard<mutex>  lock(mtx);
        smatch  sm;
        jr.parse(is, jv);
        for (int s = 0; s < jv["services"].size(); ++s) {
          string  nicName;
          for (int rds = 0; rds < jv["services"][s][1].size(); ++rds) {
            nicName = "";
            for (auto rxm : nicMap) {
              string  rdsUrl(jv["services"][s][1][rds].asString());
              if (regex_search(rdsUrl, sm, rxm.first)) {
                nicName = rxm.second;
                break;
              }
            }
          }
          if (! nicName.empty()) {
            for (int n = 0; n < jv["services"][s][0].size(); ++n) {
              Ipv4Prefix  pfx(jv["services"][s][0][n].asString());
              _defaultRegistryRoutes[pfx] = nicName;
            }
          }
        }
        is.close();
      }
      else {
        Syslog(LOG_ERR, "Failed to open /usr/local/etc/rdap_ipv4.json");
      }
      return (! _defaultRegistryRoutes.Empty());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string DbEntry::GetDefaultRegistry(const Ipv4Prefix & prefix)
    {
      pair<Ipv4Prefix,string>  rc;
      if (_defaultRegistryRoutes.Empty()) {
        LoadDefaultRegistryRoutes();
      }
      _defaultRegistryRoutes.FindLongest(prefix.Network(), rc);
      return rc.second;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::string DbEntry::GetCountryName() const
    {
      return GetCountryName(std::get<3>(_data));
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string DbEntry::GetCountryName(const string & countryCode)
    {
      string  rc;
      if (_countryCodes.Empty()) {
        _countryCodes.ReadJson("/usr/local/etc/country_codes.json");
      }
      if (! _countryCodes.Empty() && (! countryCode.empty())) {
        rc = _countryCodes.FindByCode(countryCode).Name();
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    DbEntry::DbEntry()
        : _data()
    {
      get<2>(_data) = "";
      get<3>(_data) = "";
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    DbEntry::DbEntry(const Ipv4Prefix & prefix)
    {
      get<0>(_data) = prefix;
      get<1>(_data).Start(TimeValue(true));
      TimeValue  endTime(true);
      endTime.Set(endTime.Secs() + (30 * 24 * 60 * 60), 0);
      get<1>(_data).End(endTime);
      get<2>(_data) = GetDefaultRegistry(prefix.Network());
      get<3>(_data) = "";
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    DbEntry::DbEntry(const Ipv4Prefix & prefix, const TimeInterval & interval,
                     const string & registry, const string & country)
    {
      get<0>(_data) = prefix;
      get<1>(_data) = interval;
      if (! registry.empty()) {
        get<2>(_data) = registry;
      }
      else {
        get<2>(_data) = GetDefaultRegistry(prefix.Network());
      }
      get<3>(_data) = country;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool DbEntry::IsActive() const
    {
      TimeValue now(true);
      return get<1>(_data).Contains(now);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static string FormatTimeInterval(const TimeInterval & ti)
    {
      DateTime  st(ti.Start()), et(ti.End());
      string    rc(st.Formatted("%Y/%m/%d"));
      rc += " - ";
      rc += et.Formatted("%Y/%m/%d");
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string DbEntry::DurationString() const
    {
      ostringstream  os;
      TimeValue  duration = get<1>(_data).Duration();
      uint64_t  days = (duration.Secs() + 1) / (24 * 60 * 60);
      uint64_t  years = days / 365;
      if (years) {
        os << years << "y";
      }
      os << days % 365 << "d";
      return os.str();
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & operator << (ostream & os, const DbEntry & dbe)
    {
      if (os) {
        ostringstream  oss;
        oss << setiosflags(ios::left) << setw(18)
            << get<0>(dbe._data).ToShortString() << " "
            << FormatTimeInterval(get<1>(dbe._data)) << " ";
        if (! get<2>(dbe._data).empty()) {
          oss << setw(8) << get<2>(dbe._data);
        }
        else {
          oss << setw(8) << DbEntry::GetDefaultRegistry(get<0>(dbe._data));
        }
        oss << " " << get<3>(dbe._data);
        os << oss.str();
      }
      return os;
    }
    
  }  // namespace McBlock

}  // namespace Dwm
