//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmMcBlockRequestMessage.hh 9686 2017-06-30 08:10:06Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcBlockRequestMessage.hh
//!  \brief Dwm::McBlock::RequestMessage class definition
//---------------------------------------------------------------------------

#include <string>
#include <vector>
#include <json/json.h>

#include "DwmIpv4Prefix.hh"
#include "DwmTimeValue64.hh"

namespace Dwm {

  namespace McBlock {

    //------------------------------------------------------------------------
    //!  Convenience class to create an 'activate' request for mcblockd.
    //------------------------------------------------------------------------
    class ActivateReq : public Json::Value
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct an 'activate' request for the given pf @c table and
      //!  given @c prefix.
      //----------------------------------------------------------------------
      ActivateReq(const std::string & table,
                  const std::vector<Ipv4Prefix> & prefixes);
    };

    //------------------------------------------------------------------------
    //!  Convenience class to create a 'deactivate' request for mcblockd.
    //------------------------------------------------------------------------
    class DeactivateReq : public Json::Value
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct a 'deactivate' request for the given pf @c table and
      //!  given @c prefix.
      //----------------------------------------------------------------------
      DeactivateReq(const std::string & table,
                    const std::vector<Ipv4Prefix> & prefixes);
    };

    //------------------------------------------------------------------------
    //!  Convenience class to create a 'search' request for mcblockd.
    //------------------------------------------------------------------------
    class SearchReq : public Json::Value
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct a 'search' request for the given @c addr.
      //----------------------------------------------------------------------
      SearchReq(const Ipv4Address & addr);
    };
    
    //------------------------------------------------------------------------
    //!  Convenience class to create a 'getActive' request for mcblockd.
    //------------------------------------------------------------------------
    class GetActiveReq : public Json::Value
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct a 'getActive' request for the given pf @c table.
      //----------------------------------------------------------------------
      GetActiveReq(const std::string & table);
    };

    //------------------------------------------------------------------------
    //!  Convenience class to create a 'getAddRules' request for mcblockd.
    //------------------------------------------------------------------------
    class GetAddRulesReq : public Json::Value
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct a 'getAddRules' request for the given pf @c table.
      //----------------------------------------------------------------------
      GetAddRulesReq(const std::string & table);
    };
    
    //------------------------------------------------------------------------
    //!  NOT IMPLEMENTED
    //------------------------------------------------------------------------
    class GetMatchesReq : public Json::Value
    {
    public:
      GetMatchesReq(const std::string & table, const Ipv4Prefix & prefix);
    };

    //------------------------------------------------------------------------
    //!  Convenience class to create a 'logHit' request for mcblockd.
    //------------------------------------------------------------------------
    class LogHitReq : public Json::Value
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct a 'logHit' request for the given pf @c table, given
      //!  @c addr and given log timestamp @c logTime.
      //----------------------------------------------------------------------
      LogHitReq(const std::string & table, const Ipv4Address & addr,
                const TimeValue64 & logTime);
    };

    //------------------------------------------------------------------------
    //!  Convenience class to create a 'get' request for mcblockd.
    //------------------------------------------------------------------------
    class GetReq : public Json::Value
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct a 'get' request for the given pf @c table, given
      //!  @c addr.
      //----------------------------------------------------------------------
      GetReq(const std::string & table, const Ipv4Address & addr);
    };
    
    //------------------------------------------------------------------------
    //!  Encapsulates a request message for mcblockd.
    //------------------------------------------------------------------------
    class RequestMessage
    {
    public:
      //----------------------------------------------------------------------
      //!  Default constructor
      //----------------------------------------------------------------------
      RequestMessage();
      
      //----------------------------------------------------------------------
      //!  Construct from a Json::Value reference
      //----------------------------------------------------------------------
      RequestMessage(const Json::Value & req);
      
      //----------------------------------------------------------------------
      //!  Construct from a Json::Value rvalue reference.
      //----------------------------------------------------------------------
      RequestMessage(const Json::Value && req);
      
      //----------------------------------------------------------------------
      //!  Reads the message from a descriptor using the given AES128
      //!  @c secret for decryption.  Returns the number of bytes read on
      //!  success, -1 on failure.
      //----------------------------------------------------------------------
      bool Read(int fd, const std::string & secret);
      
      //----------------------------------------------------------------------
      //!  Writes the message to a descriptor using the given AES128
      //!  @c secret for encryption.  Returns the number of bytes written on
      //!  success, -1 on failure.
      //----------------------------------------------------------------------
      bool Write(int fd, const std::string & secret) const;

      //----------------------------------------------------------------------
      //!  Returns a const reference to the contained Json::Value
      //!  representation of the message.
      //----------------------------------------------------------------------
      const Json::Value & Json() const;
      
    private:
      Json::Value  _value;
    };
    
  }  // namespace McBlock

}  // namespace Dwm
