//===========================================================================
// @(#) $Name$
// @(#) $Id: DwmMcBlockIpv4Routes.hh 9147 2017-04-18 23:11:43Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 1999-2005
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//    endorse or promote products derived from this software without
//    specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

#ifndef _DWMMCBLOCKIPV4ROUTES_HH_
#define _DWMMCBLOCKIPV4ROUTES_HH_

#include <algorithm>
#include <initializer_list>
#include <map>
#include <shared_mutex>
#include <vector>

#include "DwmPortability.hh"

#include "DwmIO.hh"
#include "DwmGZIO.hh"
#include "DwmBZ2IO.hh"
#include "DwmIpv4Prefix.hh"
#include "DwmOperators.hh"
#include "DwmReadable.hh"
#include "DwmWritable.hh"
#include "DwmGZReadable.hh"
#include "DwmGZWritable.hh"
#include "DwmBZ2Readable.hh"
#include "DwmBZ2Writable.hh"

namespace Dwm {

  namespace McBlock {

    //------------------------------------------------------------------------
    //!  This template class provides an associative container keyed by
    //!  IPv4 addresses, with longest-match searching.  While this isn't
    //!  as speedy to search as Patricia or radix, it is based on STL
    //!  containers and is hence easy to understand and maintain (note how 
    //!  few lines of actual code are here).
    //!
    //!  I/O functionality is provided, but the real work there is done
    //!  in the Dwm::IO class.
    //!
    //!  Note that the difference between this class template and
    //!  Dwm::Ipv4Routes is the container used per prefix length.  Here
    //!  I used map instead of unordered_map, because I needed ordering in
    //!  derived classes so that I could more easily support coalescing of
    //!  adjacent prefixes.
    //------------------------------------------------------------------------
    template <typename _valueT>
    class Ipv4Routes
      : public Readable, public Writable, 
        public GZReadable, public GZWritable,
        public BZ2Readable, public BZ2Writable
    {
    public:
      typedef std::map<Ipv4Address, _valueT> _RepSubType;
      typedef typename _RepSubType::const_iterator  const_iterator;
      
      //----------------------------------------------------------------------
      //!  Constructor
      //----------------------------------------------------------------------
      Ipv4Routes<_valueT>()
          : _hashMaps(33)
      {
      }

      Ipv4Routes<_valueT>(std::initializer_list<std::pair<Ipv4Prefix,_valueT>> il)
          : _hashMaps(33)
      {
        for (auto i = il.begin(); i != il.end(); ++i) {
          Add(i->first, i->second);
        }
      }
      
      //----------------------------------------------------------------------
      //!  Clears all entries.
      //----------------------------------------------------------------------
      void Clear()
      {
        std::lock_guard<std::shared_mutex>  lock(_mtx);
        for (uint8_t i = 0; i < 33; ++i)
          _hashMaps[i].clear();
        return;
      }
      
      //----------------------------------------------------------------------
      //!  Returns true if there are no entries.
      //----------------------------------------------------------------------
      bool Empty()
      {
        bool  rc = true;
        std::shared_lock<std::shared_mutex>  lock(_mtx);
        for (uint8_t i = 0; i < 33; ++i) {
          if (! _hashMaps[i].empty()) {
            rc = false;
            break;
          }
        }
        return(rc);
      }

      //----------------------------------------------------------------------
      //!  Adds an entry.  Returns false (and does nothing) if there was
      //!  already an entry present for \c prefix.
      //----------------------------------------------------------------------
      bool Add(const Ipv4Prefix & prefix, const _valueT & value) 
      {
        bool  rc = false;
        std::lock_guard<std::shared_mutex>  lock(_mtx);
        typename _RepSubType::iterator  iter = 
          _hashMaps[prefix.MaskLength()].find(prefix.Network());
        if (iter == _hashMaps[prefix.MaskLength()].end()) {
          _hashMaps[prefix.MaskLength()][prefix.Network()] = value;
          rc = true;
        }
        return(rc);
      }
      
      //----------------------------------------------------------------------
      //!  operator [] works like you would expect from an STL map.
      //----------------------------------------------------------------------
      _valueT & operator [] (const Ipv4Prefix & prefix)
      {
        std::lock_guard<std::shared_mutex>  lock(_mtx);
        return(_hashMaps[prefix.MaskLength()][prefix.Network()]);
      }
      
      //----------------------------------------------------------------------
      //!  Deletes the entry for \c prefix.  Returns true on success, false
      //!  if an entry wasn't found for \c prefix.
      //----------------------------------------------------------------------
      bool Delete(const Ipv4Prefix & prefix)
      {
        bool  rc = false;
        std::lock_guard<std::shared_mutex>  lock(_mtx);
        typename _RepSubType::iterator  iter = 
          _hashMaps[prefix.MaskLength()].find(prefix.Network());
        if (iter != _hashMaps[prefix.MaskLength()].end()) {
          _hashMaps[prefix.MaskLength()].erase(iter);
          rc = true;
        }
        return(rc);
      }
      
      //----------------------------------------------------------------------
      //!  Find the entry for the given \c prefix.  If an entry is found,
      //!  the value is stored in \c match and true is returned.  Else false
      //!  is returned.
      //----------------------------------------------------------------------
      bool Find(const Ipv4Prefix & prefix, _valueT & match) const
      {
        bool  rc = false;
        std::shared_lock<std::shared_mutex>  lock(_mtx);
        if (! _hashMaps[prefix.MaskLength()].empty()) {
          typename _RepSubType::const_iterator  iter =
            _hashMaps[prefix.MaskLength()].find(prefix.Network());
          if (iter != _hashMaps[prefix.MaskLength()].end()) {
            match = iter->second;
            rc = true;
          }
        }
        return(rc);
      }
      
      //----------------------------------------------------------------------
      //!  Finds the longest match for \c ipAddr.  Places the result in
      //!  \c match and returns true on success.  Returns false if no
      //!  match was found for \c ipAddr.
      //----------------------------------------------------------------------
      bool FindLongest(const Ipv4Address & ipAddr,
                       std::pair<Ipv4Prefix,_valueT> & match) const
      {
        bool  rc = false;
        
        Ipv4Prefix   lp(ipAddr, 32);
        
        typename _RepSubType::const_iterator  iter;
        for (int8_t i = 32; i >= 0; --i) {
          std::shared_lock<std::shared_mutex>  lock(_mtx);
          if (_hashMaps[i].empty())
            continue;
          lp.MaskLength(i);
          iter = _hashMaps[i].find(lp.Network());
          if (iter != _hashMaps[i].end()) {
            match.first = lp;
            match.second = iter->second;
            rc = true;
            break;
          }
        }
        return(rc);
      }
      
      //----------------------------------------------------------------------
      //!  Finds the longest match for \c ipAddr.  Places the result in
      //!  \c match and returns true on success.  Returns false if no
      //!  match was found for \c ipAddr.
      //!  Note that match.second is a pointer to const for the value stored
      //!  under the prefix.  That means you need to be careful using this
      //!  member; don't call free() or delete() on match.second.
      //----------------------------------------------------------------------
      bool
      FindLongest(const Ipv4Address & ipAddr,
                  std::pair<Ipv4Prefix, const _valueT *> & match) const
      {
        bool  rc = false;
        Ipv4Prefix   lp(ipAddr, 32);
        typename _RepSubType::const_iterator  iter;
        for (int8_t i = 32; i >= 0; --i) {
          std::shared_lock<std::shared_mutex>  lock(_mtx);
          if (_hashMaps[i].empty())
            continue;
          lp.MaskLength(i);
          iter = _hashMaps[i].find(lp.Network());
          if (iter != _hashMaps[i].end()) {
            match.first = lp;
            match.second = &(iter->second);
            rc = true;
            break;
          }
        }
        return(rc);
      }
      
      //----------------------------------------------------------------------
      //!  Finds all matches for \c ipAddr.  Places the results in \c matches
      //!  (in longest-match-first order) and returns true if any matches were 
      //!  found.  Returns false if no matches were found.
      //----------------------------------------------------------------------
      bool Find(const Ipv4Address & ipAddr,
                std::vector<std::pair<Ipv4Prefix,_valueT> > & matches) const
      {
        if (! matches.empty())
          matches.clear();
        
        typename _RepSubType::const_iterator  iter;
        
        for (int8_t i = 32; i >= 0; --i) {
          std::shared_lock<std::shared_mutex>  lock(_mtx);
          if (_hashMaps[i].empty())
            continue;
          Ipv4Prefix  prefix(ipAddr, i);
          iter = _hashMaps[i].find(prefix.Network());
          if (iter != _hashMaps[i].end()) {
            std::pair<Ipv4Prefix,_valueT> match(prefix, iter->second);
            matches.push_back(match);
          }
        }
        return(! matches.empty());
      }
      
      //----------------------------------------------------------------------
      //!  operator ==
      //!  It's unlikely you'd ever need to use this, and it's expensive.
      //!  It's mainly here for unit testing.
      //----------------------------------------------------------------------
      bool operator == (const Ipv4Routes<_valueT> & r) const
      {
        for (int8_t i = 32; i >= 0; --i) {
          std::shared_lock<std::shared_mutex>  lock(_mtx);
          std::shared_lock<std::shared_mutex>  r_lock(r._mtx);
          if (_hashMaps[i] != r._hashMaps[i])
            return(false);
        }
        return(true);
      }
      
      //----------------------------------------------------------------------
      //!  operator !=
      //!  It's unlikely you'd ever need to use this, and it's expensive.
      //!  It's mainly here for unit testing.
      //----------------------------------------------------------------------
      bool operator != (const Ipv4Routes<_valueT> & r) const
      {
        return(! (*this == r));
      }
      
      //----------------------------------------------------------------------
      //!  Returns the number of routes.
      //----------------------------------------------------------------------
      uint32_t Size() const
      {
        uint32_t  rc = 0;
        for (uint8_t i = 0; i < 33; ++i) {
          std::shared_lock<std::shared_mutex>  lock(_mtx);
          rc += _hashMaps[i].size();
        }
        return(rc);
      }
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      void HashSizes(std::vector<std::pair<uint8_t, uint32_t> > & sizes) const
      {
        if (! sizes.empty())
          sizes.clear();
        for (uint8_t i = 0; i < 33; ++i) {
          std::shared_lock<std::shared_mutex>  lock(_mtx);
          if (! _hashMaps[i].empty()) {
            sizes.push_back(std::pair<uint8_t,uint32_t>(i,_hashMaps[i].size()));
          }
        }
        return;
      }
    
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      uint32_t StreamedLength() const
      {
        std::shared_lock<std::shared_mutex>  lock(_mtx);
        return(IO::StreamedLength(_hashMaps));
      }
      
      //----------------------------------------------------------------------
      //!  Reads the routes from an istream.  Returns the istream.
      //----------------------------------------------------------------------
      std::istream & Read(std::istream & is)
      {
        std::lock_guard<std::shared_mutex>  lock(_mtx);
        return(IO::Read(is, _hashMaps));
      }
      
      //----------------------------------------------------------------------
      //!  Writes the routes to an ostream.  Returns the ostream.
      //----------------------------------------------------------------------
      std::ostream & Write(std::ostream & os) const
      {
        std::shared_lock<std::shared_mutex>  lock(_mtx);
        return(IO::Write(os, _hashMaps));
      }
      
      //----------------------------------------------------------------------
      //!  Reades the routes from a FILE pointer.  Returns 1 on success,
      //!  0 on failure.
      //----------------------------------------------------------------------
      size_t Read(FILE *f)
      {
        std::lock_guard<std::shared_mutex>  lock(_mtx);
        return(IO::Read(f, _hashMaps));
      }
      
      //----------------------------------------------------------------------
      //!  Writes the routes to a FILE pointer.  Returns 1 on success,
      //!  0 on failure.
      //----------------------------------------------------------------------
      size_t Write(FILE *f) const
      {
        std::shared_lock<std::shared_mutex>  lock(_mtx);
        return(IO::Write(f, _hashMaps));
      }
      
      //----------------------------------------------------------------------
      //!  Reads the routes from a file descriptor.  Returns the number of
      //!  bytes read on success, -1 on failure.
      //----------------------------------------------------------------------
      ssize_t Read(int fd)
      {
        std::lock_guard<std::shared_mutex>  lock(_mtx);
        return(IO::Read(fd, _hashMaps));
      }
      
      //----------------------------------------------------------------------
      //!  Writes the routes to a file descriptor.  Returns the number of
      //!  bytes written on success, -1 on failure.
      //----------------------------------------------------------------------
      ssize_t Write(int fd) const
      {
        std::shared_lock<std::shared_mutex>  lock(_mtx);
        return(IO::Write(fd, _hashMaps));
      }
      
      //----------------------------------------------------------------------
      //!  Reads the routes from a gzFile.  Returns the number of
      //!  bytes read on success, -1 on failure.
      //----------------------------------------------------------------------
      int Read(gzFile gzf)
      {
        std::lock_guard<std::shared_mutex>  lock(_mtx);
        return(GZIO::Read(gzf, _hashMaps));
      }
      
      //----------------------------------------------------------------------
      //!  Writes the routes to a gzFile.  Returns the number of bytes
      //!  written on success, -1 on failure.
      //----------------------------------------------------------------------
      int Write(gzFile gzf) const
      {
        std::shared_lock<std::shared_mutex>  lock(_mtx);
        return(GZIO::Write(gzf, _hashMaps));
      }
      
      //----------------------------------------------------------------------
      //!  Reads the routes from a BZFILE pointer.  Returns the number of
      //!  bytes read on success, -1 on failure.
      //----------------------------------------------------------------------
      int BZRead(BZFILE *bzf)
      {
        std::lock_guard<std::shared_mutex>  lock(_mtx);
        return(BZ2IO::BZRead(bzf, _hashMaps));
      }
      
      //----------------------------------------------------------------------
      //!  Writes the routes to a BZFILE pointer.  Returns the number of bytes
      //!  written on success, -1 on failure.
      //----------------------------------------------------------------------
      int BZWrite(BZFILE *bzf) const
      {
        std::shared_lock<std::shared_mutex>  lock(_mtx);
        return(BZ2IO::BZWrite(bzf, _hashMaps));
      }
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      void SortByKey(std::vector<std::pair<Ipv4Prefix,_valueT>> & target,
                     bool ascending = true) const
      {
        if (! target.empty())
          target.clear();
        if (! this->_hashMaps.empty()) {
          target.resize(this->Size());
          uint32_t  pfx = 0;
          for (uint8_t hashNum = 0; hashNum < 33; ++hashNum) {
            std::shared_lock<std::shared_mutex>  lock(_mtx);
            if (! this->_hashMaps[hashNum].empty()) {
              typename _RepSubType::const_iterator  hiter = 
                this->_hashMaps[hashNum].begin();
              for ( ; hiter != _hashMaps[hashNum].end(); ++hiter) {
                target[pfx].first = Ipv4Prefix(hiter->first, hashNum);
                target[pfx].second = hiter->second;
                ++pfx;
              }
            }
          }
          if (! target.empty()) {
            if (ascending) {
              std::sort(target.begin(), target.end(), KeyLess());
            }
            else {
              std::sort(target.begin(), target.end(), KeyGreater());
            }
          }
        }
        
        return;
      }
      
      //----------------------------------------------------------------------
      //!  Sorts the contained pair<Ipv4Prefix,_valueT> values into a
      //!  vector, in descending order by the value stored for each prefix.
      //!  For example, if you had an Ipv4Routes<uint32_t> object,
      //!  \c target would contain the pair<Ipv4Prefix,_valueT> objects
      //!  sorted in descending order by the uint32_t values.
      //----------------------------------------------------------------------
      void SortByValue(std::vector<std::pair<Ipv4Prefix,_valueT> > & target)
      {
        if (! target.empty())
          target.clear();
        std::shared_lock<std::shared_mutex>  lock(_mtx);
        if (! this->_hashMaps.empty()) {
          target.resize(this->Size());
          typename std::vector<_RepSubType>::const_iterator  iter = 
            this->_hashMaps.begin();
          uint32_t  pfx = 0;
          for (uint8_t hashNum = 0; hashNum < 33; ++hashNum) {
            if (! this->_hashMaps[hashNum].empty()) {
              typename _RepSubType::const_iterator  hiter = 
                this->_hashMaps[hashNum].begin();
              for ( ; hiter != iter->end(); ++hiter) {
                target[pfx].first = Ipv4Prefix(hiter->first, hashNum);
                target[pfx].second = hiter->second;
                ++pfx;
              }
            }
          }
          if (! target.empty())
            std::sort(target.begin(), target.end(), ValueGreater());
        }
        
        return;
      }
      
      struct KeyGreater
      {
      public:
        bool operator () (const std::pair<Ipv4Prefix,_valueT> & e1,
                          const std::pair<Ipv4Prefix,_valueT> & e2) const
        {
          return(e1.first > e2.first);
        }
      };
      
      struct KeyLess
      {
      public:
        bool operator () (const std::pair<Ipv4Prefix,_valueT> & e1,
                          const std::pair<Ipv4Prefix,_valueT> & e2) const
        {
          return(e1.first < e2.first);
        }
      };
      
      struct ValueGreater
      {
      public:
        bool operator () (const std::pair<Ipv4Prefix,_valueT> & e1,
                          const std::pair<Ipv4Prefix,_valueT> & e2) const
        {
          return(e1.second > e2.second);
        }
      };
      
      
    protected:
      std::vector<_RepSubType>   _hashMaps;
      mutable std::shared_mutex  _mtx;
    };
    

  }  // namespace McBlock
  
}  // namespace Dwm

#endif  // _DWMMCBLOCKIPV4ROUTES_HH_

//---------------------------- emacs settings -----------------------------
//  Local Variables:
//  mode: C++/la
//  tab-width: 2
//  indent-tabs-mode: nil
//  c-basic-offset: 2
//  End:
//-------------------------------------------------------------------------
