//===========================================================================
// @(#) $Name:$
// @(#) $Id: mcblockd.cc 9274 2017-04-26 04:52:50Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file mcblockd.cc
//!  \brief mcblock daemon
//---------------------------------------------------------------------------

extern "C" {
  #include <fcntl.h>
  #include <unistd.h>
}

#include "DwmAuthPKCrypto.hh"
#include "DwmDaemonUtils.hh"
#include "DwmOptArgs.hh"
#include "DwmSignal.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmCfg2Json.hh"
#include "DwmMcBlockServer.hh"

static const Dwm::SvnTag  svntag("@(#) $DwmPath: dwm/mcplex/mcblock/tags/mcblock-0.1.1/apps/mcblockd/mcblockd.cc 9274 $");

using namespace std;
using namespace Dwm;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void SavePID(const string & pidFile)
{
  pid_t  pid = getpid();
  int    fd = open(pidFile.c_str(), O_WRONLY|O_CREAT|O_TRUNC|O_EXLOCK, 0644);
  if (fd >= 0) {
    string  &&pidstr = to_string(pid) + "\n";
    write(fd, pidstr.c_str(), pidstr.size());
    close(fd);
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void ConfigureSyslog(const Cfg2Json & mcblockConfig, int logOpts)
{
  const Json::Value *jvp = mcblockConfig.Find("Syslog");
  if (jvp) {
    const Json::Value  *p = mcblockConfig.Find(jvp, "Facility");
    if (p && p->isString()) {
      SysLogger::Open("mcblockd", logOpts, p->asString());
    }
    else {
      SysLogger::Open("mcblockd", logOpts, "daemon");
    }
    p = mcblockConfig.Find(jvp, "MinimumPriority");
    if (p && p->isString()) {
      SysLogger::MinimumPriority(p->asString());
    }
    else {
      SysLogger::MinimumPriority("info");
    }
    p = mcblockConfig.Find(jvp, "ShowFileLocations");
    if (p && p->isString()) {
      SysLogger::ShowFileLocation(p->asString() == "true");
    }
    else {
      SysLogger::ShowFileLocation(false);
    }
  }
  else {
    SysLogger::Open("mcblockd", logOpts, "daemon");
    SysLogger::MinimumPriority("info");
    SysLogger::ShowFileLocation(false);
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  OptArgs  optargs;
  optargs.AddOptArg("c:", "config", false, "", "configuration file");
  optargs.AddOptArg("d", "debug", false, "false", "do not daemonize");
  optargs.AddOptArg("p:", "pidfile", false, "", "PID file");
  optargs.Parse(argc, argv);

  int  logOpts = LOG_PID|LOG_PERROR;
  if (! optargs.Get<bool>('d')) {
    DaemonUtils::Daemonize();
    logOpts = LOG_PID;
  }
  
  Signal SigPipe(SIGPIPE);
  SigPipe.Block();

  Signal SigTerm(SIGTERM);
  SigTerm.Block();

  string  pidFile = optargs.Get<string>('p');
  if (pidFile.empty()) {
    pidFile = "/var/run/mcblockd.pid";
  }
  SavePID(pidFile);

  string  configFile = optargs.Get<string>('c');
  if (configFile.empty()) {
    configFile = "/usr/local/etc/mcblockd.conf";
  }

  int  rc = 1;
  Cfg2Json  mcblockConfig;
  if (mcblockConfig.Parse(configFile)) {
    ConfigureSyslog(mcblockConfig, logOpts);
    McBlock::Server  server(mcblockConfig);
    if (server.Start()) {
      for (;;) {
        sleep(1);
        if (SigTerm.IsPending()) {
          break;
        }
      }
      server.Stop();
      rc = 0;
    }
  }
  else {
    Syslog(LOG_ERR, "Failed to parse config file!");
  }

  remove(pidFile.c_str());
  return rc;
}

