//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmMcBlockTcpDrop.cc 9420 2017-06-04 05:28:54Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcBlockTcpDrop.cc
//!  \brief Dwm::McBlock::TcpDropPrefix implementation and support functions
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/param.h>
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <sys/socketvar.h>
  #include <sys/sysctl.h>

  #include <netinet/in.h>
  #include <netinet/in_pcb.h>
  #include <arpa/inet.h>
#define TCPSTATES
  #include <netinet/tcp_fsm.h>
  #include <netinet/tcp_var.h>
#undef TCPSTATES
  #include <netdb.h>
  #include <unistd.h>
}

#include <cstdlib>
#include <iostream>
#include <iomanip>

#include "DwmIpv4Prefix.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"

using namespace std;
using namespace Dwm;

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcblock/tags/mcblock-0.1.1/apps/mcblockd/DwmMcBlockTcpDrop.cc 9420 $");

#define TCPDROP_FOREIGN     0
#define TCPDROP_LOCAL       1

namespace Dwm {

  namespace McBlock {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static bool TcpDrop(const struct sockaddr *localSockAddr,
                        const struct sockaddr *foreignSockAddr)
    {
      bool  rc = false;
#ifndef __APPLE__
      struct sockaddr_storage  addrs[2];
      memcpy(&addrs[TCPDROP_FOREIGN], foreignSockAddr,
             foreignSockAddr->sa_len);
      memcpy(&addrs[TCPDROP_LOCAL], localSockAddr,
             localSockAddr->sa_len);
      if (sysctlbyname("net.inet.tcp.drop", NULL, NULL, &addrs, sizeof addrs)
          != -1) {
        rc = true;
        char  fabuf[INET_ADDRSTRLEN];
        inet_ntop(AF_INET, &((sockaddr_in *)foreignSockAddr)->sin_addr,
                  fabuf, INET_ADDRSTRLEN);
        Syslog(LOG_INFO, "Dropped TCP connection from %s:%hu",
               fabuf, ntohs(((sockaddr_in *)foreignSockAddr)->sin_port));
      }
#endif
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static bool TcpDropIpv4Conn(const struct in_conninfo *inc)
    {
      bool  rc = false;
#ifndef __APPLE__
      struct sockaddr_in   sin4[2];
      memset(sin4, 0, sizeof sin4);
      
      sin4[TCPDROP_LOCAL].sin_len = sizeof sin4[TCPDROP_LOCAL];
      sin4[TCPDROP_LOCAL].sin_family = AF_INET;
      sin4[TCPDROP_LOCAL].sin_port = inc->inc_lport;
      memcpy(&sin4[TCPDROP_LOCAL].sin_addr, &inc->inc_laddr,
             sizeof inc->inc_laddr);
      struct sockaddr  *local = (struct sockaddr *)&sin4[TCPDROP_LOCAL];
      
      sin4[TCPDROP_FOREIGN].sin_len = sizeof sin4[TCPDROP_FOREIGN];
      sin4[TCPDROP_FOREIGN].sin_family = AF_INET;
      sin4[TCPDROP_FOREIGN].sin_port = inc->inc_fport;
      memcpy(&sin4[TCPDROP_FOREIGN].sin_addr, &inc->inc_faddr,
             sizeof inc->inc_faddr);
      struct sockaddr  *foreign = (struct sockaddr *)&sin4[TCPDROP_FOREIGN];
      rc = TcpDrop(local, foreign);
#endif
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static struct xinpgen *GetXPcbList(const char *name)
    {
      struct xinpgen  *xinp = nullptr;
      size_t           len = 0;
      
      if (sysctlbyname(name, NULL, &len, NULL, 0) != -1) {
        if (len > 0) {
          xinp = (struct xinpgen *)malloc(len);
          if (xinp != nullptr) {
            if (sysctlbyname(name, xinp, &len, NULL, 0) == -1) {
              Syslog(LOG_ERR, "sysctlbyname %s failed", name);
              free(xinp);
              xinp = nullptr;
            }
          }
          else {
            Syslog(LOG_ERR, "malloc(%lu) failed", len);
          }
        }
        else {
          Syslog(LOG_ERR, "sysctlbyname(%s) returned 0 length", name);
        }
      }
      else {
        Syslog(LOG_ERR, "sysctlbyname(%s) failed", name);
      }
      return (xinp);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool TcpDropPrefix(Ipv4Prefix & prefix)
    {
#ifdef __APPLE__
      return false;
#else

      bool  rc = true;
      struct xinpgen *head, *xinp;
      struct xtcpcb  *xpcb;
      struct tcpcb   *tp;
      struct inpcb   *inp;

#define XINP_NEXT(xinp)                                         \
      ((struct xinpgen *)((uintptr_t)(xinp) + (xinp)->xig_len))

      head = GetXPcbList("net.inet.tcp.pcblist");
      for (xinp = XINP_NEXT(head); xinp->xig_len > sizeof *xinp;
           xinp = XINP_NEXT(xinp)) {
        xpcb = (struct xtcpcb *)xinp;
        tp = &xpcb->xt_tp;
        inp = &xpcb->xt_inp;

        // XXX Check protocol?
        
        // Ignore PCBs which were freed during copyout.
        if (inp->inp_gencnt > head->xig_gen)
          continue;
        // Skip listening sockets.
        if (tp->t_state == TCPS_LISTEN)
          continue;
        //  Skip IPv6.
        if ((inp->inp_inc.inc_flags & INC_ISIPV6) != 0)
          continue;
        
        if (prefix.Contains(inp->inp_inc.inc_faddr.s_addr)) {
          if (! TcpDropIpv4Conn(&inp->inp_inc)) {
            rc = false;
          }
        }
      }
      free(head);
      return rc;
#endif
    }

    
  }  // namespace McBlock

}  // namespace Dwm
