//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmMcBlockServer.cc 9443 2017-06-06 00:26:17Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcBlockServer.cc
//!  \brief Dwm::McBlock::Server class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <netinet/in.h>
  #include <arpa/inet.h>
#ifndef __APPLE__
  #include <pthread_np.h>
#endif
}

#include <fstream>
#include <sstream>

#include "DwmSocket.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmTimeValue64.hh"
#include "DwmMcBlockPortability.hh"
#include "DwmMcBlockServer.hh"
#include "DwmAuthSymCryptoMessage.hh"
#include "DwmMcBlockResponseMessage.hh"

static const Dwm::SvnTag  svntag("@(#) $DwmPath: dwm/mcplex/mcblock/tags/mcblock-0.1.1/apps/mcblockd/DwmMcBlockServer.cc 9443 $");

using namespace std;

namespace Dwm {

  namespace McBlock {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Server::Server(const Cfg2Json & config)
        :  _config(config), _addRules(config), _pfdev(_config.Device()),
           _socket(), _mcastSocket(), _run(false), _thread(), _responders(),
          _logEntryTrackers(), _databases()
    {
      for (auto & rft : _addRules.RulesForTables()) {
        string  dbFileName(_config.DatabaseDirectory() + "/pf." + rft.first
                           + ".db");
        _databases[rft.first].Load(dbFileName);
      }
      _nextListSaveTime = chrono::system_clock::now() + chrono::hours(24);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const ServerConfig & Server::Config() const
    {
      return _config;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const AddRules & Server::GetAddRules() const
    {
      return _addRules;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const Pf::Device & Server::GetPfDevice() const
    {
      return _pfdev;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Socket & Server::GetSocket()
    {
      return _socket;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    LogEntryTrackers & Server::GetLogEntryTrackers()
    {
      return _logEntryTrackers;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::map<std::string,Db> & Server::GetDatabases()
    {
      return _databases;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Server::Start()
    {
      bool  rc = false;
      if (_socket.Open(PF_INET, SOCK_STREAM, 0)) {
        int  on = 1;
        _socket.Setsockopt(SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));
        _socket.Setsockopt(SOL_SOCKET, SO_REUSEPORT, &on, sizeof(on));

        if (_socket.Bind(_config.TCPAddr(), _config.TCPPort())) {
          if (_mcastSocket.Open(PF_INET, SOCK_DGRAM, 0)) {
            if (_socket.Listen(10)) {
              rc = true;
              _run = true;
              _thread = std::thread(&Server::Run, this);
            }
            else {
              Syslog(LOG_ERR, "Listen(10) failed! %m");
            }
          }
          else {
            Syslog(LOG_ERR, "Failed to open UDP socket for multicast send");
            _socket.Close();
          }
        }
        else {
          Syslog(LOG_ERR, "socket.Bind(%s,%hu) failed! %m",
                 ((string)_config.TCPAddr()).c_str(), _config.TCPPort());
        }
      }
      else {
        Syslog(LOG_ERR, "socket.Open() failed! %m");
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::StopResponders()
    {
      auto  ri = _responders.begin();
      for ( ; ri != _responders.end(); ++ri) {
        (*ri)->Stop();
        Responder  *respToDel = *ri;
        _responders.erase(ri);
        delete respToDel;
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::Stop()
    {
      _run = false;
      StopResponders();
      if (_thread.joinable()) {
        _thread.join();
        Syslog(LOG_INFO, "Server stopped");
      }
      SavePfLists();
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::CleanupResponders()
    {
      auto  ri = _responders.begin();
      for ( ; ri != _responders.end(); ++ri) {
        if ((*ri)->Join()) {
          Responder  *respToDel = *ri;
          _responders.erase(ri);
          delete respToDel;
          break;
        }
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::CleanupLogTrackers()
    {
      for (auto & tr : _logEntryTrackers) {
        auto ari = _addRules.RulesForTable(tr.first);
        if (ari != _addRules.RulesForTables().end()) {
          tr.second.ClearOld(ari->second);
        }
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::Run()
    {
      set_pthread_name(_thread.native_handle(), "server");
      
      Syslog(LOG_INFO, "Server started");
      _pfdev.Open();
      CleanupInactiveDbEntries();
      SavePfLists();

      while (_run) {
        struct timeval  timeout;
        timeout.tv_sec = 0;
        timeout.tv_usec = 200000;
        fd_set  readFdSet, errFdSet;
        FD_ZERO(&readFdSet);
        FD_ZERO(&errFdSet);
        _socket.FdSet(readFdSet);
        _socket.FdSet(errFdSet);
        if (select(FD_SETSIZE, &readFdSet, 0, &errFdSet, &timeout) > 0) {
          if (_socket.FdIsSet(readFdSet)) {
            _responders.push_back(new Responder(*this));
          }
        }
#if 0
        SendMcastStatus();
#endif
        CleanupResponders();
        CleanupLogTrackers();
        
        chrono::system_clock::time_point  now = chrono::system_clock::now();
        if (now > _nextListSaveTime) {
          CleanupInactiveDbEntries();
          SavePfLists();
          _nextListSaveTime = now + chrono::hours(24);
        }
      }
      _pfdev.Close();
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::CleanupInactiveDbEntries()
    {
      for (auto & db : _databases) {
        //  Remove pf table entries for inactive db entries
        vector<DbEntry>  inactiveEntries;
        db.second.GetAllInactive(inactiveEntries);
        Pf::Table  table(_pfdev, "", db.first);
        for (auto & entry : inactiveEntries) {
          table.Remove(entry.Prefix());
        }
        //  Remove inactive entries from database
        db.second.DeleteInactive();
        //  Save the database
        db.second.Save("");
      }
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::SavePfLists() const
    {
      for (auto & db : _databases) {
        string  fileName(_config.DatabaseDirectory() + "/pf." + db.first);
        ofstream  os(fileName.c_str());
        if (os) {
          vector<Ipv4Prefix>  pfxList;
          db.second.GetMinimalPfList(pfxList);
          if (! pfxList.empty()) {
            for (auto pfx : pfxList) {
              os << pfx.ToShortString() << '\n';
            }
          }
          os.close();
        }
      }
      return;
    }
    
#if 0
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::SendMcastStatus()
    {
      static TimeValue64  lastSend(true);
      TimeValue64         now(true);
      TimeValue64         elapsed(now);
      elapsed -= lastSend;
      if (elapsed > TimeValue64(0, 500000)) {
        Json::FastWriter  fw;
        Auth::SymCrypto::Message  msg(_mySecret,
                                      fw.write(_doors.StatusJson()));
        if (msg.SendTo(_mcastSocket, 0, _config.MulticastAddr(),
                       _config.MulticastPort())) {
          lastSend = now;
        }
      }
      return;
    }
#endif

  }  // namespace McBlock

}  // namespace Dwm
