//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmMcBlockAddRules.cc 9146 2017-04-18 20:57:42Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmMcBlockAddRules.cc
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmMcBlockAddRules.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/mcplex/mcblock/tags/mcblock-0.1.1/apps/mcblockd/DwmMcBlockAddRules.cc 9146 $");

using namespace std;

namespace Dwm {

  namespace McBlock {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    AddRule::AddRule(const Cfg2Json & config, const string & tableName,
                     const string & ruleName)
        : _name(ruleName), _countries(".*"),
          _rgx(_countries, regex::ECMAScript|regex::optimize),
          _widestMask(24), _days(30), _logThresh(1), _logDays(1)
    {
      const Json::Value  *jvp =
        config.Find("AddRules/" + tableName + "/" + _name);
      if (jvp) {
        const Json::Value  *p = config.Find(jvp, "countries");
        if (p && p->isString()) {
          _countries = p->asString();
        }
        _rgx = regex(_countries, regex::ECMAScript|regex::optimize);
        p = config.Find(jvp, "widestMask");
        if (p && p->isString()) {
          _widestMask = stol(p->asString());
          if (_widestMask > 32) {
            _widestMask = 24;
          }
        }
        p = config.Find(jvp, "days");
        if (p && p->isString()) {
          _days = stol(p->asString());
        }
        p = config.Find(jvp, "logThresh");
        if (p && p->isString()) {
          _logThresh = stoul(p->asString());
        }
        p = config.Find(jvp, "logDays");
        if (p && p->isString()) {
          _logDays = stoul(p->asString());
        }
      }
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string	& AddRule::Name() const
    {
      return _name;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::regex & AddRule::Rgx() const
    {
      return _rgx;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t AddRule::WidestMask() const
    {
      return _widestMask;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t AddRule::Days() const
    {
      return _days;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t AddRule::LogThresh() const
    {
      return _logThresh;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t AddRule::LogDays() const
    {
      return _logDays;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void AddRule::Apply(DbEntry & dbEntry, const Ipv4Prefix & pfx) const
    {
      if (pfx.MaskLength() < _widestMask) {
        dbEntry.Prefix(Ipv4Prefix(dbEntry.Prefix().Network(), _widestMask));
      }
      else {
        dbEntry.Prefix(pfx);
      }
      TimeValue  startTime(true);
      TimeValue  endTime;
      endTime.Set(startTime.Secs() + (_days * 24 * 60 * 60) + 1, 0);
      TimeInterval  ti(startTime, endTime);
      dbEntry.Interval(ti);

      Syslog(LOG_DEBUG, "Applied rule %s, %s -> %s, %d (%s)",
             _name.c_str(), pfx.ToShortString().c_str(),
             dbEntry.Prefix().ToShortString().c_str(),
             _days, dbEntry.Country().c_str());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Json::Value AddRule::Json() const
    {
      Json::Value  rc;
      rc["countries"] = _countries;
      rc["widestMask"] = to_string((unsigned)_widestMask);
      rc["days"] = to_string(_days);
      rc["logThresh"] = to_string(_logThresh);
      rc["logDays"] = to_string(_logDays);
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & operator << (ostream & os, const AddRule & addRule)
    {
      if (os) {
        os << "    " << addRule._name << " {\n"
           << "      countries: \"" << addRule._countries << "\";\n"
           << "      widestMask: " << (uint16_t)addRule._widestMask << ";\n"
           << "      days: " << addRule._days << ";\n"
           << "      logThresh: " << addRule._logThresh << ";\n"
           << "      logDays: " << addRule._logDays << ";\n"
           << "    }\n";
      }
      return os;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    AddRulesForTable::AddRulesForTable(const Cfg2Json & config,
                                       const string & tableName)
        : _tableName(tableName)
    {
      const Json::Value  *jvp = config.Find("AddRules/" + tableName);
      if (jvp) {
        if (jvp->size() > 0) {
          vector<string>  &&ruleNames = jvp->getMemberNames();
          for (auto name : ruleNames) {
            AddRule  rule(config, tableName, name);
            _addRules.push_back(rule);
          }
        }
      }
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void AddRulesForTable::Apply(DbEntry & dbEntry,
                                 const Ipv4Prefix & pfx) const
    {
      auto  ri = _addRules.begin();
      for ( ; ri != _addRules.end(); ++ri) {
        smatch  sm;
        if (regex_search(dbEntry.Country(), sm, ri->Rgx())) {
          ri->Apply(dbEntry, pfx);
          break;
        }
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const std::string	& AddRulesForTable::Name() const
    {
      return _tableName;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AddRulesForTable::FindRuleForCountry(const string & country,
                                              AddRule & addRule) const
    {
      bool  rc = false;
      for (auto & rule : _addRules) {
        smatch  sm;
        if (regex_search(country, sm, rule.Rgx())) {
          addRule = rule;
          rc = true;
          break;
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Json::Value AddRulesForTable::Json() const
    {
      Json::Value  rc;
      for (auto & ar : _addRules) {
        rc[ar.Name()] = ar.Json();
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & operator << (ostream & os, const AddRulesForTable & addRules)
    {
      if (os) {
        os << "  " <<  addRules._tableName << " {\n";
        auto  ri = addRules._addRules.begin();
        for ( ; ri != addRules._addRules.end(); ++ri) {
          os << *ri;
        }
        os << "  }\n";
      }
      return os;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    AddRules::AddRules(const Cfg2Json & config)
    {
      const Json::Value  *jvp = config.Find("AddRules");
      if (jvp) {
        if (jvp->size() > 0) {
          vector<string>  &&tableNames = jvp->getMemberNames();
          for (auto tableName : tableNames) {
            AddRulesForTable  rulesForTable(config, tableName);
            _addRules[tableName] = rulesForTable;
          }
        }
      }
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const AddRules::map_type & AddRules::RulesForTables() const
    {
      return _addRules;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    AddRules::const_iterator
    AddRules::RulesForTable(const string & tableName) const
    {
      return _addRules.find(tableName);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Json::Value AddRules::Json() const
    {
      Json::Value  rc;
      for (auto & art : _addRules) {
        rc["AddRules"][art.second.Name()] = art.second.Json();
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & operator << (std::ostream & os, const AddRules & addRules)
    {
      if (os) {
        os << "AddRules {\n";
        auto  ti = addRules._addRules.begin();
        for ( ; ti != addRules._addRules.end(); ++ti) {
          os << ti->second;
        }
        os << "}\n";
      }
      return os;
    }
    
  }  // namespace McBlock

}  // namespace Dwm

