//===========================================================================
// @(#) $Name$
// @(#) $Id: mcblock.cc 9418 2017-06-04 03:08:30Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2015
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file mcblock.cc
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

extern "C" {
  #include <netdb.h>
  #include <unistd.h>
}

#include <fstream>

#include "DwmCvsTag.hh"
#include "DwmDateTime.hh"
#include "DwmOptArgs.hh"
#include "DwmSocket.hh"

#include "DwmAuthPeerAuthenticator.hh"
#include "DwmRDAPRequestMessage.hh"
#include "DwmRDAPResponseMessage.hh"
#include "DwmMcBlockAuthLogParser.hh"
#include "DwmMcBlockMailLogParser.hh"
#include "DwmMcBlockDb.hh"

#define DEFAULT_DB "/etc/pf.losers.db"

using namespace Dwm;
using namespace std;

static const Dwm::CvsTag cvstag("@(#) $Name$ $Id: mcblock.cc 9418 2017-06-04 03:08:30Z dwm $");

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
class AddRule
{
public:
  AddRule(const std::regex & rgx, uint8_t widestMask, uint32_t days)
      : _rgx(rgx), _widestMask(widestMask), _days(days)
  {}
  const std::regex & Rgx() const  { return _rgx; }
  uint8_t WidestMask() const      { return _widestMask; }
  uint32_t Days() const           { return _days; }
  
private:
  std::regex  _rgx;
  uint8_t     _widestMask;
  uint32_t    _days;
};

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void ApplyAddRule(McBlock::DbEntry & dbEntry, const Ipv4Prefix & pfx)
{
  static const vector<AddRule>  addRules = {
    { regex("CA|GB|MX|UM|US|VI",regex::ECMAScript|regex::optimize), 24, 30 },
    { regex("AF|AM|AR|AZ|BA|BG|BH|BO|BR|BZ|CN|CO|CL|HK|HR|IQ|IR|KG|"
            "KH|KR|KZ|LR|LT|LV|MD|PE|PK|PS|RO|RS|RU|SA|SI|SK|SY|TR|"
            "UA|UZ|VN", regex::ECMAScript|regex::optimize), 10, 180 },
    { regex(".*", regex::ECMAScript|regex::optimize), 10, 60 }
  };
  auto  i = addRules.begin();
  for ( ; i != addRules.end(); ++i) {
    smatch  sm;
    if (regex_search(dbEntry.Country(), sm, i->Rgx())) {
      if (pfx.MaskLength() < i->WidestMask()) {
        dbEntry.Prefix(Ipv4Prefix(dbEntry.Prefix().Network(),
                                  i->WidestMask()));
      }
      else {
        dbEntry.Prefix(pfx);
      }
      Dwm::TimeValue  startTime(true);
      Dwm::TimeValue  endTime;
      endTime.Set(startTime.Secs() + (i->Days() * 24 * 60 * 60) + 1, 0);
      TimeInterval  ti(startTime, endTime);
      dbEntry.Interval(ti);
      break;
    }
  }
  return;
}
        
//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool GetRDAPConnection(Socket & s, string & agreedKey)
{
  bool   rc = false;
  char  *rhost = getenv("RDAPDHOST");
  if (rhost) {
    struct hostent  *hostEntry = gethostbyname2(rhost, AF_INET);
    if (s.Open(PF_INET, SOCK_STREAM, 0)) {
      struct sockaddr_in  servAddr;
      memset(&servAddr, 0, sizeof(servAddr));
      servAddr.sin_len = sizeof(servAddr);
      servAddr.sin_family = AF_INET;
      servAddr.sin_port = htons(6363);
      servAddr.sin_addr.s_addr = *(in_addr_t *)hostEntry->h_addr_list[0];
      if (s.Connect(servAddr)) {
        int  noDelay = 1;
        // s.Setsockopt(IPPROTO_TCP, TCP_NODELAY, &noDelay, sizeof(noDelay));
        Auth::PeerAuthenticator  peer("", "");
        string  theirId;
        if (peer.Authenticate(s, theirId, agreedKey)) {
          rc = true;
        }
        else {
          s.Close();
        }
      }
    }
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool DeleteContained(McBlock::Db & db, const Ipv4Prefix & prefix,
                            bool deletePrefix)
{
  bool  rc = true;
  vector<pair<Ipv4Prefix,McBlock::DbEntry> >  contained;
  if (db.FindContained(prefix, contained)) {
    for (auto i : contained) {
      if ((i.first != prefix) || deletePrefix) {
        if (! db.Delete(i.first)) {
          rc = false;
          break;
        }
      }
    }
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool DoubleInterval(McBlock::Db & db, McBlock::DbEntry & entry)
{
  bool  rc = false;
  Dwm::TimeValue  startTime(true);
  Dwm::TimeValue  endTime;
  uint64_t   secs = entry.Interval().Duration().Secs();
  secs <<= 1;
  endTime.Set(startTime.Secs() + secs, 0);
  entry.Interval(TimeInterval(startTime, endTime));
  
  //  Delete the old entry.
  if (db.DeleteEntry(entry.Prefix())) {
    //  Add it back with new interval.
    rc = db.AddEntry(entry);
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool AddHostEntry(McBlock::Db & db, const Ipv4Address & hostAddr,
                         uint32_t days,
                         const std::string & registry = "",
                         const std::string & country = "")
{
  bool  rc = false;
  
  vector<pair<Ipv4Prefix,McBlock::DbEntry>>  matches;
  if (db.Find(hostAddr, matches)) {
    rc = true;
    //  one or more entries found.  Double their intervals.
    for (auto match : matches) {
      if (! match.second.IsActive()) {
        if (! DoubleInterval(db, match.second)) {
          rc = false;
          break;
        }
      }
    }
  }
  else {
    // No entry found.  Add a /24 for given days.
    Dwm::TimeValue  startTime(true);
    Dwm::TimeValue  endTime;
    endTime.Set(startTime.Secs() + (days * 24 * 60 * 60) + 1, 0);
    TimeInterval  ti(startTime, endTime);
    McBlock::DbEntry  newEntry(Ipv4Prefix(hostAddr, 24), ti,
                               registry, country);
    rc = db.AddEntry(newEntry);
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool AddHostEntry(const string & dbFilename,
                         const std::string & host,
                         uint32_t days,
                         const std::string & registry = "",
                         const std::string & country = "")
{
  bool  rc = false;
  McBlock::Db  db;
  Ipv4Address  hostAddr(host);

  //  Load existing database.
  db.Load(dbFilename);

  rc = AddHostEntry(db, hostAddr, days, registry, country);
  
  if (rc) {
    rc = db.Save(dbFilename);
    if (! rc) {
      cerr << "Failed to save to " << dbFilename << '\n';
    }
  }
  
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static string StartColor(const string & color)
{
  static map<string,string>  colors = {
    { "red",     "[31m" },
    { "cyan",    "[36m" },
    { "default", "[39m" }
  };
  string  rc("");
  if (isatty(STDOUT_FILENO)) {
    auto  c = colors.find(color);
    if (c != colors.end()) {
      rc = c->second;
    }
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static string StartAttr(const string & attr)
{
  static map<string,string>  attrs = {
    { "reset",  "[0m" },
    { "bright", "[1m" },
    { "dim",    "[2m" }
  };
  string  rc("");
  if (isatty(STDOUT_FILENO)) {
    auto  a = attrs.find(attr);
    if (a != attrs.end()) {
      rc = a->second;
    }
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void PrintMatchChange(const pair<Ipv4Address,uint64_t> & logEntry,
                             const McBlock::DbEntry & dbEntry)
{
  cout << setiosflags(ios::left) << setw(15) << logEntry.first << " "
       << resetiosflags(ios::left) << setw(5) << logEntry.second
       << " hits";
  if (dbEntry.IsActive()) {
    cout << StartAttr("dim") << " match  ";
  }
  else {
      cout << StartAttr("reset") << StartColor("cyan") << " update ";
  }
  cout << setiosflags(ios::left)
       << setw(18) << dbEntry.Prefix()
       << resetiosflags(ios::left) << " ";
  uint64_t   days = dbEntry.Interval().Duration().Secs() / (24 * 60 * 60);
  Dwm::TimeValue  endTime(true);
  if (dbEntry.IsActive()) {
    endTime = dbEntry.Interval().End();
  }
  else {
    days *= 2;
    endTime.Set(endTime.Secs() + (days * 24 * 60 * 60), 0);
  }
  DateTime   endDate(endTime);
  ostringstream  os;
  if (days >= 365) {
    os << days/365 << "y";
  }
  os << days % 365 << "d";
  cout << setw(7) << os.str();
  cout << endDate.Formatted(" %Y/%m/%d") << ' '
       << McBlock::DbEntry::GetDefaultRegistry(logEntry.first);
  cout << StartColor("default") << StartAttr("reset") << '\n';
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void
PrintMatchChanges(const pair<Ipv4Address,uint64_t> & logEntry,
                  const vector<pair<Ipv4Prefix,McBlock::DbEntry> > & matches)
{
  cout.flush();
  for (auto m : matches) {
    PrintMatchChange(logEntry, m.second);
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool ForeignRegistry(const Ipv4Address & addr)
{
  string  registry = McBlock::DbEntry::GetDefaultRegistry(addr);
  return ((registry == "RIPE")
          || (registry == "APNIC")
          || (registry == "LACNIC")
          || (registry == "AFRINIC"));
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool RDAPGet(Socket & s, const string & agreedKey,
             const Ipv4Address & addr, Ipv4Prefix & pfx,
             string & ccode)
{
  bool  rc = false;
  vector<Ipv4Address>   addrs;
  addrs.push_back(addr);
  RDAP::RequestMessage  req(addrs);
  if (req.Write(s, agreedKey)) {
    RDAP::ResponseMessage  resp;
    if (resp.Read(s, agreedKey)) {
      if ((! resp.Json().empty()) && (resp.Json().isArray())) {
        pfx = Ipv4Prefix(resp.Json()[0]["prefix"].asString());
        ccode = resp.Json()[0]["country"].asString();
        rc = true;
      }
    }
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void PrintAdd(const pair<Ipv4Address,uint64_t> & logEntry,
                     const McBlock::DbEntry & newEntry)
{
  cout << setiosflags(ios::left) << setw(15) << logEntry.first << " "
       << resetiosflags(ios::left) << setw(5) << logEntry.second
       << " hits";
  cout << StartAttr("bright") << StartColor("red") << " add    "
       << setiosflags(ios::left)
       << setw(2) << newEntry.Country() << ' '
       << setw(18) << newEntry.Prefix().ToShortString() << " "
       << resetiosflags(ios::left);
  ostringstream os;
  uint32_t  days =
    newEntry.Interval().Duration().Secs() / (24 * 60 * 60);
  if (days >= 365) {
    os << days/365 << "y";
  }
  os << days % 365 << "d";
  cout << setw(7) << os.str();
  DateTime  endDate(newEntry.Interval().End());
  cout << endDate.Formatted(" %Y/%m/%d") << ' '
    // << McBlock::DbEntry::GetDefaultRegistry(logEntry.first)
       << StartColor("default") << StartAttr("reset") << '\n';
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool ShowAuthLogEntries(const string & dbFilename,
                               const string & authLogFilename,
                               int threshold, uint32_t days)
{
  bool  rc = false;
  McBlock::AuthLogParser  authLogParser;
  authLogParser.Parse(authLogFilename);
  vector<pair<Ipv4Address,uint64_t>>  &&logEntries = authLogParser.SortedHosts();
  if (! logEntries.empty()) {
    McBlock::Db  db;
    db.Load(dbFilename);
    Socket  rdapSock;
    string  agreedKey;
    bool    haveRdap = GetRDAPConnection(rdapSock, agreedKey);
    for (auto logEntry : logEntries) {
      if ((logEntry.second >= threshold)
          || (ForeignRegistry(logEntry.first))) {
        vector<pair<Ipv4Prefix,McBlock::DbEntry> >  matches;
        if (db.Find(logEntry.first, matches)) {
          PrintMatchChanges(logEntry, matches);
        }
        else {
          // No entry found.  Would add a /24 for days.
          Dwm::TimeValue  startTime(true);
          Dwm::TimeValue  endTime;
          endTime.Set(startTime.Secs() + (days * 24 * 60 * 60) + 1, 0);
          TimeInterval  ti(startTime, endTime);
          McBlock::DbEntry  newEntry;
          if (haveRdap) {
            string      ccode;
            Ipv4Prefix  pfx;
            if (RDAPGet(rdapSock, agreedKey, logEntry.first, pfx, ccode)) {
              newEntry = McBlock::DbEntry(Ipv4Prefix(logEntry.first, 24),
                                          ti, "", ccode);
              ApplyAddRule(newEntry, pfx);
            }
            else {
              newEntry = McBlock::DbEntry(Ipv4Prefix(logEntry.first, 24), ti,
                                          "", "");
            }
          }
          else {
            newEntry = 
              McBlock::DbEntry(Ipv4Prefix(logEntry.first, 24), ti,
                               "", "");
          }
          PrintAdd(logEntry, newEntry);
#if 0
          cout << setiosflags(ios::left) << setw(15) << logEntry.first << " "
               << resetiosflags(ios::left) << setw(5) << logEntry.second
               << " hits";
          cout << StartAttr("bright") << StartColor("red") << " add    "
               << setiosflags(ios::left)
               << setw(18) << newEntry.Prefix().ToShortString() << " "
               << resetiosflags(ios::left);
          ostringstream os;
          if (days >= 365) {
            os << days/365 << "y";
          }
          os << days % 365 << "d";
          cout << setw(7) << os.str();
          DateTime  endDate(endTime);
          cout << endDate.Formatted(" %Y/%m/%d") << ' '
               << McBlock::DbEntry::GetDefaultRegistry(logEntry.first)
               << StartColor("default") << StartAttr("reset") << '\n';
#endif
        }
      }
    }
    rc = true;
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool ShowMailLogEntries(const string & dbFilename,
                               const string & mailLogFilename,
                               int threshold, uint32_t days)
{
  bool  rc = false;
  McBlock::MailLogParser  mailLogParser;
  mailLogParser.Parse(mailLogFilename);
  vector<pair<Ipv4Address,uint64_t>>  &&logEntries =
    mailLogParser.SortedHosts();
  if (! logEntries.empty()) {
    McBlock::Db  db;
    db.Load(dbFilename);
    for (auto logEntry : logEntries) {
      if ((logEntry.second >= threshold)
          || (ForeignRegistry(logEntry.first))) {
        vector<pair<Ipv4Prefix,McBlock::DbEntry> >  matches;
        if (db.Find(logEntry.first, matches)) {
          PrintMatchChanges(logEntry, matches);
        }
        else {
          // No entry found.  Would add a /24 for days.
          Dwm::TimeValue  startTime(true);
          Dwm::TimeValue  endTime;
          endTime.Set(startTime.Secs() + (days * 24 * 60 * 60) + 1, 0);
          TimeInterval  ti(startTime, endTime);
          McBlock::DbEntry  newEntry(Ipv4Prefix(logEntry.first, 24), ti,
                                     "", "");
          cout << setiosflags(ios::left) << setw(15) << logEntry.first << " "
               << resetiosflags(ios::left) << setw(5) << logEntry.second
               << " hits";
          cout << StartAttr("bright") << StartColor("red") << " add    "
               << setiosflags(ios::left)
               << setw(18) << newEntry.Prefix().ToShortString() << " "
               << resetiosflags(ios::left);
          ostringstream os;
          if (days >= 365) {
            os << days/365 << "y";
          }
          os << days % 365 << "d";
          cout << setw(7) << os.str();
          DateTime  endDate(endTime);
          cout << endDate.Formatted(" %Y/%m/%d") << ' '
               << McBlock::DbEntry::GetDefaultRegistry(logEntry.first)
               << StartColor("default") << StartAttr("reset") << '\n';
        }
      }
    }
    rc = true;
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool AddRuledEntry(McBlock::Db & db, McBlock::DbEntry & entry)
{
  bool  rc = false;
  McBlock::DbEntry  foundEntry;
  if (db.Find(entry.Prefix(), foundEntry)) {
    // If it was not active, double our interval.
    if (! foundEntry.IsActive()) {
      Dwm::TimeValue  startTime(true);
      Dwm::TimeValue  endTime;
      endTime.Set(startTime.Secs()
                  + (foundEntry.Interval().Duration().Secs() * 2), 0);
      entry.Interval(TimeInterval(startTime, endTime));
      //  Delete contained entries, including the prefix we're adding.
      DeleteContained(db, entry.Prefix(), true);
      //  Add our entry with new interval.
      rc = db.AddEntry(entry);
    }
    else {
      //  Delete contained entries, except the prefix we're adding.
      rc = DeleteContained(db, entry.Prefix(), false);
    }
  }
  else {
    //  Entry was not in database.  Delete contained entries.
    DeleteContained(db, entry.Prefix(), false);
    //  Add new entry.
    rc = db.AddEntry(entry);
  }

  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool AddAuthLogEntries(const string & dbFilename,
                              const string & authLogFilename,
                              int threshold, uint32_t days)
{
  bool  rc = false;
  
  McBlock::AuthLogParser  authLogParser;
  authLogParser.Parse(authLogFilename);
  vector<pair<Ipv4Address,uint64_t>>  &&logEntries = authLogParser.SortedHosts();
  if (! logEntries.empty()) {
    McBlock::Db  db;
    db.Load(dbFilename);
    Socket  rdapSock;
    string  agreedKey;
    bool    haveRdap = GetRDAPConnection(rdapSock, agreedKey);
    for (auto logEntry : logEntries) {
      if ((logEntry.second >= threshold) || ForeignRegistry(logEntry.first)) {
        if (haveRdap) {
          string      ccode;
          Ipv4Prefix  pfx;
          if (RDAPGet(rdapSock, agreedKey, logEntry.first, pfx, ccode)) {
            Dwm::TimeValue  startTime(true);
            Dwm::TimeValue  endTime;
            endTime.Set(startTime.Secs() + (days * 24 * 60 * 60) + 1, 0);
            TimeInterval  ti(startTime, endTime);
            McBlock::DbEntry  newEntry =
              McBlock::DbEntry(Ipv4Prefix(logEntry.first, 24), ti, "", ccode);
            ApplyAddRule(newEntry, pfx);
            rc = AddRuledEntry(db, newEntry);
          }
          else {
            rc = AddHostEntry(db, logEntry.first, days);
          }
        }
        else {
          rc = AddHostEntry(db, logEntry.first, days);
        }
      }
    }
    if (rc) {
      rc = db.Save(dbFilename);
      if (! rc) {
        cerr << "Failed to save to " << dbFilename << '\n';
      }
    }
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool AddEntry(const string & dbFilename,
                     const std::string & network,
                     uint32_t days = 30)
{
  bool  rc = false;
  
  Dwm::TimeValue  startTime(true);
  Dwm::TimeValue  endTime(true);
  endTime.Set(endTime.Secs() + (days * 24 * 60 * 60) + 1, 0);
  TimeInterval  ti(startTime, endTime);
  McBlock::DbEntry  entry(network, ti, "", "");

  McBlock::Db  db;

  //  Load existing database
  db.Load(dbFilename);

  McBlock::DbEntry  foundEntry;
  if (db.Find(entry.Prefix(), foundEntry)) {
    // entry was in the database.  Copy its country.
    entry.Country(foundEntry.Country());
    // If it was not active, double our interval.
    if (! foundEntry.IsActive()) {
      endTime.Set(startTime.Secs()
                  + (foundEntry.Interval().Duration().Secs() * 2), 0);
      entry.Interval(TimeInterval(startTime, endTime));
      //  Delete contained entries, including the prefix we're adding.
      DeleteContained(db, entry.Prefix(), true);
      //  Add our entry with new interval.
      rc = db.AddEntry(entry);
    }
    else {
      //  Delete contained entries, except the prefix we're adding.
      rc = DeleteContained(db, entry.Prefix(), false);
    }
  }
  else {
    //  Entry was not in database.  Delete contained entries.
    DeleteContained(db, entry.Prefix(), false);
    //  Add new entry.
    rc = db.AddEntry(entry);
  }

  if (rc) {
    rc = db.Save(dbFilename);
    if (! rc) {
      cerr << "Failed to save to " << dbFilename << '\n';
    }
  }
  
  return rc;
}


//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool ListEntries(const string & dbFilename, bool activeOnly,
                        bool noRegistry, bool noCountry, bool pfFormat = false)
{
  bool  rc = false;
  McBlock::Db  db;
  ifstream  is(dbFilename.c_str());
  if (is) {
    if (db.Read(is)) {
      if (pfFormat) {
        db.Coalesce();
      }
      vector<pair<Ipv4Prefix,McBlock::DbEntry>>  entries;
      db.SortByKey(entries);
      for (auto it : entries) {
        if ((! it.second.IsActive()) && activeOnly) {
          continue;
        }
        if (pfFormat) {
          if (it.second.IsActive()) {
            cout << it.first.ToShortString() << '\n';
          }
        }
        else {
          if (! it.second.IsActive()) {
            cout << StartAttr("dim");
          }
          if (it.second.Registry().empty() || (! noRegistry)) {
            if (it.second.Country().empty() || (! noCountry)) {
              DateTime  endDate(it.second.Interval().End());
              cout << setiosflags(ios::left) << setw(18)
                   << it.second.Prefix().ToShortString()
                   << resetiosflags(ios::left)
                   << " " << setw(7) << it.second.DurationString() << "  "
                   << endDate.Formatted("%Y/%m/%d")
                   << "  " << setiosflags(ios::left) << setw(7)
                   << it.second.Registry() << "  " << it.second.Country()
                   << resetiosflags(ios::left)
                   << '\n';
            }
          }
          if (! it.second.IsActive()) {
            cout << StartAttr("reset");
          }
        }
      }
      rc = true;
    }
    else {
      cerr << "Failed to read database from " << dbFilename << '\n';
    }
    is.close();
  }
  else {
    cerr << "Failed to open " << dbFilename << '\n';
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool EditEntry(const string & dbFilename, const string & prefix)
{
  bool  rc = false;
  McBlock::Db  db;
  if (db.Load(dbFilename)) {
    Ipv4Prefix  pfx(prefix);
    McBlock::DbEntry  foundEntry;
    if (db.Find(Ipv4Prefix(pfx), foundEntry)) {
      string  s;
    getStartTime:
      DateTime  startTime = foundEntry.Interval().Start();
      cout << "start time [" << startTime.Formatted("%Y/%m/%d %H:%M") << "]: "
           << std::flush;
      std::getline(cin, s);
      if (! s.empty()) {
        if (startTime.Parse(s, "%Y/%m/%d %H:%M")) {
          foundEntry.Interval().Start(startTime.GetTimeValue());
        }
        else {
          cerr << "Bad start time!\n";
          goto getStartTime;
        }
      }
    getEndTime:
      DateTime  endTime = foundEntry.Interval().End();
      cout << "end time [" << endTime.Formatted("%Y/%m/%d %H:%M") << "]: "
           << std::flush;
      std::getline(cin, s);
      if (! s.empty()) {
        if (endTime.Parse(s, "%Y/%m/%d %H:%M")) {
          foundEntry.Interval().End(endTime.GetTimeValue());
        }
        else {
          cerr << "Bad end time!\n";
          goto getEndTime;
        }
      }
      if (endTime.GetTimeValue() <= startTime.GetTimeValue()) {
        cerr << "endTime must be greater than startTime.\n";
        goto getStartTime;
      }
    getRegistry:
      string  registry = foundEntry.Registry();
      cout << "registry [" << registry << "]: " << std::flush;
      std::getline(cin, s, '\n');
      if (! s.empty()) {
        foundEntry.Registry(s);
      }
    getCountry:
      string  country = foundEntry.Country();
      cout << "country [" << country << "]: " << std::flush;
      std::getline(cin, s);
      if (! s.empty()) {
        foundEntry.Country(s);
      }
      
      // Delete the old entry
      db.DeleteEntry(pfx);

      // Add it back with changes
      rc = db.AddEntry(foundEntry);

      if (rc) {
        //  Save the database
        rc = db.Save(dbFilename);
      }
    }
    else {
      cerr << "No entry found for " << prefix << '\n';
    }
  }
  else {
    cerr << "Failed to open " << dbFilename << '\n';
  }
  if (rc) {
    cout << "Entry updated.\n";
  }
  else {
    cerr << "Entry NOT updated!\n";
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool ImportPrefixes(const string & dbFilename,
                           const string & prefixFilename,
                           uint32_t days)
{
  bool  rc = false;
  McBlock::Db  db;
  db.Load(dbFilename);

  ifstream  is(prefixFilename.c_str());
  if (is) {
    Ipv4Address  badNet("255.255.255.255");
    string  s;
    while (getline(is, s, '\n')) {
      Ipv4Prefix  prefix(s);
      if (prefix.Network() != badNet) {
        Dwm::TimeValue  startTime(true);
        Dwm::TimeValue  endTime;
        endTime.Set(startTime.Secs() + (days * 24 * 60 * 60) + 1, 0);
        TimeInterval  ti(startTime, endTime);
        McBlock::DbEntry  dbEntry(prefix, ti, "", "");
        if (db.AddEntry(dbEntry)) {
          rc = true;
        }
      }
    }
    is.close();
  }
  if (rc) {
    rc = db.Save(dbFilename);
    if (! rc) {
      cerr << "Failed to save to " << dbFilename << '\n';
    }
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool RemoveEntry(const string & dbFilename,
                        const string & prefix)
{
  bool  rc = false;
  McBlock::Db  db;
  //  Load existing database
  if (db.Load(dbFilename)) {
    Ipv4Prefix  network(prefix);
    McBlock::DbEntry  foundEntry;
    if (db.Find(network, foundEntry)) {
      //  Remove entry from database.
      rc = db.DeleteEntry(network);
    }
    if (rc) {
      //  Save database.
      rc = db.Save(dbFilename);
      if (! rc) {
        cerr << "Failed to save to " << dbFilename << '\n';
      }
    }
  }
  else {
    cerr << "Failed to load from " << dbFilename << '\n';
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool CoalesceEntries(const string & dbFilename)
{
  bool  rc = false;
  McBlock::Db  db;
  //  Load existing database
  rc = db.Load(dbFilename);

  if (rc) {
    //  Coalesce
    if (db.Coalesce()) {
      rc = db.Save(dbFilename);
      if (! rc) {
        cerr << "Failed to save to " << dbFilename << '\n';
      }
    }
  }
  else {
    cerr << "Failed to load from " << dbFilename << '\n';
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static bool ShowEntries(const string & dbFilename, const string & prefix)
{
  bool  rc = false;
  McBlock::Db  db;
  if (db.Load(dbFilename)) {
    Ipv4Prefix  pfx(prefix);
    if (pfx.MaskLength() < 32) {
      McBlock::DbEntry  entry;
      if (db.Find(pfx, entry)) {
        DateTime  endDate(entry.Interval().End());
        cout << setiosflags(ios::left) << setw(18)
             << entry.Prefix().ToShortString()
             << resetiosflags(ios::left)
             << " " << entry.DurationString() << "  "
             << endDate.Formatted("%Y/%m/%d")
             << "  " << setiosflags(ios::left) << setw(7)
             << entry.Registry() << "  " << entry.Country()
             << resetiosflags(ios::left)
             << '\n';
        rc = true;
      }
    }
    else {
      vector<pair<Ipv4Prefix,McBlock::DbEntry> >  entries;
      if (db.Find(pfx.Network(), entries)) {
        for (auto entry : entries) {
          // cout << entry.second << '\n';
          DateTime  endDate(entry.second.Interval().End());
          cout << setiosflags(ios::left) << setw(18)
               << entry.second.Prefix().ToShortString()
               << resetiosflags(ios::left)
               << " " << entry.second.DurationString() << "  "
               << endDate.Formatted("%Y/%m/%d")
               << "  " << setiosflags(ios::left) << setw(7)
               << entry.second.Registry() << "  " << entry.second.Country()
               << resetiosflags(ios::left)
               << '\n';
        }
        rc = true;
      }
    }
  }
  else {
    cerr << "Failed to load from " << dbFilename << '\n';
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  OptArgs  optargs;
  optargs.AddOptArg("A:", "addauth", false, "",
                    "add auth.log violators."
                    "  Argument is auth log filename or - for stdin.");
  optargs.AddOptArg("a:", "add",     false, "",
                    "add a network.  Argument is a prefix (a.b.c.d/n).");
  optargs.AddOptArg("c", "coalesce", false, "false",
                    "Coalesce covered narrow and adjacent prefixes into"
                    " wider prefixes.");
  optargs.AddOptArg("d:", "days",    false, "30",
                    "days to block (for add, import, addauth).  Default: 30");
  optargs.AddOptArg("e:", "edit",    false, "",
                    "edit an entry.  Argument is a prefix (a.b.c.d/n).");
  optargs.AddOptArg("f:", "file",    false, DEFAULT_DB,
                    "database file.  Default: " DEFAULT_DB);
  optargs.AddOptArg("i:", "import",  false, "", "import prefixes from file");
  optargs.AddOptArg("l",  "list",    false, "false", "list entries");
  optargs.AddOptArg("L",  "listactive", false, "false", "list active entries");
  optargs.AddOptArg("n",  "nocountry",    false, "false",
                    "restrict 'list' display to entries with no country");
  optargs.AddOptArg("N",  "noregistry",    false, "false",
                    "restrict 'list' display to entries with no registry");
  optargs.AddOptArg("O:", "showauth", false, "",
                    "show auth log entries."
                    "  Argument is auth log filename or - for stdin.");
  optargs.AddOptArg("M:", "showmail", false, "",
                    "show mail log entries."
                    "  Argument is mail log filename or - for stdin.");
  optargs.AddOptArg("r:", "remove",  false, "",
                    "remove a network.  Argument is a prefix (a.b.c.d/n).");
  optargs.AddOptArg("s:", "show",    false, "", "show a network");
  optargs.AddOptArg("t:", "threshold", false, "3",
                    "auth.log threshold.  This is the minimum number of hits"
                    "\n  in the input that will cause a match.");
  optargs.AddOptArg("p",  "pf",      false, "false", "output in pf format");

  int  nextArg = optargs.Parse(argc, argv);

  if (argc == 1) {
    optargs.Usage(argv[0]);
    return 1;
  }
  
  bool  rc = false;
  
  if (! optargs.Get<string>("addauth").empty()) {
    rc = AddAuthLogEntries(optargs.Get<string>("file"),
                           optargs.Get<string>("addauth"),
                           atoi(optargs.Get<string>("threshold").c_str()),
                           atoi(optargs.Get<string>("days").c_str()));
  }
  else if (! optargs.Get<string>("add").empty()) {
    rc = AddEntry(optargs.Get<string>("file"), optargs.Get<string>("add"),
                  atoi(optargs.Get<string>("days").c_str()));
  }
  else if (! optargs.Get<string>("edit").empty()) {
    rc = EditEntry(optargs.Get<string>("file"), optargs.Get<string>("edit"));
  }
  else if (! optargs.Get<string>("import").empty()) {
    rc = ImportPrefixes(optargs.Get<string>("file"),
                        optargs.Get<string>("import"),
                        atoi(optargs.Get<string>("days").c_str()));
  }
  else if (! optargs.Get<string>("showauth").empty()) {
    rc = ShowAuthLogEntries(optargs.Get<string>("file"),
                            optargs.Get<string>("showauth"),
                            atoi(optargs.Get<string>("threshold").c_str()),
                            atoi(optargs.Get<string>("days").c_str()));
  }
  else if (! optargs.Get<string>("showmail").empty()) {
    rc = ShowMailLogEntries(optargs.Get<string>("file"),
                            optargs.Get<string>("showmail"),
                            atoi(optargs.Get<string>("threshold").c_str()),
                            atoi(optargs.Get<string>("days").c_str()));
  }
  else if (! optargs.Get<string>("show").empty()) {
    rc = ShowEntries(optargs.Get<string>("file"),
                     optargs.Get<string>("show"));
  }
  else if (optargs.Get<bool>("list")
           || optargs.Get<bool>("listactive")) {
    rc = ListEntries(optargs.Get<string>("file"),
                     optargs.Get<bool>("listactive"),
                     optargs.Get<bool>("noregistry"),
                     optargs.Get<bool>("nocountry"),
                     optargs.Get<bool>("pf"));
  }
  else if (optargs.Get<bool>("pf")) {
    rc = ListEntries(optargs.Get<string>("file"), false, false, true);
  }
  else if (! optargs.Get<string>("remove").empty()) {
    rc = RemoveEntry(optargs.Get<string>("file"),
                     optargs.Get<string>("remove"));
  }
  
  if (optargs.Get<bool>("coalesce")) {
    rc = CoalesceEntries(optargs.Get<string>("file"));
  }

  if (rc) {
    exit(0);
  }
  else {
    exit(1);
  }
}
