#include <iostream>
#include <fstream>
#include <map>
#include <regex>
#include <string>

#include "DwmOptArgs.hh"
#include "DwmSvnTag.hh"
#include "DwmIpv4CountryDb.hh"

using namespace std;
using namespace Dwm;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
map<uint32_t,string> LoadCountryById(const string & filename)
{
  map<uint32_t,string>  rc;
  ifstream  is(filename);
  regex   rgx("([^,]+)", regex::ECMAScript|regex::optimize);
  if (is) {
    string  s;
    getline(is, s);  // skip header line
    while (getline(is, s)) {
      auto  vbeg = sregex_iterator(s.begin(), s.end(), rgx);
      auto  vend = sregex_iterator();
      auto  it = vbeg;
      uint32_t  country_id = stoul(it->str());
      ++it; ++it; ++it; ++it;
      rc[country_id] = it->str();
    }
    is.close();
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool AddIpv4Blocks(const string & filename,
                   const map<uint32_t, string> & countryById,
                   Ipv4CountryDb & db)
{
  bool    rc = false;
  regex   rgx("([^,]+)", regex::ECMAScript|regex::optimize);
  ifstream  is(filename);
  if (is) {
    string  s;
    getline(is, s);  // skip header line
    while (getline(is, s)) {
      auto  vbeg = sregex_iterator(s.begin(), s.end(), rgx);
      auto  vend = sregex_iterator();
      auto  it = vbeg;
      if (it != vend) {
        Ipv4Prefix  pfx(it->str());
        ++it;
        if (it != vend) {
          auto  cit = countryById.find(stoul(it->str()));
          if (cit != countryById.end()) {
            Ipv4CountryDbValue  val(cit->second, Dwm::TimeValue64(true));
            ((RDAP::Ipv4Routes<Ipv4CountryDbValue> &)db).Add(pfx, val);
            rc = true;
          }
        }
      }
    }
    is.close();
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
    OptArgs  optargs;
  optargs.AddOptArg("d:", "directory", true, "",
                    "directory containing GeoLite2 csv files");
  optargs.AddOptArg("o:", "output", true, "", "output filename");
  optargs.Parse(argc, argv);

  string  locationFile(optargs.Get<string>('d')
                       + "/GeoLite2-Country-Locations-en.csv");
  map<uint32_t,string> && countryById = LoadCountryById(locationFile);
  if (! countryById.empty()) {
    string  ipv4BlockFile(optargs.Get<string>('d')
                          + "/GeoLite2-Country-Blocks-IPv4.csv");
    Ipv4CountryDb  db(optargs.Get<string>('o'));
    if (AddIpv4Blocks(ipv4BlockFile, countryById, db)) {
      db.AggregateAdjacents();
      db.DeleteRedundantSpecifics();
      if (! db.Save()) {
        cerr << "Failed to save database in " << optargs.Get<string>('o')
             << "!\n";
        return 1;
      }
    }
    else {
      cerr << "Failed to load IPv4 addresses from " << ipv4BlockFile << '\n';
      return 1;
    }
  }
  else {
    cerr << "Failed to load countries from " << locationFile << '\n';
    return 1;
  }
  return 0;
}
