//===========================================================================
// @(#) $Name:$
// @(#) $Id: ipdeny2db.cc 9406 2017-06-03 03:50:27Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file ipdeny2db.cc
//!  \brief Created database for dwmrdapd from zone files from ipdeny.com
//---------------------------------------------------------------------------

#include <fstream>
#include <iostream>
#include <map>
#include <regex>

#include "DwmDirectoryEntry.hh"
#include "DwmOptArgs.hh"
#include "DwmSvnTag.hh"
#include "DwmIpv4CountryDb.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmRDAP/tags/libDwmRDAP-0.3.3/apps/ipdeny2db/ipdeny2db.cc 9406 $");

using namespace std;
using namespace Dwm;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
map<string,string> GetFiles(DirectoryEntry & de)
{
  regex  frgx(".*/[a-z]{2}\\.zone$", regex::ECMAScript|regex::optimize);
  regex  crgx(".*/([a-z]{2})\\.zone$", regex::ECMAScript|regex::optimize);
  map<string,string>      rc;
  vector<DirectoryEntry>  files;
  de.GetChildren(files, DirectoryEntry::e_typeRegular);
  auto  it = files.begin();
  for ( ; it != files.end(); ++it) {
    smatch  sm;
    if (regex_search(it->Path(), sm, frgx)) {
      if (regex_search(it->Path(), sm, crgx)) {
        string  cc = sm[1].str();
        for (auto & c : cc) { c = toupper(c); }
        rc[cc] = it->Path();
      }
    }
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool AddCountryFile(const pair<string,string> & cntryFile, Ipv4CountryDb & db)
{
  bool  rc = false;
  ifstream  is(cntryFile.second.c_str());
  if (is) {
    string  s;
    while (getline(is, s, '\n')) {
      Ipv4Prefix  pfx(s);
      Ipv4CountryDbValue  val(cntryFile.first, Dwm::TimeValue64(true));
      ((RDAP::Ipv4Routes<Ipv4CountryDbValue> &)db).Add(pfx, val);
    }
    is.close();
    rc = true;
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  OptArgs  optargs;
  optargs.AddOptArg("d:", "directory", true, "",
                    "directory containing .zone files");
  optargs.AddOptArg("o:", "output", true, "", "output filename");
  optargs.Parse(argc, argv);

  DirectoryEntry        de(optargs.Get<string>('d'));
  map<string,string>  &&files = GetFiles(de);
  if (files.empty()) {
    cerr << "No .zone files found in " << optargs.Get<string>('d') << "!\n";
    return 1;
  }
  
  Ipv4CountryDb  db(optargs.Get<string>('o'));
  for (auto f : files) {
    if (! AddCountryFile(f, db)) {
      cerr << "Failed to load country file " << f.second << '\n';
    }
  }
  db.AggregateAdjacents();
  if (! db.Save()) {
    cerr << "Failed to save database in " << optargs.Get<string>('o') << "!\n";
    return 1;
  }
  return 0;
}
