//===========================================================================
// @(#) $DwmPath: dwm/libDwmRDAP/tags/libDwmRDAP-0.2.1/include/DwmRDAPIpv4Routes.hh 9335 $
// @(#) $Id: DwmRDAPIpv4Routes.hh 9335 2017-05-10 17:36:56Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 1999-2005, 2016, 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//    endorse or promote products derived from this software without
//    specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmRDAPIpv4Routes.hh
//!  \brief Dwm::RDAP::Ipv4Routes class template definition
//---------------------------------------------------------------------------

#ifndef _DWMRDAPIPV4ROUTES_HH_
#define _DWMRDAPIPV4ROUTES_HH_

extern "C" {
  #include <assert.h>
}

#include <algorithm>
#include <unordered_map>
#include <shared_mutex>
#include <vector>

#include "DwmPortability.hh"
#include "DwmIO.hh"
#include "DwmGZIO.hh"
#include "DwmBZ2IO.hh"
#include "DwmIpv4Prefix.hh"
#include "DwmOperators.hh"
#include "DwmReadable.hh"
#include "DwmWritable.hh"
#include "DwmGZReadable.hh"
#include "DwmGZWritable.hh"
#include "DwmBZ2Readable.hh"
#include "DwmBZ2Writable.hh"

namespace Dwm {

  namespace RDAP {
    
    //------------------------------------------------------------------------
    //!  This template class provides an associative container keyed by
    //!  IPv4 addresses, with longest-match searching.  While this isn't
    //!  as speedy to search as Patricia or radix, it is based on STL
    //!  containers and is hence easy to understand and maintain (note how 
    //!  few lines of actual code are here).
    //!
    //!  I/O functionality is provided, but the real work there is done
    //!  in the Dwm::IO class.
    //!
    //------------------------------------------------------------------------
    template <typename _valueT>
    class Ipv4Routes
      : public Readable, public Writable, 
        public GZReadable, public GZWritable,
        public BZ2Readable, public BZ2Writable
    {
    public:
      typedef std::map<Ipv4Address, _valueT>        _RepSubType;
      typedef typename _RepSubType::const_iterator  const_iterator;
      
      //----------------------------------------------------------------------
      //!  Constructor
      //----------------------------------------------------------------------
      Ipv4Routes<_valueT>()
          : _maps(33)
      {
      }
      
      //----------------------------------------------------------------------
      //!  Clears all entries.
      //----------------------------------------------------------------------
      void Clear()
      {
        std::shared_lock<std::shared_mutex>  lock(_mtx);
        for (uint8_t i = 0; i < 33; ++i)
          _maps[i].clear();
        return;
      }
      
      //----------------------------------------------------------------------
      //!  Returns true if there are no entries.
      //----------------------------------------------------------------------
      bool Empty() const
      {
        bool  rc = true;
        std::shared_lock<std::shared_mutex>  lock(_mtx);
        for (uint8_t i = 0; i < 33; ++i) {
          if (! _maps[i].empty()) {
            rc = false;
            break;
          }
        }
        return(rc);
      }

      //----------------------------------------------------------------------
      //!  Adds an entry.  Returns false (and does nothing) if there was
      //!  already an entry present for \c prefix.
      //----------------------------------------------------------------------
      bool Add(const Ipv4Prefix & prefix, const _valueT & value) 
      {
        bool  rc = false;

        std::shared_lock<std::shared_mutex>  lock(_mtx);
        typename _RepSubType::iterator  iter = 
          _maps[prefix.MaskLength()].find(prefix.Network());
        if (iter == _maps[prefix.MaskLength()].end()) {
          _maps[prefix.MaskLength()][prefix.Network()] = value;
          rc = true;
        }
        return(rc);
      }
      
      //----------------------------------------------------------------------
      //!  operator [] works like you would expect from an STL map.
      //----------------------------------------------------------------------
      _valueT & operator [] (const Ipv4Prefix & prefix)
      {
        std::shared_lock<std::shared_mutex>  lock(_mtx);
        return(_maps[prefix.MaskLength()][prefix.Network()]);
      }
      
      //----------------------------------------------------------------------
      //!  Deletes the entry for \c prefix.  Returns true on success, false
      //!  if an entry wasn't found for \c prefix.
      //----------------------------------------------------------------------
      bool Delete(const Ipv4Prefix & prefix)
      {
        bool  rc = false;
        std::lock_guard<std::shared_mutex>  lock(_mtx);
        typename _RepSubType::iterator  iter = 
          _maps[prefix.MaskLength()].find(prefix.Network());
        if (iter != _maps[prefix.MaskLength()].end()) {
          _maps[prefix.MaskLength()].erase(iter);
          rc = true;
        }
        return(rc);
      }
      
      //----------------------------------------------------------------------
      //!  Find the entry for the given \c prefix.  If an entry is found,
      //!  the value is stored in \c match and true is returned.  Else false
      //!  is returned.
      //----------------------------------------------------------------------
      bool Find(const Ipv4Prefix & prefix, _valueT & match) const
      {
        bool  rc = false;
        std::shared_lock<std::shared_mutex>  lock(_mtx);
        if (! _maps[prefix.MaskLength()].empty()) {
          typename _RepSubType::const_iterator  iter =
            _maps[prefix.MaskLength()].find(prefix.Network());
          if (iter != _maps[prefix.MaskLength()].end()) {
            match = iter->second;
            rc = true;
          }
        }
        return(rc);
      }
      
      //----------------------------------------------------------------------
      //!  Finds the longest match for \c ipAddr.  Places the result in
      //!  \c match and returns true on success.  Returns false if no
      //!  match was found for \c ipAddr.
      //----------------------------------------------------------------------
      bool FindLongest(const Ipv4Address & ipAddr,
                       std::pair<Ipv4Prefix,_valueT> & match) const
      {
        bool  rc = false;
        
        Ipv4Prefix   lp(ipAddr, 32);
        
        typename _RepSubType::const_iterator  iter;
        for (int8_t i = 32; i >= 0; --i) {
          std::shared_lock<std::shared_mutex>  lock(_mtx);
          if (_maps[i].empty())
            continue;
          lp.MaskLength(i);
          iter = _maps[i].find(lp.Network());
          if (iter != _maps[i].end()) {
            match.first = lp;
            match.second = iter->second;
            rc = true;
            break;
          }
        }
        return(rc);
      }
      
      //----------------------------------------------------------------------
      //!  Finds the longest match for \c ipAddr.  Places the result in
      //!  \c match and returns true on success.  Returns false if no
      //!  match was found for \c ipAddr.
      //!  Note that match.second is a pointer to const for the value stored
      //!  under the prefix.  That means you need to be careful using this
      //!  member; don't call free() or delete() on match.second.
      //----------------------------------------------------------------------
      bool
      FindLongest(const Ipv4Address & ipAddr,
                  std::pair<Ipv4Prefix, const _valueT *> & match) const
      {
        bool  rc = false;
        Ipv4Prefix   lp(ipAddr, 32);
        typename _RepSubType::const_iterator  iter;
        for (int8_t i = 32; i >= 0; --i) {
          std::shared_lock<std::shared_mutex>  lock(_mtx);
          if (_maps[i].empty())
            continue;
          lp.MaskLength(i);
          iter = _maps[i].find(lp.Network());
          if (iter != _maps[i].end()) {
            match.first = lp;
            match.second = &(iter->second);
            rc = true;
            break;
          }
        }
        return(rc);
      }
        
      //----------------------------------------------------------------------
      //!  Finds all matches for \c ipAddr.  Places the results in \c matches
      //!  (in longest-match-first order) and returns true if any matches were 
      //!  found.  Returns false if no matches were found.
      //----------------------------------------------------------------------
      bool Find(const Ipv4Address & ipAddr,
                std::vector<std::pair<Ipv4Prefix,_valueT> > & matches) const
      {
        if (! matches.empty())
          matches.clear();
        
        typename _RepSubType::const_iterator  iter;
        
        for (int8_t i = 32; i >= 0; --i) {
          std::shared_lock<std::shared_mutex>  lock(_mtx);
          if (_maps[i].empty())
            continue;
          Ipv4Prefix  prefix(ipAddr, i);
          iter = _maps[i].find(prefix.Network());
          if (iter != _maps[i].end()) {
            std::pair<Ipv4Prefix,_valueT> match(prefix, iter->second);
            matches.push_back(match);
          }
        }
        return(! matches.empty());
      }
      
      //----------------------------------------------------------------------
      //!  operator ==
      //!  It's unlikely you'd ever need to use this, and it's expensive.
      //!  It's mainly here for unit testing.
      //----------------------------------------------------------------------
      bool operator == (const Ipv4Routes<_valueT> & r) const
      {
        for (int8_t i = 32; i >= 0; --i) {
          std::shared_lock<std::shared_mutex>  lock(_mtx);
          if (_maps[i] != r._maps[i])
            return(false);
        }
        return(true);
      }

      //----------------------------------------------------------------------
      //!  operator !=
      //!  It's unlikely you'd ever need to use this, and it's expensive.
      //!  It's mainly here for unit testing.
      //----------------------------------------------------------------------
      bool operator != (const Ipv4Routes<_valueT> & r) const
      {
        return(! (*this == r));
      }
      
      //----------------------------------------------------------------------
      //!  Returns the number of routes.
      //----------------------------------------------------------------------
      uint32_t Size() const
      {
        uint32_t  rc = 0;
        for (uint8_t i = 0; i < 33; ++i) {
          std::shared_lock<std::shared_mutex>  lock(_mtx);
          rc += _maps[i].size();
        }
        return(rc);
      }
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      void HashSizes(std::vector<std::pair<uint8_t, uint32_t> > & sizes) const
      {
        if (! sizes.empty())
          sizes.clear();
        for (uint8_t i = 0; i < 33; ++i) {
          std::shared_lock<std::shared_mutex>  lock(_mtx);
          if (! _maps[i].empty()) {
            sizes.push_back(std::pair<uint8_t,uint32_t>(i,_maps[i].size()));
          }
        }
        return;
      }
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      uint32_t StreamedLength() const
      {
        std::shared_lock<std::shared_mutex>  lock(_mtx);
        return(IO::StreamedLength(_maps));
      }
      
      //----------------------------------------------------------------------
      //!  Reads the routes from an istream.  Returns the istream.
      //----------------------------------------------------------------------
      std::istream & Read(std::istream & is)
      {
        std::lock_guard<std::shared_mutex>  lock(_mtx);
        return(IO::Read(is, _maps));
      }
      
      //----------------------------------------------------------------------
      //!  Writes the routes to an ostream.  Returns the ostream.
      //----------------------------------------------------------------------
      std::ostream & Write(std::ostream & os) const
      {
        std::shared_lock<std::shared_mutex>  lock(_mtx);
        return(IO::Write(os, _maps));
      }
      
      //----------------------------------------------------------------------
      //!  Reades the routes from a FILE pointer.  Returns 1 on success,
      //!  0 on failure.
      //----------------------------------------------------------------------
      size_t Read(FILE *f)
      {
        std::lock_guard<std::shared_mutex>  lock(_mtx);
        return(IO::Read(f, _maps));
      }
      
      //----------------------------------------------------------------------
      //!  Writes the routes to a FILE pointer.  Returns 1 on success,
      //!  0 on failure.
      //----------------------------------------------------------------------
      size_t Write(FILE *f) const
      {
        std::shared_lock<std::shared_mutex>  lock(_mtx);
        return(IO::Write(f, _maps));
      }
      
      //----------------------------------------------------------------------
      //!  Reads the routes from a file descriptor.  Returns the number of
      //!  bytes read on success, -1 on failure.
      //----------------------------------------------------------------------
      ssize_t Read(int fd)
      {
        std::lock_guard<std::shared_mutex>  lock(_mtx);
        return(IO::Read(fd, _maps));
      }
      
      //----------------------------------------------------------------------
      //!  Writes the routes to a file descriptor.  Returns the number of
      //!  bytes written on success, -1 on failure.
      //----------------------------------------------------------------------
      ssize_t Write(int fd) const
      {
        std::shared_lock<std::shared_mutex>  lock(_mtx);
        return(IO::Write(fd, _maps));
      }
      
      //----------------------------------------------------------------------
      //!  Reads the routes from a gzFile.  Returns the number of
      //!  bytes read on success, -1 on failure.
      //----------------------------------------------------------------------
      int Read(gzFile gzf)
      {
        std::lock_guard<std::shared_mutex>  lock(_mtx);
        return(GZIO::Read(gzf, _maps));
      }
    
      //----------------------------------------------------------------------
      //!  Writes the routes to a gzFile.  Returns the number of bytes
      //!  written on success, -1 on failure.
      //----------------------------------------------------------------------
      int Write(gzFile gzf) const
      {
        std::shared_lock<std::shared_mutex>  lock(_mtx);
        return(GZIO::Write(gzf, _maps));
      }

      //----------------------------------------------------------------------
      //!  Reads the routes from a BZFILE pointer.  Returns the number of
      //!  bytes read on success, -1 on failure.
      //----------------------------------------------------------------------
      int BZRead(BZFILE *bzf)
      {
        std::lock_guard<std::shared_mutex>  lock(_mtx);
        return(BZ2IO::BZRead(bzf, _maps));
      }
      
      //----------------------------------------------------------------------
      //!  Writes the routes to a BZFILE pointer.  Returns the number of bytes
      //!  written on success, -1 on failure.
      //----------------------------------------------------------------------
      int BZWrite(BZFILE *bzf) const
      {
        std::shared_lock<std::shared_mutex>  lock(_mtx);
        return(BZ2IO::BZWrite(bzf, _maps));
      }

      void SortByKey(std::vector<std::pair<Ipv4Prefix,_valueT>> & target,
                     bool ascending = true) const
      {
        if (! target.empty())
          target.clear();
        std::shared_lock<std::shared_mutex>  lock(_mtx);
        if (! this->_maps.empty()) {
          target.resize(this->Size());
          uint32_t  pfx = 0;
          for (uint8_t hashNum = 0; hashNum < 33; ++hashNum) {
            if (! this->_maps[hashNum].empty()) {
              auto  hiter = this->_maps[hashNum].begin();
              for ( ; hiter != this->_maps[hashNum].end(); ++hiter) {
                target[pfx].first = Ipv4Prefix(hiter->first, hashNum);
                target[pfx].second = hiter->second;
                ++pfx;
              }
            }
          }
          if (! target.empty()) {
            if (ascending) {
              std::sort(target.begin(), target.end(), KeyLess());
            }
            else {
              std::sort(target.begin(), target.end(), KeyGreater());
            }
          }
        }
        
        return;
      }
      
      //----------------------------------------------------------------------
      //!  Sorts the contained pair<Ipv4Prefix,_valueT> values into a
      //!  vector, in descending order by the value stored for each prefix.
      //!  For example, if you had an Ipv4Routes<uint32_t> object,
      //!  \c target would contain the pair<Ipv4Prefix,_valueT> objects
      //!  sorted in descending order by the uint32_t values.
      //----------------------------------------------------------------------
      void SortByValue(std::vector<std::pair<Ipv4Prefix,_valueT> > & target)
      {
        if (! target.empty())
          target.clear();
        std::shared_lock<std::shared_mutex>  lock(_mtx);
        if (! this->_maps.empty()) {
          target.resize(this->Size());
          typename std::vector<_RepSubType>::const_iterator  iter = 
            this->_maps.begin();
          uint32_t  pfx = 0;
          for (uint8_t hashNum = 0; hashNum < 33; ++hashNum) {
            if (! this->_maps[hashNum].empty()) {
              typename _RepSubType::const_iterator  hiter = 
                this->_maps[hashNum].begin();
              for ( ; hiter != iter->end(); ++hiter) {
                target[pfx].first = Ipv4Prefix(hiter->first, hashNum);
                target[pfx].second = hiter->second;
                ++pfx;
              }
            }
          }
          if (! target.empty())
            std::sort(target.begin(), target.end(), ValueGreater());
        }
        
        return;
      }

      //----------------------------------------------------------------------
      //!  Returns the number of addresses covered by the contained prefixes,
      //!  not including 0/0.
      //----------------------------------------------------------------------
      uint32_t AddressesCovered() const
      {
        uint32_t  rc = 0;
        for (int8_t hashNum = 32; hashNum > 0; --hashNum) {
          typename _RepSubType::const_iterator  hiter =
            _maps[hashNum].begin();
          for ( ; hiter != _maps[hashNum].end(); ++hiter) {
            bool  foundWider = false;
            for (int8_t widerHash = hashNum - 1; widerHash > 0; --widerHash) {
              if (_maps[widerHash].find(Ipv4Prefix(hiter->first,widerHash).Network())
                  != _maps[widerHash].end()) {
                //  found wider match, don't count
                foundWider = true;
                break;
              }
            }
            if (! foundWider) {
              rc += ((uint32_t)1 << (32 - hashNum));
            }
          }
        }
        return rc;
      }

      struct KeyGreater
      {
      public:
        bool operator () (const std::pair<Ipv4Prefix,_valueT> & e1,
                          const std::pair<Ipv4Prefix,_valueT> & e2) const
        {
          return(e1.first > e2.first);
        }
      };
      
      struct KeyLess
      {
      public:
        bool operator () (const std::pair<Ipv4Prefix,_valueT> & e1,
                          const std::pair<Ipv4Prefix,_valueT> & e2) const
        {
          return(e1.first < e2.first);
        }
      };
      
      struct ValueGreater
      {
      public:
        bool operator () (const std::pair<Ipv4Prefix,_valueT> & e1,
                          const std::pair<Ipv4Prefix,_valueT> & e2) const
        {
          return(e1.second > e2.second);
        }
      };
      
      
    protected:
      std::vector<_RepSubType>   _maps;
      mutable std::shared_mutex  _mtx;
    };

  }  // namespace RDAP
  
}  // namespace Dwm

#endif  // _DWMRDAPIPV4ROUTES_HH_
