//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmRDAPResponder.cc 9458 2017-06-06 05:08:27Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmRDAPResponder.cc
//!  \brief Dwm::RDAP::Responder class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <netinet/in.h>
  #include <netinet/tcp.h>
  #include <arpa/inet.h>
#ifndef __APPLE__
  #ifndef __linux__
    #include <pthread_np.h>
  #else
    #include <pthread.h>
  #endif
#endif
}

#include "DwmSysLogger.hh"
#include "DwmSvnTag.hh"
#include "DwmAuth.hh"
#include "DwmAuthPeerAuthenticator.hh"
#include "DwmAuthSymCryptoMessage.hh"
#include "libDwmRDAPPortability.hh"
#include "DwmRDAPQuery.hh"
#include "DwmRDAPRequestMessage.hh"
#include "DwmRDAPResponseMessage.hh"
#include "DwmRDAPResponder.hh"

static const Dwm::SvnTag  svntag("$DwmPath: dwm/libDwmRDAP/tags/libDwmRDAP-0.2.1/apps/dwmrdapd/DwmRDAPResponder.cc 9458 $");

using namespace std;

namespace Dwm {

  namespace RDAP {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Responder::Responder(Socket & s, const string & myPrivKeyPath,
                         const string & authorizedKeysPath,
                         Ipv4CountryDb & db, const CountryCodes & cc)
        : _db(db), _cc(cc), _socket(), _myPrivKeyPath(myPrivKeyPath),
          _querySessions(), _authorizedKeysPath(authorizedKeysPath),
          _run(true)
    {
      if (s.Accept(_socket, _clientAddr)) {
        Syslog(LOG_INFO, "Accepted connection from %s:%hu, fd %d",
               inet_ntoa(_clientAddr.sin_addr), ntohs(_clientAddr.sin_port),
               (int)_socket);
        _thread = thread(&Responder::Run, this);
        _running = true;
      }
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::Stop()
    {
      _run = false;
      return Join();
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::Join()
    {
      bool  rc = false;
      if (_thread.joinable() && (! _running)) {
        _thread.join();
        rc = true;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::IsValidQueryRequest(const Json::Value & request) const
    {
      bool  rc = false;
      if (request.isMember("type")) {
        if (request["type"].isString() && (request["type"] == "query")) {
          if (request.isMember("ipv4addrs")
              && request["ipv4addrs"].isArray()) {
            int  i = 0;
            for ( ; i < request["ipv4addrs"].size(); ++i) {
              if (! request["ipv4addrs"][i].isString()) {
                break;
              }
              struct in_addr  inAddr;
              if (inet_pton(AF_INET,
                            request["ipv4addrs"][i].asString().c_str(),
                            &inAddr) != 1) {
                break;
              }
            }
            if (i == request["ipv4addrs"].size()) {
              rc = true;
            }
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::IsValidDeleteRequest(const Json::Value & request) const
    {
      bool  rc = false;
      if (request.isMember("type")) {
        if (request["type"].isString() && (request["type"] == "delete")) {
          if (request.isMember("prefixes") && request["prefixes"].isArray()) {
            int  i = 0;
            for ( ; i < request["prefixes"].size(); ++i) {
              if (! request["prefixes"][i].isString()) {
                break;
              }
              struct in_addr  inAddr;
              if (inet_net_pton(AF_INET,
                                request["prefixes"][i].asString().c_str(),
                                &inAddr, sizeof(inAddr)) == -1) {
                break;
              }
            }
            if (i == request["prefixes"].size()) {
              rc = true;
            }
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::IsValidRequest(const Json::Value & request) const
    {
      return (IsValidQueryRequest(request)
              || IsValidDeleteRequest(request));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Json::Value
    Responder::DbMatchToJson(const Ipv4Address & addr,
                             const pair<Ipv4Prefix,Ipv4CountryDbValue> & match) const
    {
      Json::Value  jv = match.second.Json();
      jv["ipv4addr"] = (string)addr;
      jv["countryName"] = _cc.FindByCode(match.second.Code()).Name();
      jv["prefix"] = match.first.ToShortString();
      return jv;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Json::Value Responder::MulticastToJson(const Ipv4Address & addr)
    {
      Json::Value  jv;
      DateTime  now(TimeValue64(true));
      jv["ipv4addr"] = (string)addr;
      jv["country"] = "22";
      jv["countryName"] = "multicast";
      jv["lastUpdated"] = now.Formatted("%Y-%m-%d %H:%M");
      jv["lastChanged"] = now.Formatted("%Y-%m-%d %H:%M");
      jv["prefix"] = "224/4";
      return jv;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Json::Value Responder::PrivateToJson(const Ipv4Address & addr)
    {
      Json::Value  jv;
      DateTime  now(TimeValue64(true));
      jv["ipv4addr"] = (string)addr;
      jv["country"] = "10";
      jv["countryName"] = "private";
      jv["lastUpdated"] = now.Formatted("%Y-%m-%d %H:%M");
      jv["lastChanged"] = now.Formatted("%Y-%m-%d %H:%M");
      jv["prefix"] = Bootstrap::MatchingPrefix(addr).ToShortString();
      return jv;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Json::Value Responder::FutureToJson(const Ipv4Address & addr)
    {
      Json::Value  jv;
      DateTime  now(TimeValue64(true));
      jv["ipv4addr"] = (string)addr;
      jv["country"] = "24";
      jv["countryName"] = "future";
      jv["lastUpdated"] = now.Formatted("%Y-%m-%d %H:%M");
      jv["lastChanged"] = now.Formatted("%Y-%m-%d %H:%M");
      jv["prefix"] = "240/4";
      return jv;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool
    Responder::GetDbResult(const Ipv4Address & addr,
                           pair<Ipv4Prefix,Ipv4CountryDbValue> & match) const
    {
      return _db.FindLongest(addr, match);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Responder::IsSpecialAddress(const Ipv4Address & addr,
                                     Json::Value & jv)
    {
      bool  rc = false;
      vector<string> rdapServers = Bootstrap::RDAPServers(addr);
      if (rdapServers.size() == 1) {
        const string & rds = rdapServers.front();
        if (rds == "multicast") {
          jv = MulticastToJson(addr);
          rc = true;
        }
        else if (rds == "private") {
          jv = PrivateToJson(addr);
          rc = true;
        }
        else if (rds == "future") {
          jv = FutureToJson(addr);
          rc = true;
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Json::Value Responder::HandleDeleteRequest(const Json::Value & request,
                                               bool & dbupdated)
    {
      Json::Value  rc;
      dbupdated = false;
      Ipv4CountryDbValue  match;
      for (int i = 0; i < request["prefixes"].size(); ++i) {
        if (request["prefixes"][i].isString()) {
          Ipv4Prefix  pfx(request["prefixes"][i].asString());
          if (! _db.Find(pfx, match)) {
            rc["prefixes"][i] = string(pfx.ToShortString() + " not found");
          }
          else {
            _db.Delete(pfx);
            rc["prefixes"][i] =	string(pfx.ToShortString() + " deleted");
            dbupdated = true;
          }
        }
        else {
          rc["prefixes"][i] = string("invalid prefix\n");
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Json::Value Responder::HandleQueryRequest(const Json::Value & request,
                                              bool & dbupdated)
    {
      Json::Value  rc;
      vector<Ipv4Address>  addrs;
      for (int i = 0; i < request["ipv4addrs"].size(); ++i) {
        if (request["ipv4addrs"][i].isString()) {
          addrs.push_back(Ipv4Address(request["ipv4addrs"][i].asString()));
        }
      }
      TimeValue64  expireTime(true);
      expireTime -= TimeValue64(90 * 24 * 60 * 60, 0);
      for (int i = 0; i < addrs.size(); ++i) {
        Json::Value  dbVal;
        pair<Ipv4Prefix,Ipv4CountryDbValue>  match;
        if (IsSpecialAddress(addrs[i], rc[i])) {
        }
        else if (GetDbResult(addrs[i], match)
                 && (match.second.LastUpdated() > expireTime)) {
          rc[i] = DbMatchToJson(addrs[i], match);
        }
        else {
          Syslog(LOG_DEBUG, "Making RDAP query for %s",
                 ((string)addrs[i]).c_str());
          RDAP::IPv4Query     query(addrs[i]);
          RDAP::IPv4Response  rdapResult = _querySessions.Execute(query);
          if (_db.Update(rdapResult, 90)) {
            dbupdated = true;
          }
          else {
            Syslog(LOG_ERR, "Failed to update database for %s",
                   ((string)addrs[i]).c_str());
          }
          if (GetDbResult(addrs[i], match)) {
            rc[i] = DbMatchToJson(addrs[i], match);
          }
          else {
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Json::Value Responder::HandleRequest(const Json::Value & request,
                                         bool & dbupdated)
    {
      Json::Value       rc;
      if (IsValidQueryRequest(request)) {
        rc = HandleQueryRequest(request, dbupdated);
      }
      else if (IsValidDeleteRequest(request)) {
        rc = HandleDeleteRequest(request, dbupdated);
      }
      else {
        if (SysLogger::MinimumPriority() >= LOG_DEBUG) {
          Json::FastWriter  fw;
          Syslog(LOG_DEBUG, "invalid request %s", fw.write(request).c_str());
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Responder::Run()
    {
      set_pthread_name(_thread.native_handle(), "responder");
      Syslog(LOG_DEBUG, "Responder started on fd %d", (int)_socket);
      Json::Reader      jsonReader;
      Json::FastWriter  fw;
      RequestMessage    reqMsg;
      bool              savedb = false;

      int  noDelay = 1;
      _socket.Setsockopt(IPPROTO_TCP, TCP_NODELAY, &noDelay, sizeof(noDelay));
      Auth::PeerAuthenticator  peerAuth(_myPrivKeyPath, _authorizedKeysPath);
      string  theirId;
      if (peerAuth.Authenticate(_socket, theirId, _agreedKey)) {
        Syslog(LOG_INFO, "Authenticated client %s from %s:%hu",
               theirId.c_str(), inet_ntoa(_clientAddr.sin_addr),
               ntohs(_clientAddr.sin_port));
      }
      else {
        Syslog(LOG_INFO, "Authentication failed for client from %s:%hu",
               inet_ntoa(_clientAddr.sin_addr), ntohs(_clientAddr.sin_port));
        goto responderDone;
      }

      while (_run && (reqMsg.Read(_socket, _agreedKey) > 0)) {
        bool  dbupdated = false;
        const Json::Value  &reqJson = reqMsg.Json();
        Json::Value  &&respJson = HandleRequest(reqJson, dbupdated);
        if (dbupdated) {
          savedb = true;
        }
        ResponseMessage respMsg(respJson);
        if (! respMsg.Write(_socket, _agreedKey)) {
          break;
        }
      }
      if (savedb) {
        _db.Save();
      }
    responderDone:
      _running = false;
      Syslog(LOG_INFO, "Done with client %s:%hu",
             inet_ntoa(_clientAddr.sin_addr), ntohs(_clientAddr.sin_port));
      return;
    }

  }  // namespace RDAP

}  // namespace Dwm
