//===========================================================================
// @(#) $Name:$
// @(#) $Id: dwmrdap.cc 9458 2017-06-06 05:08:27Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file TestRDAPQuery.cc
//!  \brief trivial RDAP client
//---------------------------------------------------------------------------

#include <iomanip>
#include <iostream>
#include <cstdio>
#include <regex>

#include <Poco/Net/SSLManager.h>
#include <Poco/URI.h>

#include "DwmDateTime.hh"
#include "DwmIpv4Prefix.hh"
#include "DwmOptArgs.hh"
#include "DwmSvnTag.hh"
#include "DwmCountryCodes.hh"
#include "DwmRDAPBootstrap.hh"
#include "DwmRDAPQuery.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmRDAP/tags/libDwmRDAP-0.2.1/apps/dwmrdap/dwmrdap.cc 9458 $");

using namespace std;
using namespace Dwm;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void PrintResult(const RDAP::IPv4Response & result,
                        const CountryCodes & ccodes, bool verbose)
{
  if (verbose) {
    cout << result.toStyledString();
  }
  else {
    pair<Ipv4Address,Ipv4Address>  &&addrs = result.AddressRange();
    if ((addrs.first != INADDR_ANY) && (addrs.second != INADDR_ANY)) {
      string  &&country = result.Country();
      cout << setw(2) << country << ' ';
      CountryCode  cc = ccodes.FindByCode(country);
      if (! cc.Name().empty()) {
        cout << '(' << cc.Name() << ") ";
      }
      cout << addrs.first << " - " << addrs.second;
      DateTime  &&lastChanged = result.LastChanged();
      if (lastChanged.GetTimeValue64().Secs() > 0) {
        cout << lastChanged.Formatted("  %Y-%m-%d  ");
      }
      vector<Ipv4Prefix>  &&prefixes = result.Prefixes();
      if (! prefixes.empty()) {
        auto  it = prefixes.begin();
        cout << " (" << it->ToShortString();
        ++it;
        for ( ; it != prefixes.end(); ++it) {
          cout << ", " << it->ToShortString();
        }
        cout << ")\n";
      }
    }
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  Dwm::OptArgs  optargs;
  optargs.AddOptArg("v", "verbose", false, "false", "verbose output");
  optargs.AddNormalArg("IP_address", true);
  int nextArg = optargs.Parse(argc, argv);

  if (nextArg >= argc) {
    optargs.Usage(argv[0]);
    return 1;
  }

#if (POCO_VERSION >= 0x01040200)
  Poco::Net::initializeSSL();
#endif
  
  RDAP::Bootstrap::Load("file:///usr/local/etc/rdap_ipv4.json");
  CountryCodes  ccodes;
  ccodes.ReadJson("/usr/local/etc/country_codes.json");

  for (int arg = nextArg; arg < argc; ++arg) {
    Ipv4Address         ip(argv[arg]); 
    RDAP::IPv4Query     query(ip);
    RDAP::IPv4Response  result = query.Execute();
    PrintResult(result, ccodes, optargs.Get<bool>("verbose"));
  }
  return 0;
}
