//===========================================================================
// @(#) $Name:$
// @(#) $Id: TestRDAPQuery.cc 9458 2017-06-06 05:08:27Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file TestRDAPQuery.cc
//!  \brief Unit tests for Dwm::RDAP::Query and descendants
//---------------------------------------------------------------------------

#include <iomanip>
#include <iostream>
#include <Poco/Net/SSLManager.h>
#include <Poco/URI.h>

#include "DwmIpv4Prefix.hh"
#include "DwmOptArgs.hh"
#include "DwmSvnTag.hh"
#include "DwmUnitAssert.hh"
#include "DwmRDAPBootstrap.hh"
#include "DwmRDAPQuery.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmRDAP/tags/libDwmRDAP-0.1.9/tests/TestRDAPQuery.cc 9458 $");

using namespace std;
using namespace Dwm;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void TestQuery(const string & ip, bool debug)
{
  Ipv4Address         addr(ip);
  RDAP::IPv4Query     query(addr);
  RDAP::IPv4Response  result = query.Execute();

  pair<Ipv4Address, Ipv4Address>  &&addrs = result.AddressRange();
  
  if (UnitAssert((addrs.first != INADDR_ANY)
                 && (addrs.second != INADDR_ANY))) {
    if (debug) {
      cout << setw(2) << result.Country() << ' '
           << addrs.first << " - " << addrs.second;
      vector<Ipv4Prefix>  &&prefixes = result.Prefixes();
      if (! prefixes.empty()) {
        auto  it = prefixes.begin();
        cout << " (" << it->ToShortString();
        ++it;
        for ( ; it != prefixes.end(); ++it) {
          cout << ", " << it->ToShortString();
        }
        cout << ")\n";
      }
    }
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  Dwm::OptArgs  optargs;
  optargs.AddOptArg("p", "print", false, "false", "print some query results");
  optargs.Parse(argc, argv);
#if (POCO_VERSION >= 0x01040200)
  Poco::Net::initializeSSL();
#endif
  RDAP::Bootstrap::Load("file:./ipv4.json");

  TestQuery("192.172.226.1", optargs.Get<bool>('p'));  // ARIN
  TestQuery("222.214.218.1", optargs.Get<bool>('p'));  // APNIC
  TestQuery("213.128.85.1", optargs.Get<bool>('p'));   // RIPE
  TestQuery("201.213.164.1", optargs.Get<bool>('p'));  // LACNIC
  TestQuery("41.1.1.1", optargs.Get<bool>('p'));       // AFRINIC
  
  if (Assertions::Total().Failed() > 0) {
    Assertions::Print(cerr, true);
    return(1);
  }
  else {
    cout << Assertions::Total() << " passed\n";
  }

}
