//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmRDAPQuerySessions.cc 9306 2017-05-08 09:04:02Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmRDAPQuerySessions.cc
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

#include <regex>

#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmRDAPQuerySessions.hh"

#include <Poco/Net/HTTPSClientSession.h>
#include <Poco/Net/HTTPRequest.h>
#include <Poco/Net/HTTPResponse.h>
#include <Poco/URI.h>

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmRDAP/tags/libDwmRDAP-0.1.9/src/DwmRDAPQuerySessions.cc 9306 $");

using namespace std;
using Poco::Net::Context;
using Poco::Net::HTTPClientSession;
using Poco::Net::HTTPSClientSession;
using Poco::Net::HTTPMessage;
using Poco::Net::HTTPRequest;
using Poco::Net::HTTPResponse;

namespace Dwm {

  namespace RDAP {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    QuerySessions::QuerySessions()
        : _sessions()
    {
      _context = new Context(Context::CLIENT_USE, "", "", "",
                             Context::VERIFY_NONE, 9, false,
                             "ALL:!ADH:!LOW:!EXP:!MD5:@STRENGTH");
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    QuerySessions::~QuerySessions()
    {
      for (auto sess : _sessions) {
        delete sess.second;
        sess.second = 0;
      }
      _sessions.clear();
      //      delete _context;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void QuerySessions::FindVcardArrays(const Json::Value & jv,
                                        vector<Json::Value> & vcardArrays)
    {
      if (jv.isObject()) {
        if (jv.isMember("vcardArray") && jv["vcardArray"].isArray()) {
          vcardArrays.push_back(jv["vcardArray"]);
        }
        vector<string>  &&memberNames = jv.getMemberNames();
        remove_if(memberNames.begin(), memberNames.end(),
                  [&] (const string & name)
                  { return (name == "vcardArray"); });
        for (auto mn : memberNames) {
          FindVcardArrays(jv[mn], vcardArrays);
        }
      }
      else if (jv.isArray()) {
        for (int i = 0; i < jv.size(); ++i) {
          if (jv[i].isObject() && jv[i].isMember("vcardArray")
              && jv[i]["vcardArray"].isArray()) {
            vcardArrays.push_back(jv[i]["vcardArray"]);
          }
          FindVcardArrays(jv[i], vcardArrays);
        }
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string QuerySessions::FindCountryInVcard(const Json::Value & jv)
    {
      static const map<string,string>  ccodeMap = {
        { "ANGUILLA",                             "AI" },
        { "ANTARCTICA",                           "AQ" },
        { "ANTIGUA AND BARBUDA",                  "AG" },
        { "BAHAMAS",                              "BS" },
        { "BARBADOS",                             "BB" },
        { "BERMUDA",                              "BM" },
        { "BOUVET ISLAND",                        "BV" },
        { "CANADA",                               "CA" },
        { "CAYMAN ISLANDS",                       "KY" },
        { "DOMINICA",                             "DM" },
        { "GRENADA",                              "GD" },
        { "GUADELOUPE",                           "GP" },
        { "HEARD AND MC DONALD ISLANDS",          "HM" },
        { "JAMAICA",                              "JM" },
        { "MARTINIQUE",                           "MQ" },
        { "MONTSERRAT",                           "MS" },
        { "PUERTO RICO",                          "PR" },
        { "SAINT BARTHELEMY",                     "BL" },
        { "SAINT KITTS AND NEVIS",                "KN" },
        { "SAINT LUCIA",                          "LC" },
        { "SAINT VINCENT AND THE GRENADINES",     "VC" },
        { "ST. PIERRE AND MIQUELON",              "PM" },
        { "ST. HELENA",                           "SH" },
        { "ST. MARTIN",                           "MF" },
        { "TURKS AND CAICOS ISLANDS",             "TC" },
        { "UNITED STATES",                        "US" },
        { "UNITED STATES MINOR OUTLYING ISLANDS", "UM" },
        { "VIRGIN ISLANDS (BRITISH)",             "VG" },
        { "VIRGIN ISLANDS (U.S.)",                "VI" }
      };
      
      vector<Json::Value>  vcardArrays;
      FindVcardArrays(jv, vcardArrays);
      for (auto vca : vcardArrays) {
        for (int vi = 0; vi < vca.size(); ++vi) {
          const Json::Value &  vc = vca[vi];
          if (vc.isArray()) {
            for (int vci = 0; vci < vc.size(); ++vci) {
              if (vc[vci].isArray()) {
                const Json::Value & vcm = vc[vci];
                for (int vcmi = 0; vcmi < vcm.size(); ++vcmi) {
                  if (vcm[vcmi].isObject()) {
                    if (vcm[vcmi].isMember("label")
                        && vcm[vcmi]["label"].isString()) {
                      string  label = vcm[vcmi]["label"].asString();
                      string::size_type  idx = label.find_last_of('\n');
                      if ((idx != string::npos) && (idx < (label.size() - 1))) {
                        auto  cc = ccodeMap.find(label.substr(idx+1));
                        if (cc != ccodeMap.end()) {
                          if (cc->second != "US") {
                            return cc->second;
                          }
                        }
                      }
                    }
                  }
                }
              }
            }
          }
        }
      }
      return string("US");
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RDAP::IPv4Response QuerySessions::Execute(const Query & query)
    {
      RDAP::IPv4Response  rc;
      HTTPClientSession  *sess = GetSession(query);
      if (sess) {
        Poco::URI    uri(query.Uri());
        HTTPRequest  req(HTTPRequest::HTTP_GET, uri.getPathAndQuery(),
                         HTTPMessage::HTTP_1_1);
        try {
          ostream & os = sess->sendRequest(req);
          if (os) {
            HTTPResponse  resp;
            istream  & is = sess->receiveResponse(resp);
            Json::Reader  reader;
            if (reader.parse(is, rc)) {
              Query::FixIpv4Stupidity(rc["startAddress"]);
              Query::FixIpv4Stupidity(rc["endAddress"]);
              if (uri.getHost().find_last_of("arin.net") != string::npos) {
                if ((! rc.isMember("country"))
                    || (rc["country"].isString()
                        && rc["country"].asString().empty())) {
                  //  Fudge for more ARIN stupidity
                  string  &&vcardCcode = FindCountryInVcard(rc);
                  if (! vcardCcode.empty()) {
                    rc["country"] = vcardCcode;
                  }
                  else {
                    rc["country"] = "US";
                  }
                }
              }
            }
            else {
              Syslog(LOG_ERR, "Failed to parse response from %s", uri.getHost().c_str());
            }
          }
          else {
            Syslog(LOG_ERR, "%s sendRequest failed", uri.getHost().c_str());
          }
        }
        catch (Poco::Exception &ex) {
          Syslog(LOG_ERR, "Exception %s for URI %s: %s",
                 ex.name(), uri.toString().c_str(),
                 ex.displayText().c_str());
        }
      }
      else {
      }
      
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    HTTPClientSession *QuerySessions::GetSession(const Query & query)
    {
      HTTPClientSession  *rc = (HTTPClientSession *)nullptr;
      Poco::URI  uri(query.Uri());
      string     sessKey(uri.getScheme() + "://" + uri.getHost() + ":"
                         + to_string(uri.getPort()));
      auto  sit = _sessions.find(sessKey);
      if (sit != _sessions.end()) {
        rc = sit->second;
      }
      else {
        HTTPClientSession  *sess;
        if (uri.getScheme() == "https") {
          rc = new HTTPSClientSession(uri.getHost(), uri.getPort(),
                                        _context);
        }
        else if (uri.getScheme() == "http") {
          rc = new HTTPClientSession(uri.getHost());
        }
        if (rc != nullptr) {
          rc->setKeepAlive(true);
          _sessions[sessKey] = rc;
        }
      }
      return rc;
    }
    
  }  // namespace RDAP

}  // namespace Dwm
