//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmRDAPResponse.hh 8926 2017-03-29 05:40:25Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmRDAPResponse.hh
//!  \brief Dwm::RDAP::Response class definition
//---------------------------------------------------------------------------

#ifndef _DWMRDAPRESPONSE_HH_
#define _DWMRDAPRESPONSE_HH_

#include <utility>
#include <json/json.h>

#include "DwmDateTime.hh"
#include "DwmIpv4Prefix.hh"

namespace Dwm {

  namespace RDAP {

    //------------------------------------------------------------------------
    //!  Generically encapsulate an RDAP response.  This is just a wrapper
    //!  for Json::Value at the moment.
    //------------------------------------------------------------------------
    class Response : public Json::Value
    {
    public:
      Response() : Json::Value() { }
      Response(const Json::Value & jv) : Json::Value(jv) { }
    };
    
    //------------------------------------------------------------------------
    //!  Encapsulate an RDAP ip query response.  No data of our own, this
    //!  just prevides some convenient member functions.
    //------------------------------------------------------------------------
    class IPv4Response : public Response
    {
    public:
      //----------------------------------------------------------------------
      //!  Default constructor
      //----------------------------------------------------------------------
      IPv4Response() : Response() { }
      //----------------------------------------------------------------------
      //!  Construct from a Json::Value (a generic RDAP response).
      //----------------------------------------------------------------------
      IPv4Response(const Json::Value & jv) : Response(jv) { }

      //----------------------------------------------------------------------
      //!  Returns the time of the 'last changed' event in the response.
      //!  If it doesn't exist, returns the UNIX epoch.
      //----------------------------------------------------------------------
      DateTime LastChanged() const;

      //----------------------------------------------------------------------
      //!  Returns the address range in the response.  If not found,
      //!  both members of the returned pair will he set to INADDR_ANY.
      //----------------------------------------------------------------------
      std::pair<Ipv4Address,Ipv4Address> AddressRange() const;

      //----------------------------------------------------------------------
      //!  Returns the prefixes required to cover the address range in the
      //!  response.  If this fails, an empty vector will be returned.
      //----------------------------------------------------------------------
      std::vector<Ipv4Prefix> Prefixes() const;
      
      //----------------------------------------------------------------------
      //!  Returns the country code in the response.  If there was no
      //!  country code, returns an empty string.
      //----------------------------------------------------------------------
      std::string Country() const;
    };
    
  }  // namespace RDAP

}  // namespace Dwm

#endif  // _DWMRDAPRESPONSE_HH_
