//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmRDAPFetcher.hh 9156 2017-04-19 05:36:59Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmRDAPFetcher.hh
//!  \brief Dwm::RDAP::Fetcher and Dwm::RDAP::FetchedEntry class definitions
//---------------------------------------------------------------------------

#ifndef _DWMRDAPFETCHER_HH_
#define _DWMRDAPFETCHER_HH_

#include <map>
#include <string>
#include <vector>

#include "DwmIpv4Prefix.hh"
#include "DwmSocket.hh"
#include "DwmTimeValue64.hh"
#include "DwmRDAPRequestMessage.hh"
#include "DwmRDAPResponseMessage.hh"

namespace Dwm {

  namespace RDAP {

    //------------------------------------------------------------------------
    //!  Encapsulates the IPv4 prefix, country code and lastUpdated parts
    //!  of a response from dwmrdapd.
    //------------------------------------------------------------------------
    class FetchedEntry
    {
    public:
      //----------------------------------------------------------------------
      //!  Default constructor
      //----------------------------------------------------------------------
      FetchedEntry() = default;
      
      //----------------------------------------------------------------------
      //!  Construct from the given IPv4 @c prefix, @c country and
      //!  @c lastUpdated.  This is a convenience constuctor used internally
      //!  by the Fetcher class.
      //----------------------------------------------------------------------
      FetchedEntry(const Ipv4Prefix & prefix, const std::string & country,
                   const TimeValue64 & lastUpdated);

      //----------------------------------------------------------------------
      //!  Returns the contained IPv4 prefix.
      //----------------------------------------------------------------------
      const Ipv4Prefix & Prefix() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained IPv4 prefix.
      //----------------------------------------------------------------------
      const Ipv4Prefix & Prefix(const Ipv4Prefix & prefix);
      
      //----------------------------------------------------------------------
      //!  Returns the contained country code.
      //----------------------------------------------------------------------
      const std::string & Country() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the contained country code.
      //----------------------------------------------------------------------
      const std::string & Country(const std::string & country);
      
      //----------------------------------------------------------------------
      //!  Returns the contained 'last updated' field.  This represents the
      //!  last time that dwmrdapd updated the entry by querying the RDAP
      //!  servers of the Network Information Center.
      //----------------------------------------------------------------------
      const TimeValue64 & LastUpdated() const;
      
      //----------------------------------------------------------------------
      //!  Sets and returns the 'last updated' field.  This represents the
      //!  last time that dwmrdapd updated the entry by querying the RDAP
      //!  servers of the Network Information Center.
      //----------------------------------------------------------------------
      const TimeValue64 & LastUpdated(const TimeValue64 & lastUpdated);
      
    private:
      Ipv4Prefix   _prefix;
      std::string  _country;
      TimeValue64  _lastUpdated;
    };

    typedef std::map<Ipv4Address,FetchedEntry>  FetchedEntryMap;
    
    //------------------------------------------------------------------------
    //!  Encapsulates interaction with dwmrdapd for fetching IPv4 prefix and
    //!  country information for IPv4 host addresses.
    //------------------------------------------------------------------------
    class Fetcher
    {
    public:
      //----------------------------------------------------------------------
      //!  Constructor
      //----------------------------------------------------------------------
      Fetcher();
      
      //----------------------------------------------------------------------
      //!  Opens a session with dwmrdapd on the given @c server and uses
      //!  @c privKeyPath and @c knownServicesPath for authentication.
      //!  Returns true on success, false on failure.
      //!
      //!  @c privKeyPath must be the path to a file containing your 2048-bit
      //!  RSA private key (and there must be a corresponding .pub file
      //!  containing the public part of the same key).  @c knownServicesPath
      //!  must be the path to a DwmAuth 'known_services' type of file and
      //!  must contain the server's public key.
      //----------------------------------------------------------------------
      bool OpenSession(const std::string & server,
                       const std::string & privKeyPath,
                       const std::string & knownServicesPath);

      //----------------------------------------------------------------------
      //!  Destructor.  This will close the session if it is open.
      //----------------------------------------------------------------------
      ~Fetcher();

      //----------------------------------------------------------------------
      //!  Fetches the entries corresponding to each of the IPv4 addresses
      //!  in @c addrs.  Note that the returned map is keyed by the addresses
      //!  in @c addrs, and that the number of entries in the map may be
      //!  smaller than the number of entries in @c addrs since we may fail
      //!  to find some entries.
      //----------------------------------------------------------------------
      FetchedEntryMap GetEntries(const std::vector<Ipv4Address> & addrs);

      //----------------------------------------------------------------------
      //!  Fetches the entry for the given @c addr.  On success, the first
      //!  member of the return value will be true and the second member
      //!  will contain the fetched entry.  On failure, the first member
      //!  of the return value will be false and the second member's contents
      //!  are undefined.
      //----------------------------------------------------------------------
      std::pair<bool,FetchedEntry> GetEntry(const Ipv4Address & addr);
      
      //----------------------------------------------------------------------
      //!  Closes the session with dwmrdapd.
      //----------------------------------------------------------------------
      void CloseSession();

    private:
      Socket       _rdapSocket;
      std::string  _agreedKey;

      bool RequestResponse(const RequestMessage & request,
                           ResponseMessage & response);
      static bool ResponseEntryHasData(const Json::Value & jv);
    };
    
  }  // namespace RDAP

}  // namespace Dwm

#endif  // _DWMRDAPFETCHER_HH_
