//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmIpv4CountryDb.hh 9367 2017-05-23 04:12:33Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmIpv4CountryDb.hh
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

#ifndef _DWMIPV4COUNTRYDB_HH_
#define _DWMIPV4COUNTRYDB_HH_

#include <iostream>
#include <mutex>
#include <string>
#include <tuple>

#include "DwmIpv4Prefix.hh"
#include "DwmRDAPIpv4Routes.hh"
#include "DwmReadable.hh"
#include "DwmWritable.hh"
#include "DwmTimeValue64.hh"
#include "DwmRDAPResponse.hh"

namespace Dwm {

  //--------------------------------------------------------------------------
  //!  Encapsulates a value stored in a Ipv4CountryDb.  Just a country code
  //!  and a timestamp.
  //--------------------------------------------------------------------------
  class Ipv4CountryDbValue
    : public Dwm::Readable, public Dwm::Writable,
      public GZReadable, public GZWritable,
      public BZ2Readable, public BZ2Writable
  {
  public:
    //------------------------------------------------------------------------
    //!  Default constructor.
    //------------------------------------------------------------------------
    Ipv4CountryDbValue();
    
    //------------------------------------------------------------------------
    //!  Construct from a 2-letter country code and a last updated timestamp.
    //------------------------------------------------------------------------
    Ipv4CountryDbValue(const std::string & code,
                       const TimeValue64 & lastChanged,
                       const TimeValue64 & lastUpdated = TimeValue64(true));

    //------------------------------------------------------------------------
    //!  Construct from an ip RDAP response.
    //------------------------------------------------------------------------
    Ipv4CountryDbValue(const Json::Value & rdapResponse);
    
    //------------------------------------------------------------------------
    //!  Returns the country code.
    //------------------------------------------------------------------------
    const std::string & Code() const;
    
    //------------------------------------------------------------------------
    //!  Sets and returns the country code.
    //------------------------------------------------------------------------
    const std::string & Code(const std::string code);
    
    //------------------------------------------------------------------------
    //!  Returns the last-updated timestamp.
    //------------------------------------------------------------------------
    const TimeValue64 & LastUpdated() const;
    
    //------------------------------------------------------------------------
    //!  Sets and returns the last-updated timestamp.
    //------------------------------------------------------------------------
    const TimeValue64 & LastUpdated(const TimeValue64 & lastUpdated);

    //------------------------------------------------------------------------
    //!  Returns the last-changed timestamp.  This comes from the registry
    //!  (ARIN, APNIC, RIPE, LACNIC, AFRINIC, et. al.).
    //------------------------------------------------------------------------
    const TimeValue64 & LastChanged() const;
    
    //------------------------------------------------------------------------
    //!  Sets and returns the last-updated timestamp.  This should be the
    //!  "last changed" event from the registry (ARIN, APNIC, RIPE, LACNIC,
    //!  AFRINIC, et. al.).
    //------------------------------------------------------------------------
    const TimeValue64 & LastChanged(const TimeValue64 & lastUpdated);

    //------------------------------------------------------------------------
    //!  Reads the value from an istream.  Returns the istream.
    //------------------------------------------------------------------------
    std::istream & Read(std::istream & is);
    
    //------------------------------------------------------------------------
    //!  Reads the value from a descriptor.  Returns the number of bytes
    //!  read on success, -1 on failure.
    //------------------------------------------------------------------------
    ssize_t Read(int fd);
    
    //------------------------------------------------------------------------
    //!  Reads the value from a FILE.  Returns 1 on success, 0 on failure
    //!  (fread() semantics).
    //------------------------------------------------------------------------
    size_t Read(FILE * f);
    
    //------------------------------------------------------------------------
    //!  Reads the value from a gzFile.  Returns the number of bytes read
    //!  on success, -1 on failure.
    //------------------------------------------------------------------------
    int Read(gzFile gzf);
    
    //------------------------------------------------------------------------
    //!  Reads the value from a BZFILE.  Returns the number of bytes read
    //!  on success, -1 on failure.
    //------------------------------------------------------------------------
    int BZRead(BZFILE *bzf);
    
    //------------------------------------------------------------------------
    //!  Writes the value to an ostream.  Returns the ostream.
    //------------------------------------------------------------------------
    std::ostream & Write(std::ostream & os) const;
    
    //------------------------------------------------------------------------
    //!  Writes the value to a descriptor.  Returns the number of bytes
    //!  written on success, -1 on failure.
    //------------------------------------------------------------------------
    ssize_t Write(int fd) const;
    
    //------------------------------------------------------------------------
    //!  Writes the value to a FILE.  Returns 1 on success, 0 on failure
    //!  (fwrite() semantics).
    //------------------------------------------------------------------------
    size_t Write(FILE *f) const;
    
    //------------------------------------------------------------------------
    //!  Writes the value to a gzFile.  Returns the number of bytes written
    //!  on success, -1 on failure.
    //------------------------------------------------------------------------
    int Write(gzFile gzf) const;
    
    //------------------------------------------------------------------------
    //!  Writes the value to a BZFILE.  Returns the number of bytes written
    //!  on success, -1 on failure.
    //------------------------------------------------------------------------
    int BZWrite(BZFILE *bzf) const;
    
    //------------------------------------------------------------------------
    //!  Returns the number of bytes that would be written if one of the
    //!  Write() members were called.
    //------------------------------------------------------------------------
    uint32_t StreamedLength() const;

    //------------------------------------------------------------------------
    //!  operator ==
    //------------------------------------------------------------------------
    bool operator == (const Ipv4CountryDbValue & val) const;

    Json::Value Json() const;
    
  private:
    //  <countryCode, lastUpdated, lastChanged>
    std::tuple<std::string,TimeValue64,TimeValue64>  _data;
  };
  
  //--------------------------------------------------------------------------
  //!  Encapsulates a database to map Ipv4Prefix to Ipv4CountryDbValue.
  //--------------------------------------------------------------------------
  class Ipv4CountryDb
    : public RDAP::Ipv4Routes<Ipv4CountryDbValue>
  {
  public:
    //------------------------------------------------------------------------
    //!  Construct from the given filename @c fn.  Will load the contents of
    //!  the file if possible.
    //------------------------------------------------------------------------
    Ipv4CountryDb(const std::string & fn = "/usr/local/etc/ipv4country.db");

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Add(const RDAP::IPv4Response & rdapResponse);

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Update(const RDAP::IPv4Response & rdapResponse, uint64_t expireDays);

    //------------------------------------------------------------------------
    //!  Delete entries that are more than @c days old and covered by
    //!  @c prefix.  Returns true if any entries were deleted.
    //------------------------------------------------------------------------
    bool DeleteCoveredExpired(const Ipv4Prefix & prefix, uint64_t days = 90);
      
    //------------------------------------------------------------------------
    //!  Reloads.
    //------------------------------------------------------------------------
    bool Reload();
    
    //------------------------------------------------------------------------
    //!  Saves the database.
    //------------------------------------------------------------------------
    bool Save() const;

    //------------------------------------------------------------------------
    //!  Dump the database to an ostream in JSON form.
    //------------------------------------------------------------------------
    void PrintJson(std::ostream & os) const;

    void CreateCountryAggregates();

    //------------------------------------------------------------------------
    //!  Deletes redundant specific prefixes (those covered by wider prefixes
    //!  in the same country).
    //------------------------------------------------------------------------
    void DeleteRedundantSpecifics();
    
    std::vector<Ipv4Prefix>
    PrefixesForCountries(const std::vector<std::string> & countries);

    std::vector<Ipv4Prefix>
    PrefixesForCountries(const std::string & regExpression,
                         bool minimalSet = true);

    void AggregateAdjacents();
    
  private:
    std::string          _filename;

    void CreateCountryAggregates(uint8_t maslken);
    static void CoalesceAdjacentPrefixes(std::vector<Ipv4Prefix> & vp);
    static void CoalesceContainedPrefixes(std::vector<Ipv4Prefix> & vp);
  };
  
}  // namespace Dwm

#endif  // _DWMIPV4COUNTRYDB_HH_
