//===========================================================================
// @(#) $Name:$
// @(#) $Id: dwmrdapd.cc 9458 2017-06-06 05:08:27Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file dwmrdapd.cc
//!  \brief dwmrdap daemon
//---------------------------------------------------------------------------

extern "C" {
  #include <fcntl.h>
  #include <unistd.h>
#ifdef __linux__
  #define O_SHLOCK  0
  #define O_EXLOCK  0
  #include <sys/file.h>
#endif
}

#include <cstdio>

#include <Poco/Net/SSLManager.h>

#include "DwmDaemonUtils.hh"
#include "DwmOptArgs.hh"
#include "DwmSignal.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmRDAPBootstrap.hh"
#include "DwmRDAPServerConfig.hh"
#include "DwmRDAPServer.hh"

static const Dwm::SvnTag  svntag("@(#) $DwmPath: dwm/libDwmRDAP/tags/libDwmRDAP-0.1.9/apps/dwmrdapd/dwmrdapd.cc 9458 $");

using namespace std;
using namespace Dwm;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void SavePID(const std::string & pidFile)
{
  pid_t  pid = getpid();
  int    fd = open(pidFile.c_str(), O_WRONLY|O_CREAT|O_TRUNC|O_EXLOCK, 0644);
  if (fd >= 0) {
#ifdef __linux__
      flock(fd, LOCK_EX);
#endif    
    string  &&pidstr = to_string(pid) + "\n";
    write(fd, pidstr.c_str(), pidstr.size());
    close(fd);
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  OptArgs  optargs;
  optargs.AddOptArg("c:", "config", false, "", "configuration file");
  optargs.AddOptArg("d", "debug", false, "false", "do not daemonize");
  optargs.AddOptArg("p:", "pidfile", false, "", "PID file");
  optargs.Parse(argc, argv);

  int  logOpts = LOG_PERROR;
  if (! optargs.Get<bool>('d')) {
    DaemonUtils::Daemonize();
    logOpts = 0;
  }
  
  Signal SigPipe(SIGPIPE);
  SigPipe.Block();

  Signal SigTerm(SIGTERM);
  SigTerm.Block();

  string  pidFile = optargs.Get<string>('p');
  if (pidFile.empty()) {
    pidFile = "/var/run/dwmrdapd.pid";
  }
  SavePID(pidFile);
  
  string  configFile = optargs.Get<string>('c');
  if (configFile.empty()) {
    configFile = "/usr/local/etc/dwmrdapd.conf";
  }

  int  rc = 1;
  RDAP::ServerConfig  serverConfig(configFile);
  if (serverConfig.Parse()) {
#if (POCO_VERSION >= 0x01040200)
    Poco::Net::initializeSSL();
#endif
    SysLogger::Open("dwmrdapd", logOpts, serverConfig.SyslogFacility());
    SysLogger::MinimumPriority(serverConfig.SyslogMinimumPriority());
    SysLogger::ShowFileLocation(serverConfig.SyslogShowFileLocations());

    RDAP::Bootstrap::Load(serverConfig.RDAPIpv4BootstrapFile());

    RDAP::Server  server("/usr/local/etc/dwmrdapd/id_rsa",
                         "/usr/local/etc/dwmrdapd/authorized_keys",
                         serverConfig);
    if (server.Start()) {
      for (;;) {
        sleep(1);
        if (SigTerm.IsPending()) {
          break;
        }
      }
      server.Stop();
      rc = 0;
    }
  }
  else {
    Syslog(LOG_ERR, "Failed to parse config file!");
    rc = 1;
  }

  remove(pidFile.c_str());
  return rc;
}

