//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmRDAPServer.cc 9458 2017-06-06 05:08:27Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmRDAPServer.cc
//!  \brief Dwm::RDAP::Server class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <netinet/in.h>
  #include <arpa/inet.h>
#ifndef __APPLE__
  #ifndef __linux__
    #include <pthread_np.h>
  #else
    #include <pthread.h>
  #endif
#endif
}

#include <sstream>

#include "DwmSocket.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmTimeValue64.hh"
#include "libDwmRDAPPortability.hh"
#include "DwmRDAPServer.hh"
#include "DwmAuthSymCryptoMessage.hh"

static const Dwm::SvnTag  svntag("@(#) $DwmPath: dwm/libDwmRDAP/tags/libDwmRDAP-0.1.9/apps/dwmrdapd/DwmRDAPServer.cc 9458 $");

using namespace std;

namespace Dwm {

  namespace RDAP {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Server::Server(const string & myPrivKeyPath,
                   const string & authorizedKeysPath,
                   const ServerConfig & config)
        : _myPrivKeyPath(myPrivKeyPath),
          _authorizedKeysPath(authorizedKeysPath), _config(config),
          _socket(), _cc()
    {
      _nextKeygenTime = chrono::system_clock::now() + chrono::hours(168);
      _cc.ReadJson(_config.CountryCodesFile());
      _db.AggregateAdjacents();
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Server::Start()
    {
      bool  rc = false;
      if (_socket.Open(PF_INET, SOCK_STREAM, 0)) {
        if (_socket.Bind(_config.TCPAddress(), _config.TCPPort())) {
          if (_mcastSocket.Open(PF_INET, SOCK_DGRAM, 0)) {
            if (_socket.Listen(10)) {
              rc = true;
              _run = true;
              _thread = thread(&Server::Run, this);
              Syslog(LOG_INFO, "Server started");
            }
            else {
              Syslog(LOG_ERR, "Listen(10) failed! %m");
            }
          }
          else {
            Syslog(LOG_ERR, "Failed to open UDP socket for multicast send");
            _socket.Close();
          }
        }
        else {
          Syslog(LOG_ERR, "socket.Bind(%s,%hu) failed! %m",
                 ((string)_config.TCPAddress()).c_str(), _config.TCPPort());
        }
      }
      else {
        Syslog(LOG_ERR, "socket.Open() failed! %m");
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::StopResponders()
    {
      auto  ri = _responders.begin();
      for ( ; ri != _responders.end(); ++ri) {
        (*ri)->Stop();
        Responder  *respToDel = *ri;
        _responders.erase(ri);
        delete respToDel;
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::Stop()
    {
      _run = false;
      StopResponders();
      if (_thread.joinable()) {
        _thread.join();
        _db.Save();
        Syslog(LOG_INFO, "Server stopped");
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::CleanupResponders()
    {
      auto  ri = _responders.begin();
      for ( ; ri != _responders.end(); ++ri) {
        if ((*ri)->Join()) {
          Responder  *respToDel = *ri;
          _responders.erase(ri);
          delete respToDel;
          break;
        }
      }
      return;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::Run()
    {
      set_pthread_name(_thread.native_handle(), "server");
      while (_run) {
        struct timeval  timeout;
        timeout.tv_sec = 0;
        timeout.tv_usec = 200000;
        fd_set  readFdSet, errFdSet;
        FD_ZERO(&readFdSet);
        FD_ZERO(&errFdSet);
        _socket.FdSet(readFdSet);
        _socket.FdSet(errFdSet);
        if (select(FD_SETSIZE, &readFdSet, 0, &errFdSet, &timeout) > 0) {
          if (_socket.FdIsSet(readFdSet)) {
            _responders.push_back(new Responder(_socket, _myPrivKeyPath,
                                                _authorizedKeysPath,
                                                _db, _cc));
          }
        }
#if 0
        SendMcastStatus();
#endif
        CleanupResponders();
        if (_responders.empty()) {
          chrono::system_clock::time_point  now =
            chrono::system_clock::now();
          if (now > _nextKeygenTime) {
            _nextKeygenTime = now + chrono::hours(168);
          }
        }
      }
      return;
    }

#if 0
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    void Server::SendMcastStatus()
    {
      static TimeValue64  lastSend(true);
      TimeValue64         now(true);
      TimeValue64         elapsed(now);
      elapsed -= lastSend;
      if (elapsed > TimeValue64(0, 500000)) {
        Json::FastWriter  fw;
        Auth::SymCrypto::Message  msg(_mySecret,
                                      fw.write(_doors.StatusJson()));
        if (msg.SendTo(_mcastSocket, 0, _config.MulticastAddr(),
                       _config.MulticastPort())) {
          lastSend = now;
        }
      }
      return;
    }
#endif
    
  }  // namespace RDAP

}  // namespace Dwm
