//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmPfTable.hh 9421 2017-06-05 05:08:37Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmPfTable.hh
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

#ifndef _DWMPFTABLE_HH_
#define _DWMPFTABLE_HH_

extern "C" {
  #include <sys/types.h>
  #include <net/if.h>
  #include <netinet/in_pcb.h>
#ifndef __APPLE__
  #include <net/pfvar.h>
#else
  #include "macos_net_pfvar.h"
#endif
}

#include <stdexcept>
#include <string>
#include <vector>

#include "DwmIpv4Prefix.hh"
#include "DwmTimeValue64.hh"

namespace Dwm {

  namespace Pf {

    //------------------------------------------------------------------------
    //!  Encapsulates statistics for a single pf table entry.
    //------------------------------------------------------------------------
    class TableEntryStat
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct from a pfr_astats structure.
      //----------------------------------------------------------------------
      TableEntryStat(const pfr_astats & stat)
      {
        _data = stat;
      }
      //----------------------------------------------------------------------
      //!  Returns the prefix.
      //----------------------------------------------------------------------
      Ipv4Prefix Prefix() const
      {
        if (_data.pfras_a.pfra_af == AF_INET) {
          return Ipv4Prefix(_data.pfras_a.pfra_ip4addr.s_addr,
                            _data.pfras_a.pfra_net);
        }
        else {
          throw std::domain_error("Table entry is not IPv4");
        }
      }
      //----------------------------------------------------------------------
      //!  Returns the input packets blocked counter.
      //----------------------------------------------------------------------
      uint64_t InPktsBlocked() const
      {
        return _data.pfras_packets[PFR_DIR_IN][PFR_OP_BLOCK];
      }
      //----------------------------------------------------------------------
      //!  Returns the input packets passed counter.
      //----------------------------------------------------------------------
      uint64_t InPktsPassed() const
      {
        return _data.pfras_packets[PFR_DIR_IN][PFR_OP_PASS];
      }
      //----------------------------------------------------------------------
      //!  Returns the output packets blocked counter.
      //----------------------------------------------------------------------
      uint64_t OutPktsBlocked() const
      {
        return _data.pfras_packets[PFR_DIR_OUT][PFR_OP_BLOCK];
      }
      //----------------------------------------------------------------------
      //!  Returns the output packets passed counter.
      //----------------------------------------------------------------------
      uint64_t OutPktsPassed() const
      {
        return _data.pfras_packets[PFR_DIR_OUT][PFR_OP_PASS];
      }
      //----------------------------------------------------------------------
      //!  Returns the input bytes blocked counter.
      //----------------------------------------------------------------------
      uint64_t InBytesBlocked() const
      {
        return _data.pfras_bytes[PFR_DIR_IN][PFR_OP_BLOCK];
      }
      //----------------------------------------------------------------------
      //!  Returns the input bytes passed counter.
      //----------------------------------------------------------------------
      uint64_t InBytesPassed() const
      {
        return _data.pfras_bytes[PFR_DIR_IN][PFR_OP_PASS];
      }
      //----------------------------------------------------------------------
      //!  Returns the output bytes blocked counter.
      //----------------------------------------------------------------------
      uint64_t OutBytesBlocked() const
      {
        return _data.pfras_bytes[PFR_DIR_OUT][PFR_OP_BLOCK];
      }
      //----------------------------------------------------------------------
      //!  Returns the output bytes passed counter.
      //----------------------------------------------------------------------
      uint64_t OutBytesPassed() const
      {
        return _data.pfras_bytes[PFR_DIR_OUT][PFR_OP_PASS];
      }
      //----------------------------------------------------------------------
      //!  Returns the time the entry's counters were last cleared.
      //----------------------------------------------------------------------
      TimeValue64 LastZeroed() const
      {
        return TimeValue64(_data.pfras_tzero, 0);
      }
      
    private:
      struct pfr_astats  _data;
    };

    //------------------------------------------------------------------------
    //!  Pre-declare the Device class to avoid header file inclusion loops.
    //------------------------------------------------------------------------
    class Device;
    
    //------------------------------------------------------------------------
    //!  Encapsulates basic operations for a pf address table.
    //------------------------------------------------------------------------
    class Table
    {
    public:
      Table(const Device & dev, const std::string & anchor,
            const std::string & name,
            uint32_t flags = PFR_TFLAG_PERSIST, uint8_t fback = 0);
      Table(const Device & dev, const struct pfr_table & pfrTable);

      //----------------------------------------------------------------------
      //!  Copy constructor.
      //----------------------------------------------------------------------
      Table(const Table & table);
      
      //----------------------------------------------------------------------
      //!  Returns the table's anchor.
      //----------------------------------------------------------------------
      const std::string & Anchor() const;

      //----------------------------------------------------------------------
      //!  Returns the table's name.
      //----------------------------------------------------------------------
      const std::string & Name() const;

      //----------------------------------------------------------------------
      //!  Adds the given @c prefix to the table.  Returns true on success,
      //!  false on failure.
      //----------------------------------------------------------------------
      bool Add(const Ipv4Prefix & prefix) const;

      //----------------------------------------------------------------------
      //!  Adds the given address to the table.  Returns true on success,
      //!  false on failure.
      //----------------------------------------------------------------------
      bool Add(const Ipv4Address & address) const;

      //----------------------------------------------------------------------
      //!  Removes the given @c prefix from the table.  Returns true on
      //!  success, false on failure.
      //----------------------------------------------------------------------
      bool Remove(const Ipv4Prefix & prefix) const;

      //----------------------------------------------------------------------
      //!  Removes the given @c address from the table.  Returns true on
      //!  success, false on failure.
      //----------------------------------------------------------------------
      bool Remove(const Ipv4Address & address) const;

      //----------------------------------------------------------------------
      //!  Returns true if the table contains the given @c prefix.  Note that
      //!  if a wider prefix that covers the given @c prefix is in the table,
      //!  this will return true.  If the prefix is contained or covered,
      //!  @c result will be filled with the containing prefix.
      //----------------------------------------------------------------------
      bool Contains(const Ipv4Prefix & prefix, Ipv4Prefix & result) const;

      //----------------------------------------------------------------------
      //!  Returns true if the table contains the given @c address.  Note
      //!  that this will return true if a prefix covering the given
      //!  @c address is in the table.
      //----------------------------------------------------------------------
      bool Contains(const Ipv4Address & address, Ipv4Prefix & result) const;

      //----------------------------------------------------------------------
      //!  Gets all entries in the table.
      //----------------------------------------------------------------------
      std::vector<Ipv4Prefix> GetEntries() const;

      //----------------------------------------------------------------------
      //!  Gets all statistics in the table.  Note that the counters will
      //!  all be zero unless counters are enabled for the table.
      //----------------------------------------------------------------------
      std::vector<TableEntryStat> GetStats() const;

      //----------------------------------------------------------------------
      //!  Returns true if the table is persistent.
      //----------------------------------------------------------------------
      bool Persist() const;
      
      //----------------------------------------------------------------------
      //!  Sets the table's persistence to @c persist.  Returns true on
      //!  success, false on failure.
      //----------------------------------------------------------------------
      bool Persist(bool persist) const;

      //----------------------------------------------------------------------
      //!  Returns true if the table is constant.
      //----------------------------------------------------------------------
      bool Constant() const;

      //----------------------------------------------------------------------
      //!  Sets the table's constance to @c constant.  Returns true on
      //!  success, false on failure.
      //----------------------------------------------------------------------
      bool Constant(bool constant) const;

#ifndef __APPLE__
      //----------------------------------------------------------------------
      //!  Returns true if the table's address counters are enabled.
      //----------------------------------------------------------------------
      bool CountersEnabled() const;

      //----------------------------------------------------------------------
      //!  Enables or disables the table's address counters.  Returns true
      //!  on success, false on failure.
      //----------------------------------------------------------------------
      bool CountersEnabled(bool enabled) const;
#endif
      
      //----------------------------------------------------------------------
      //!  A helper class that just wraps a pfioc_table with memory
      //!  management.
      //----------------------------------------------------------------------
      class IOCTable
      {
      public:
        //--------------------------------------------------------------------
        //!  Construct for the given @c table.
        //--------------------------------------------------------------------
        IOCTable(const Table & table);
        //--------------------------------------------------------------------
        //!  Default constructor.
        //--------------------------------------------------------------------
        IOCTable();
        //--------------------------------------------------------------------
        //!  Copy constructor.  Does a deep copy.
        //--------------------------------------------------------------------
        IOCTable(const Table::IOCTable & iocTable);
        //--------------------------------------------------------------------
        //!  operator =
        //--------------------------------------------------------------------
        IOCTable & operator = (const Table::IOCTable & iocTable);
        //--------------------------------------------------------------------
        //!  Destructor
        //--------------------------------------------------------------------
        ~IOCTable();
        //--------------------------------------------------------------------
        //!  A convenience function that allocated space in the underlying
        //!  buffer and stores the given @c prefix in it.
        //--------------------------------------------------------------------
        bool AddPrefix(const Ipv4Prefix & prefix);
        //--------------------------------------------------------------------
        //!  A convenience function that allocated space in the underlying
        //!  buffer and stores the given @c prefixes in it.
        //--------------------------------------------------------------------
        bool AddPrefixes(const std::vector<Ipv4Prefix> & prefixes);
        //--------------------------------------------------------------------
        //!  Clears the table.
        //--------------------------------------------------------------------
        void Clear();
        //--------------------------------------------------------------------
        //!  Clears the uderlying buffer.
        //--------------------------------------------------------------------
        void ClearBuffer();
        //--------------------------------------------------------------------
        //!  Allocates space in the underlying buffer for @c numAddrs
        //!  pfr_addr structures.  Returns true on success, false on failure.
        //--------------------------------------------------------------------
        bool AllocPfrAddrs(int numAddrs);
        //--------------------------------------------------------------------
        //!  Allocates space in the underlying buffer for @c numTables
        //!  pfr_table structures.  Returns true on success, false on failure.
        //--------------------------------------------------------------------
        bool AllocPfrTables(int numTables);
        //--------------------------------------------------------------------
        //!  Allocates space in the underlying buffer for @c numAddrs
        //!  pfr_astats structures.  Returns true on success, false on failure.
        //--------------------------------------------------------------------
        bool AllocPfrAstats(int numAddrs);
        //--------------------------------------------------------------------
        //!  Returns the address of our encapsulated pfioc_table structure.
        //--------------------------------------------------------------------
        pfioc_table *pfioc()
        {
          return &_tbl;
        }
        
      private:
        struct pfioc_table  _tbl;
      };

    private:
      const Device  &_dev;
      std::string    _anchor;
      std::string    _name;
      uint32_t       _flags;
      uint8_t        _fback;

      uint32_t GetFlags() const;
      bool SetFlags(uint32_t flags) const;
    };

  }  // namespace Pf

}  // namespace Dwm

#endif  // _DWMPFTABLE_HH_
