//===========================================================================
// @(#) $DwmPath: dwm/libDwmAuth/tags/libDwmAuth-0.3.9/src/DwmAuthSymCrypto.cc 9844 $
// @(#) $Id: DwmAuthSymCrypto.cc 9844 2017-12-16 23:58:55Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmAuthSymCrypto.cc
//!  \brief Symmetric cryptography utility implementations
//---------------------------------------------------------------------------

#include <iostream>

#include <cryptopp/aes.h>
#include <cryptopp/cryptlib.h>
#include <cryptopp/filters.h>
#include <cryptopp/gcm.h>
#include <cryptopp/osrng.h>

#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmAuthSymCrypto.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmAuth/tags/libDwmAuth-0.3.9/src/DwmAuthSymCrypto.cc 9844 $");

using namespace std;

namespace Dwm {

  namespace Auth {

    namespace SymCrypto {
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      string Encrypt(const string & keystr,
                     const string & ivstr,
                     const string & plaintext)
      {
        using namespace CryptoPP;
        using AEF = AuthenticatedEncryptionFilter;
        
        uint8_t  key[AES::DEFAULT_KEYLENGTH];
        uint8_t  iv[AES::BLOCKSIZE];
        for (uint8_t i = 0; i < (sizeof(key)/sizeof(key[0])); ++i) {
          key[i] = keystr[i % keystr.size()];
        }
        for (uint8_t i = 0; i < (sizeof(iv)/sizeof(iv[0])); ++i) {
          iv[i] = ivstr[i % ivstr.size()];
        }
        string  ciphertext;
        try {
          GCM<AES>::Encryption  enc;
          const int  TAGSIZE = 16;
          enc.SetKeyWithIV(key, sizeof(key), iv, sizeof(iv));
          memset(key, 0, sizeof(key));
          memset(iv, 0, sizeof(iv));
          StringSource(plaintext, true,
                       new AEF(enc, new StringSink(ciphertext),
                               false, TAGSIZE));
        }
        catch (CryptoPP::Exception & e) {
          Syslog(LOG_ERR, "Exception: %s", e.what());
        }
        return ciphertext;
      }

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      string Encrypt(const string & keystr,
                     const string & ivstr,
                     const uint8_t *plaintext,
                     size_t plaintextLen)
      {
        using namespace CryptoPP;
        using AEF = AuthenticatedEncryptionFilter;
        
        uint8_t  key[AES::DEFAULT_KEYLENGTH];
        uint8_t  iv[AES::BLOCKSIZE];
        for (uint8_t i = 0; i < (sizeof(key)/sizeof(key[0])); ++i) {
          key[i] = keystr[i % keystr.size()];
        }
        for (uint8_t i = 0; i < (sizeof(iv)/sizeof(iv[0])); ++i) {
          iv[i] = ivstr[i % ivstr.size()];
        }
        string  ciphertext;
        try {
          GCM<AES>::Encryption  enc;
          const int  TAGSIZE = 16;
          enc.SetKeyWithIV(key, sizeof(key), iv, sizeof(iv));
          memset(key, 0, sizeof(key));
          memset(iv, 0, sizeof(iv));
          StringSource(plaintext, plaintextLen, true,
                       new AEF(enc, new StringSink(ciphertext),
                               false, TAGSIZE));
        }
        catch (CryptoPP::Exception & e) {
          Syslog(LOG_ERR, "Exception: %s", e.what());
        }
        return ciphertext;
      }
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      string Decrypt(const string & keystr,
                     const string & ivstr,
                     const string & ciphertext)
      {
        using namespace CryptoPP;
        using ADF = AuthenticatedDecryptionFilter;
        
        string   plaintext;
        uint8_t  key[AES::DEFAULT_KEYLENGTH];
        uint8_t  iv[AES::BLOCKSIZE];
        for (uint8_t i = 0; i < (sizeof(key)/sizeof(key[0])); ++i) {
          key[i] = keystr[i % keystr.size()];
        }
        for (uint8_t i = 0; i < (sizeof(iv)/sizeof(iv[0])); ++i) {
          iv[i] = ivstr[i % ivstr.size()];
        }
        try {
          GCM<AES>::Decryption  d;
          const int  TAGSIZE = 16;
          d.SetKeyWithIV(key, sizeof(key), iv, sizeof(iv));
          ADF  adf(d, new StringSink(plaintext), ADF::DEFAULT_FLAGS, TAGSIZE);
          memset(key, 0, sizeof(key));
          memset(iv, 0, sizeof(iv));
          StringSource ss(ciphertext, true, new Redirector(adf));
          if (! adf.GetLastResult()) {
            Syslog(LOG_ERR, "Decryption failed");
            plaintext.clear();
          }
        }
        catch (CryptoPP::Exception & e) {
          Syslog(LOG_ERR, "Exception: %s", e.what());
        }
        
        return plaintext;
      }

    }  // namespace SymCrypto

  }  // namespace Auth

}  // namespace Dwm
