//===========================================================================
// @(#) $DwmPath: dwm/libDwmAuth/tags/libDwmAuth-0.3.9/src/DwmAuthEd25519Keys.cc 11162 $
// @(#) $Id: DwmAuthEd25519Keys.cc 11162 2020-09-08 06:46:41Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018, 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmAuthEd25519Keys.cc
//!  \author Daniel W. McRobb
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

extern "C" {
  #include <unistd.h>
}

#include <fstream>
#include <regex>
#include <cryptopp/base64.h>

#include "DwmSysLogger.hh"
#include "DwmDescriptorIO.hh"
#include "DwmFileIO.hh"
#include "DwmStreamIO.hh"
#include "DwmAuthEd25519Keys.hh"

using namespace std;

namespace Dwm {

  namespace Auth {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Ed25519::PublicKey::IsValid() const
    {
      CryptoPP::AutoSeededRandomPool   rng;
      return Validate(rng, 2);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::istream & Ed25519::PublicKey::Read(std::istream & is)
    {
      if (is) {
        uint16_t  len;
        if (StreamIO::Read(is, len)) {
          char  *buf = (char *)calloc(1, len);
          if (buf) {
            if (is.read(buf, len)) {
              string  s(buf, len);
              FromString(s);
            }
            else {
              Syslog(LOG_ERR, "Failed to read public key");
            }
            memset(buf, 0, len);
            free(buf);
          }
          else {
            Syslog(LOG_ERR, "Failed to allocate %hu bytes", len);
          }
        }
        else {
          Syslog(LOG_ERR, "Failed to read public key length");
        }
      }
      else {
        Syslog(LOG_ERR, "Invalid istream");
      }
      return is;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::ostream & Ed25519::PublicKey::Write(std::ostream & os) const
    {
      if (os) {
        string  s = ToString();
        uint16_t  len = s.size();
        if (StreamIO::Write(os, len)) {
          os.write(s.data(), len);
        }
      }
      return os;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t Ed25519::PublicKey::Read(FILE *f)
    {
      size_t  rc = 0;
      if (f) {
        uint16_t  len;
        if (FileIO::Read(f, len)) {
          char  *buf = (char *)calloc(1, len);
          if (buf) {
            if (fread(buf, len, 1, f) == 1) {
              string  s(buf, len);
              FromString(s);
              rc = 1;
            }
            else {
              Syslog(LOG_ERR, "Failed to read public key");
            }
            memset(buf, 0, len);
            free(buf);
          }
          else {
            Syslog(LOG_ERR, "Failed to allocate %hu bytes", len);
          }
        }
        else {
          Syslog(LOG_ERR, "Failed to read public key length");
        }
      }
      return rc;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t Ed25519::PublicKey::Write(FILE *f) const
    {
      size_t  rc = 0;
      if (f) {
        string  s = ToString();
        uint16_t  len = s.size();
        if (FileIO::Write(f, len)) {
          if (len > 0) {
            rc = fwrite(s.data(), len, 1, f);
          }
          else {
            rc = 1;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t Ed25519::PublicKey::Read(int fd)
    {
      ssize_t  rc = -1;
      if (0 <= fd) {
        uint16_t  len;
        ssize_t  bytesRead = DescriptorIO::Read(fd, len);
        if (0 < bytesRead) {
          rc = bytesRead;
          if (len) {
            char  *buf = (char *)calloc(1, len);
            if (buf) {
              if (DescriptorIO::Read(fd, buf, len) == len) {
                string  s(buf, len);
                FromString(s);
                rc += s.size();
              }
              else {
                Syslog(LOG_ERR, "Failed to read public key");
              }
              memset(buf, 0, len);
              free(buf);
            }
            else {
              Syslog(LOG_ERR, "Failed to allocate %hu bytes", len);
            }
          }
        }
        else {
          Syslog(LOG_ERR, "Failed to read public key length");
        }
      }
      return rc;
    }
        
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t Ed25519::PublicKey::Write(int fd) const
    {
      ssize_t  rc = -1;
      if (0 <= fd) {
        string  s = ToString();
        uint16_t  len = s.size();
        if (Dwm::DescriptorIO::Write(fd, len)) {
          rc = sizeof(len);
          if (len > 0) {
            ssize_t  bytesWritten = DescriptorIO::Write(fd, s.data(), len);
            if (0 < bytesWritten) {
              rc += bytesWritten;
            }
            else {
              Syslog(LOG_ERR, "Failed to write public key");
              rc = -1;
            }
          }
        }
        else {
          Syslog(LOG_ERR, "Failed to write public key length");
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t Ed25519::PublicKey::StreamedLength() const
    {
      return (sizeof(uint16_t) + ToString().size());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::string Ed25519::PublicKey::ToString() const
    {
      string  s;
      CryptoPP::StringSink  ssink(s);
      // this->DEREncode(ssink);
      this->Save(ssink);
      return s;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::string Ed25519::PublicKey::ToBase64String() const
    {
      string  s = ToString();
      string  sb;
      CryptoPP::StringSource
        ss((uint8_t *)(s.c_str()), s.size(), true,
           new CryptoPP::Base64Encoder(new CryptoPP::StringSink(sb), false));
      return sb;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Ed25519::PublicKey::FromString(const std::string & s)
    {
      bool  rc = false;
      if (! s.empty()) {
        CryptoPP::StringSource  ssource(s, true);
        try {
          // this->BERDecode(ssource);
          this->Load(ssource);
          rc = true;
        }
        catch (...) {
          Syslog(LOG_ERR, "Failed to decode public key '%s'", s.c_str());
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Ed25519::PublicKey::FromBase64String(const std::string & s)
    {
      bool  rc = false;
      if (! s.empty()) {
        string  decoded;
        CryptoPP::StringSource
          //          ss2((const uint8_t *)(s.c_str()), s.size(), true,
          ss2(s, true,
              new CryptoPP::Base64Decoder(new CryptoPP::StringSink(decoded)));
        rc = FromString(decoded);
        if (! rc) {
          Syslog(LOG_ERR, "Failed to decode public key '%s'", s.c_str());
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Ed25519::PublicKey::operator == (const PublicKey & pk) const
    {
      return (ToString() == pk.ToString());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Ed25519::PrivateKey::IsValid() const
    {
      CryptoPP::AutoSeededRandomPool   rng;
      return Validate(rng, 2);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::istream & Ed25519::PrivateKey::Read(std::istream & is)
    {
      if (is) {
        uint16_t  len;
        if (StreamIO::Read(is, len)) {
          char  *buf = (char *)calloc(1, len);
          if (buf) {
            if (is.read(buf, len)) {
              string  s(buf, len);
              FromString(s);
            }
            else {
              Syslog(LOG_ERR, "Failed to read private key");
            }
            memset(buf, 0, len);
            free(buf);
          }
          else {
            Syslog(LOG_ERR, "Failed to allocated %hu bytes", len);
          }
        }
        else {
          Syslog(LOG_ERR, "Failed to read private key length");
        }
      }
      else {
        Syslog(LOG_ERR, "Invalid istream");
      }
      return is;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::ostream & Ed25519::PrivateKey::Write(std::ostream & os) const
    {
      if (os) {
        string  s = ToString();
        uint16_t  len = s.size();
        if (StreamIO::Write(os, len)) {
          os.write(s.data(), len);
        }
      }
      return os;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t Ed25519::PrivateKey::Read(FILE *f)
    {
      size_t  rc = 0;
      if (f) {
        uint16_t  len;
        if (FileIO::Read(f, len)) {
          char  *buf = (char *)calloc(1, len);
          if (buf) {
            if (fread(buf, len, 1, f) == 1) {
              string  s(buf, len);
              FromString(s);
              rc = 1;
            }
            else {
              Syslog(LOG_ERR, "Failed to read private key");
            }
            memset(buf, 0, len);
            free(buf);
          }
          else {
            Syslog(LOG_ERR, "Failed to allocate %hu bytes", len);
          }
        }
        else {
          Syslog(LOG_ERR, "Failed to read private key length");
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t Ed25519::PrivateKey::Write(FILE *f) const
    {
      size_t  rc = 0;
      if (f) {
        string  s = ToString();
        uint16_t  len = s.size();
        if (Dwm::FileIO::Write(f, len)) {
          if (len > 0) {
            rc = fwrite(s.data(), len, 1, f);
          }
          else {
            rc = 1;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t Ed25519::PrivateKey::Read(int fd)
    {
      ssize_t  rc = -1;
      if (0 <= fd) {
        uint16_t  len;
        ssize_t  bytesRead = DescriptorIO::Read(fd, len);
        if (0 < bytesRead) {
          rc = bytesRead;
          if (len) {
            char  *buf = (char *)calloc(1, len);
            if (buf) {
              if (DescriptorIO::Read(fd, buf, len) == len) {
                string  s(buf, len);
                FromString(s);
                rc += s.size();
              }
              else {
                Syslog(LOG_ERR, "Failed to read private key");
              }
              memset(buf, 0, len);
              free(buf);
            }
            else {
              Syslog(LOG_ERR, "Failed to allocate %hu bytes", len);
            }
          }
        }
        else {
          Syslog(LOG_ERR, "Failed to read private key length");
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t Ed25519::PrivateKey::Write(int fd) const
    {
      ssize_t  rc = -1;
      if (0 <= fd) {
        string  s = ToString();
        uint16_t  len = s.size();
        if (Dwm::DescriptorIO::Write(fd, len)) {
          rc = sizeof(len);
          if (len > 0) {
            ssize_t  bytesWritten = write(fd, s.data(), len);
            if (0 < bytesWritten) {
              rc += bytesWritten;
            }
            else {
              Syslog(LOG_ERR, "Failed to write private key");
              rc = -1;
            }
          }
        }
        else {
          Syslog(LOG_ERR, "Failed to write private key length");
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t Ed25519::PrivateKey::StreamedLength() const
    {
      return (sizeof(uint16_t) + ToString().size());
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::string Ed25519::PrivateKey::ToString() const
    {
      string  s;
      CryptoPP::StringSink  ssink(s);
      this->DEREncode(ssink);
      return s;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::string Ed25519::PrivateKey::ToBase64String() const
    {
      string  s = ToString();
      string  sb;
      CryptoPP::StringSource
        ss((uint8_t *)(s.c_str()), s.size(), true,
           new CryptoPP::Base64Encoder(new CryptoPP::StringSink(sb), false));
      return sb;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Ed25519::PrivateKey::FromString(const std::string & s)
    {
      CryptoPP::StringSource  ssource(s, true);
      this->BERDecode(ssource);
      return true;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Ed25519::PrivateKey::FromBase64String(const std::string & s)
    {
      bool  rc = false;
      if (! s.empty()) {
        string  decoded;
        CryptoPP::StringSource
          ss2((const uint8_t *)(s.c_str()), s.size(), true,
              new CryptoPP::Base64Decoder(new CryptoPP::StringSink(decoded)));
        rc = FromString(decoded);
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Ed25519::PrivateKey::operator == (const PrivateKey & kp) const
    {
      return (ToString() == kp.ToString());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Ed25519::KeyPair::Load(const string & privKeyPath, string & name)
    {
      bool  rc = false;
      name.clear();
      ifstream  privis(privKeyPath.c_str(), ios_base::in | ios_base::binary);
      if (privis) {
        if (first.Read(privis)) {
          string  pubKeyPath(privKeyPath + ".pub");
          ifstream  pubis(pubKeyPath.c_str(), ios_base::in | ios_base::binary);
          if (pubis) {
            string  pub64;
            string  s;
            if (getline(pubis, s, '\n') && (!s.empty())) {
              regex   rgx("([^ \\t]+)[ \\t]+([^ \\t]+)[ \\t]+([^ \\t]+)",
                          regex::ECMAScript|regex::optimize);
              smatch  sm;
              if (regex_match(s, sm, rgx)) {
                if ((sm.size() == 4) && (sm[2].str() == "dwm-ed25519")) {
                  if (second.FromBase64String(sm[3].str())) {
                    if (second.IsValid()) {
                      name = sm[1].str();
                      rc = true;
                    }
                    else {
                      Syslog(LOG_ERR, "Invalid key %s", sm[1].str().c_str());
                    }
                  }
                  else {
                    Syslog(LOG_ERR, "FromBase64String(%s) failed",
                           sm[2].str().c_str());
                  }
                }
                else {
                  Syslog(LOG_ERR, "Wrong number of fields in public key"
                         " file: %d != 4", sm.size());
                }
              }
              else {
                Syslog(LOG_ERR, "Unrecognized line in public key file");
              }
            }
            else {
              Syslog(LOG_ERR, "getline() failed");
            }
            pubis.close();
          }
          else {
            Syslog(LOG_ERR, "Failed to open public key file '%s'",
                   pubKeyPath.c_str());
          }
        }
        else {
          Syslog(LOG_ERR, "Failed to read private key from '%s'",
                 privKeyPath.c_str());
        }
        privis.close();
      }
      else {
        Syslog(LOG_ERR, "Failed to open private key file '%s'",
               privKeyPath.c_str());
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Ed25519::KeyPair Ed25519::CreateKeyPair()
    {
      using ed25519PrivateKey = CryptoPP::ed25519PrivateKey;
      using ed25519PublicKey = CryptoPP::ed25519PublicKey;
      
      CryptoPP::AutoSeededRandomPool   rng;
      CryptoPP::ed25519::Signer        signer;
      signer.AccessPrivateKey().GenerateRandom(rng);
      const ed25519PrivateKey & privKey =
        dynamic_cast<const ed25519PrivateKey &>(signer.GetPrivateKey());
      CryptoPP::ed25519::Verifier      verifier(signer);
      const ed25519PublicKey  & pubKey =
        dynamic_cast<const ed25519PublicKey &>(verifier.GetPublicKey());
      return KeyPair(privKey, pubKey);
    }
      
    
  }  // namespace Auth

}  // namespace Dwm
