//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmAuthPublicKeysFile.hh 10617 2020-05-02 21:18:04Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017, 2018, 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmAuthPublicKeysFile.hh
//!  \brief Dwm::Auth::PublicKeysFile class definition
//---------------------------------------------------------------------------

#ifndef _DWMAUTHPUBLICKEYSFILE_HH_
#define _DWMAUTHPUBLICKEYSFILE_HH_

#include <map>
#include <string>

#include "DwmAuthEd25519Keys.hh"

namespace Dwm {

  namespace Auth {

    //------------------------------------------------------------------------
    //!  Encapsulates a file containing public keys in a format similar to
    //!  (but not the same as) an ssh authorized_keys file.
    //------------------------------------------------------------------------
    class PublicKeysFile
    {
    public:
      typedef std::map<std::string,Ed25519::PublicKey>  Map;
      
      //----------------------------------------------------------------------
      //!  Constructor
      //----------------------------------------------------------------------
      PublicKeysFile();

      //----------------------------------------------------------------------
      //!  Copy constructor
      //----------------------------------------------------------------------
      PublicKeysFile(const PublicKeysFile & pkf) = default;

      //----------------------------------------------------------------------
      //!  Loads the public keys from the file at the given path @c fileName.
      //!  Returns true on success, false on failure.
      //----------------------------------------------------------------------
      bool Load(const std::string & fileName);

      //----------------------------------------------------------------------
      //!  Adds a key with the given name to the contained map of keys.
      //!  Returns true on success, false on failure.
      //!  Does NOT save to disk; you must call Save() to save to disk.
      //----------------------------------------------------------------------
      bool AddKey(const std::string & name, const Ed25519::PublicKey & key);

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      bool AddKey(const std::string & name, const std::string & base64str);

      //----------------------------------------------------------------------
      //!  Removes the key with the given @c name from the contained map of
      //!  keys.  Returns true on success, false on failure.  Does NOT save
      //!  to disk; you must call Save() to save to disk.
      //----------------------------------------------------------------------
      bool RemoveKey(const std::string & name);
      
      //----------------------------------------------------------------------
      //!  Saves the contained map of keys to the file with the given
      //!  @c fileName with permissions @perms.  Returns true on success,
      //!  false on failure.
      //----------------------------------------------------------------------
      bool Save(const std::string & fileName, int perms = 0644) const;
      
      //----------------------------------------------------------------------
      //!  Searches the keys for the one with the given @c name.  Returns
      //!  true and fills @c key on success, returns false on failure.
      //----------------------------------------------------------------------
      bool Find(const std::string & name,
                Ed25519::PublicKey & key) const;

      //----------------------------------------------------------------------
      //!  Returns a const reference to the contained public keys.
      //----------------------------------------------------------------------
      const Map & Keys() const;
      
    private:
      std::map<std::string,Ed25519::PublicKey>  _keys;
    };
    
  }  // namespace Auth

}  // namespace Dwm

#endif  // _DWMAUTHPUBLICKEYSFILE_HH_
