//===========================================================================
// @(#) $DwmPath: dwm/libDwmAuth/tags/libDwmAuth-0.3.7/src/DwmAuthPKCrypto.cc 11162 $
// @(#) $Id: DwmAuthPKCrypto.cc 11162 2020-09-08 06:46:41Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmAuthPKCrypto.cc
//!  \brief Public key crypto implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <unistd.h>
}

#include <cassert>
#include <cstdlib>
#include <fstream>
#include <regex>
#include <cryptopp/base64.h>

#include "DwmDescriptorIO.hh"
#include "DwmFileIO.hh"
#include "DwmStreamIO.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmAuthPKCrypto.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmAuth/tags/libDwmAuth-0.3.7/src/DwmAuthPKCrypto.cc 11162 $");

using namespace std;
using CryptoPP::Base64Decoder;
using CryptoPP::StringSink;
using CryptoPP::StringSource;

namespace Dwm {

  namespace Auth {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PKCrypto::PublicKey::IsValid() const
    {
      CryptoPP::AutoSeededRandomPool   rng;
      return Validate(rng, 2);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::istream & PKCrypto::PublicKey::Read(std::istream & is)
    {
      if (is) {
        uint16_t  len;
        if (Dwm::StreamIO::Read(is, len)) {
          char  *buf = (char *)calloc(1, len);
          if (buf) {
            if (is.read(buf, len)) {
              string  s(buf, len);
              FromString(s);
            }
            memset(buf, 0, len);
            free(buf);
          }
        }
      }
      return is;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t PKCrypto::PublicKey::Read(int fd)
    {
      ssize_t  rc = -1;
      if (fd >= 0) {
        uint16_t  len;
        ssize_t  bytesRead = Dwm::DescriptorIO::Read(fd, len);
        if (bytesRead == sizeof(len)) {
          rc = bytesRead;
          char  *buf = (char *)calloc(1, len);
          if (buf) {
            bytesRead = read(fd, buf, len);
            if (bytesRead == len) {
              rc += bytesRead;
              string  s(buf, len);
              FromString(s);
            }
            free(buf);
          }
          else {
            rc = -1;
          }
        }
        else {
          rc = -1;
        }
      }
      return rc;
    }
        
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t PKCrypto::PublicKey::Read(FILE * f)
    {
      size_t  rc = 0;
      if (f) {
        uint16_t  len;
        if (Dwm::FileIO::Read(f, len) == sizeof(len)) {
          if (len > 0) {
            rc = sizeof(len);
            char  *buf = (char *)calloc(1, len);
            if (buf) {
              if (fread((void *)buf, len, 1, f) == 1) {
                string  s(buf, len);
                FromString(s);
                rc += len;
              }
              else {
                rc = 0;
              }
              memset(buf, 0, len);
              free(buf);
            }
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::ostream & PKCrypto::PublicKey::Write(std::ostream & os) const
    {
      if (os) {
        string  s = ToString();
        uint16_t  len = s.size();
        if (Dwm::StreamIO::Write(os, len)) {
          os.write(s.data(), len);
        }
      }
      return os;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t PKCrypto::PublicKey::Write(int fd) const
    {
      ssize_t  rc = -1;
      if (fd >= 0) {
        string  s = ToString();
        uint16_t  len = s.size();
        if (Dwm::DescriptorIO::Write(fd, len) == sizeof(len)) {
          rc = sizeof(len);
          ssize_t  bytesWritten = write(fd, s.data(), len);
          if (bytesWritten == len) {
            rc += len;
          }
          else {
            rc = -1;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t PKCrypto::PublicKey::Write(FILE *f) const
    {
      size_t  rc = 0;
      if (f) {
        string s = ToString();
        uint16_t  len = s.size();
        if (Dwm::FileIO::Write(f, len) == sizeof(len)) {
          rc = sizeof(len);
          if (fwrite((void *)s.data(), len, 1, f) == 1) {
            rc += len;
          }
          else {
            rc = 0;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t PKCrypto::PublicKey::StreamedLength() const
    {
      string  s = ToString();
      return s.size() + sizeof(uint16_t);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::string PKCrypto::PublicKey::ToString() const
    {
      string  s;
      CryptoPP::StringSink  ssink(s);
      this->DEREncode(ssink);
      return s;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::string PKCrypto::PublicKey::ToBase64String() const
    {
      string  s = ToString();
      string  sb;
      CryptoPP::StringSource
        ss((uint8_t *)(s.c_str()), s.size(), true,
           new CryptoPP::Base64Encoder(new CryptoPP::StringSink(sb), false));
      return sb;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PKCrypto::PublicKey::FromString(const std::string & s)
    {
      bool  rc = false;
      if (! s.empty()) {
        CryptoPP::StringSource  ssource(s, true);
        try {
          this->BERDecode(ssource);
          rc = true;
        }
        catch (...) {
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PKCrypto::PublicKey::FromBase64String(const std::string & s)
    {
      bool  rc = false;
      if (! s.empty()) {
        string  decoded;
        CryptoPP::StringSource
          ss2((const uint8_t *)(s.c_str()), s.size(), true,
              new Base64Decoder(new StringSink(decoded)));
        rc = FromString(decoded);
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PKCrypto::PublicKey::operator == (const PublicKey & pk) const
    {
      return (ToString() == pk.ToString());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PKCrypto::PrivateKey::IsValid() const
    {
      CryptoPP::AutoSeededRandomPool   rng;
      return Validate(rng, 2);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::istream & PKCrypto::PrivateKey::Read(std::istream & is)
    {
      if (is) {
        uint16_t  len;
        if (Dwm::StreamIO::Read(is, len)) {
          char  *buf = (char *)calloc(1, len);
          if (buf) {
            if (is.read(buf, len)) {
              string  s(buf, len);
              FromString(s);
            }
            memset(buf, 0, len);
            free(buf);
          }
        }
      }
      return is;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t PKCrypto::PrivateKey::Read(int fd)
    {
      ssize_t  rc = -1;
      if (fd >= 0) {
        uint16_t  len;
        ssize_t  bytesRead = Dwm::DescriptorIO::Read(fd, len);
        if (bytesRead == sizeof(len)) {
          rc = bytesRead;
          char  *buf = (char *)calloc(1, len);
          if (buf) {
            bytesRead = read(fd, buf, len);
            if (bytesRead == len) {
              rc += bytesRead;
              string  s(buf, len);
              FromString(s);
            }
            free(buf);
          }
          else {
            rc = -1;
          }
        }
        else {
          rc = -1;
        }
      }
      return rc;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t PKCrypto::PrivateKey::Read(FILE * f)
    {
      size_t  rc = 0;
      if (f) {
        uint16_t  len;
        if (Dwm::FileIO::Read(f, len) == sizeof(len)) {
          if (len > 0) {
            rc = sizeof(len);
            char  *buf = (char *)calloc(1, len);
            if (buf) {
              if (fread((void *)buf, len, 1, f) == 1) {
                string  s(buf, len);
                FromString(s);
                rc += len;
              }
              else {
                rc = 0;
              }
              memset(buf, 0, len);
              free(buf);
            }
          }
        }
      }
      return rc;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::ostream & PKCrypto::PrivateKey::Write(std::ostream & os) const
    {
      if (os) {
        string  s = ToString();
        uint16_t  len = s.size();
        if (Dwm::StreamIO::Write(os, len)) {
          os.write(s.data(), len);
        }
      }
      return os;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t PKCrypto::PrivateKey::Write(int fd) const
    {
      ssize_t  rc = -1;
      if (fd >= 0) {
        string  s = ToString();
        uint16_t  len = s.size();
        if (Dwm::DescriptorIO::Write(fd, len) == sizeof(len)) {
          rc = sizeof(len);
          ssize_t  bytesWritten = write(fd, s.data(), len);
          if (bytesWritten == len) {
            rc += len;
          }
          else {
            rc = -1;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    size_t PKCrypto::PrivateKey::Write(FILE *f) const
    {
      size_t  rc = 0;
      if (f) {
        string s = ToString();
        uint16_t  len = s.size();
        if (Dwm::FileIO::Write(f, len) == sizeof(len)) {
          rc = sizeof(len);
          if (fwrite((void *)s.data(), len, 1, f) == 1) {
            rc += len;
          }
          else {
            rc = 0;
          }
        }
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint32_t PKCrypto::PrivateKey::StreamedLength() const
    {
      string  s = ToString();
      return s.size() + sizeof(uint16_t);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::string PKCrypto::PrivateKey::ToString() const
    {
      string  s;
      CryptoPP::StringSink  ssink(s);
      this->DEREncode(ssink);
      return s;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::string PKCrypto::PrivateKey::ToBase64String() const
    {
      string  s = ToString();
      string  sb;
      CryptoPP::StringSource
        ss((uint8_t *)(s.c_str()), s.size(), true,
           new CryptoPP::Base64Encoder(new CryptoPP::StringSink(sb), false));
      return sb;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PKCrypto::PrivateKey::FromString(const std::string & s)
    {
      CryptoPP::StringSource  ssource(s, true);
      this->BERDecode(ssource);
      return true;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PKCrypto::PrivateKey::FromBase64String(const std::string & s)
    {
      bool  rc = false;
      if (! s.empty()) {
        string  decoded;
        CryptoPP::StringSource
          ss2((const uint8_t *)(s.c_str()), s.size(), true,
              new Base64Decoder(new StringSink(decoded)));
        rc = FromString(decoded);
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PKCrypto::PrivateKey::operator == (const PrivateKey & kp) const
    {
      return (ToString() == kp.ToString());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PKCrypto::KeyPair::Load(const string & privKeyPath, string & name)
    {
      bool  rc = false;
      name.clear();
      ifstream  privis(privKeyPath.c_str());
      if (privis) {
        if (first.Read(privis)) {
          string  pubKeyPath(privKeyPath + ".pub");
          ifstream  pubis(pubKeyPath.c_str());
          if (pubis) {
            string  pub64;
            string  s;
            if (getline(pubis, s, '\n')) {
              regex   rgx("([^ \\t]+)[ \\t]+([^ \\t]+)",
                          regex::ECMAScript|regex::optimize);
              smatch  sm;
              if (regex_search(s, sm, rgx)) {
                if (sm.size() == 3) {
                  if (second.FromBase64String(sm[2].str())) {
                    if (second.IsValid()) {
                      name = sm[1].str();
                      rc = true;
                    }
                  }
                }
              }
            }
            pubis.close();
          }
        }
        privis.close();
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    PKCrypto::KeyPair PKCrypto::CreateKeyPair(unsigned int keySize)
    {
      CryptoPP::AutoSeededRandomPool   rng;
      CryptoPP::InvertibleRSAFunction  params;
      params.GenerateRandomWithKeySize(rng, 2048);
      return KeyPair(CryptoPP::RSA::PrivateKey(params),
                     CryptoPP::RSA::PublicKey(params));
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string
    PKCrypto::Encrypt(const PublicKey & publicKey, const string & plainstr)
    {
      CryptoPP::AutoSeededRandomPool      rng;
      CryptoPP::RSAES_OAEP_SHA_Encryptor  encryptor(publicKey);
      // Now that there is a concrete object, we can validate
      size_t  maxPlainText = encryptor.FixedMaxPlaintextLength();
      assert(0 != maxPlainText);
      if (maxPlainText > plainstr.size()) {
        maxPlainText = plainstr.size();
      }
      string  rc;
      string::size_type  idx = 0;
      
      while (plainstr.size() > idx) {
        string  cipherstr;
        try {
          CryptoPP::StringSource
            ss(plainstr.substr(idx, maxPlainText), true,
               new CryptoPP::PK_EncryptorFilter(rng, encryptor,
                                                new CryptoPP::StringSink(cipherstr)));
          rc.append(cipherstr);
          idx += maxPlainText;
        }
        catch (...) {
          Syslog(LOG_ERR, "Exception in PKCrypto::Encrypt()");
          rc.clear();
          break;
        }
      }
      return rc;
    }
      
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string
    PKCrypto::Decrypt(const PrivateKey & privateKey,
                      const string & cipherstr)
    {
      CryptoPP::AutoSeededRandomPool      rng;
      CryptoPP::RSAES_OAEP_SHA_Decryptor  decryptor(privateKey);
      assert(decryptor.FixedMaxPlaintextLength() != 0);

      size_t  maxCipherText = decryptor.FixedCiphertextLength();
      if (maxCipherText > cipherstr.size()) {
        maxCipherText = cipherstr.size();
      }

      string  rc;
      string::size_type  idx = 0;
      
      while (cipherstr.size() > idx) {
        string  plainstr;
        try {
          CryptoPP::StringSource
            ss(cipherstr.substr(idx, maxCipherText), true,
               new CryptoPP::PK_DecryptorFilter(rng, decryptor,
                                                new CryptoPP::StringSink(plainstr)));
        rc.append(plainstr);
        idx += maxCipherText;
        }
        catch (...) {
          Syslog(LOG_ERR, "Exception in PKCrypto::Decrypt()");
          rc.clear();
          break;
        }
      }
      
      return rc;
    }

  }  // namespace Auth

}  // namespace Dwm
