//===========================================================================
// @(#) $DwmPath: dwm/libDwmAuth/tags/libDwmAuth-0.3.6/src/DwmAuthUtils.cc 11162 $
// @(#) $Id: DwmAuthUtils.cc 11162 2020-09-08 06:46:41Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmAuthUtils.cc
//!  \author Daniel W. McRobb
//!  \brief Dwm::Auth::Utils class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/ioctl.h>
}

#include <thread>

#include "DwmASIO.hh"
#include "DwmDescriptorIO.hh"
#include "DwmStreamIO.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmAuthUtils.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmAuth/tags/libDwmAuth-0.3.6/src/DwmAuthUtils.cc 11162 $");

namespace Dwm {

  namespace Auth {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    int Utils::BytesReady(int fd)
    {
      int  rc = -1;
      if (fd >= 0) {
        int  bytesReady;
        if (ioctl(fd, FIONREAD, &bytesReady) != -1) {
          rc = bytesReady;
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::size_t Utils::BytesReady(boost::asio::ip::tcp::socket & sck)
    {
      boost::asio::socket_base::bytes_readable  cmd(true);
      sck.io_control(cmd);
      return cmd.get();
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Utils::WaitUntilBytesReady(int fd, uint32_t numBytes,
                                    TimePoint endTime)
    {
      bool  rc = false;
      if (fd >= 0) {
        int   bytesReady = 0;
        while ((bytesReady = BytesReady(fd)) < numBytes) {
          if (Clock::now() > endTime) {
            break;
          }
          else {
            std::this_thread::sleep_for(std::chrono::milliseconds(130));
          }
        }
        rc = (bytesReady >= numBytes);
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Utils::WaitUntilBytesReady(boost::asio::ip::tcp::socket & sck,
                                    uint32_t numBytes, TimePoint endTime)
    {
      bool  rc = false;
      if (sck.is_open()) {
        std::size_t  bytesReady = 0;
        while ((bytesReady = BytesReady(sck)) < numBytes) {
          if (Clock::now() > endTime) {
            break;
          }
          else {
            std::this_thread::sleep_for(std::chrono::milliseconds(130));
          }
        }
        rc = (bytesReady >= numBytes);
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Utils::WaitForBytesReady(int fd, uint32_t numBytes,
                                  std::chrono::milliseconds timeout)
    {
      return WaitUntilBytesReady(fd, numBytes, Clock::now() + timeout);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool Utils::WaitForBytesReady(boost::asio::ip::tcp::socket & sck,
                                  uint32_t numBytes,
                                  std::chrono::milliseconds timeout)
    {
      return WaitUntilBytesReady(sck, numBytes, Clock::now() + timeout);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t Utils::ReadLengthRestrictedString(int fd, std::string & s,
                                              uint32_t maxLen) 
    {
      s.clear();
      
      ssize_t  rc = -1;
      if (fd >= 0) {
        uint32_t  len;
        if (DescriptorIO::Read(fd, len) == sizeof(len)) {
          if (len > 0) {
            if (len <= maxLen) {
              try {
                s.resize(len);
                if (DescriptorIO::Read(fd, s.data(), len)) {
                  rc = sizeof(len) + len;
                }
                else {
                  Syslog(LOG_ERR, "Failed to read string data");
                }
              }
              catch (...) {
                Syslog(LOG_ERR, "Exception reading string data");
              }
            }
            else {
              Syslog(LOG_ERR, "String length %u exceeds max of %u",
                     len, maxLen);
            }
          }
          else {
            rc = sizeof(len);
          }
        }
        else {
          Syslog(LOG_ERR, "Failed to read string length");
        }
      }
      else {
        Syslog(LOG_ERR, "Invalid descriptor %d", fd);
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t
    Utils::ReadLengthRestrictedString(boost::asio::ip::tcp::socket & sck,
                                      std::string & s, uint32_t maxLen) 
    {
      using boost::asio::buffer, boost::asio::read;
      
      s.clear();
      ssize_t  rc = -1;
      if (sck.is_open()) {
        uint32_t  len;
        if (ASIO::Read(sck, len)) {
          if (len > 0) {
            if (len <= maxLen) {
              try {
                s.resize(len);
                boost::system::error_code  ec;
                if (read(sck, buffer(s.data(), len), ec) == len) {
                  rc = sizeof(len) + len;
                }
                else {
                  Syslog(LOG_ERR, "Failed to read string data");
                }
              }
              catch (...) {
                Syslog(LOG_ERR, "Exception reading string data");
              }
            }
            else {
              Syslog(LOG_ERR, "String length %u exceeds max of %u",
                     len, maxLen);
            }
          }
          else {
            rc = sizeof(len);
          }
        }
        else {
          Syslog(LOG_ERR, "Failed to read string length");
        }
      }
      else {
        Syslog(LOG_ERR, "Invalid socket");
      }
      return rc;
    } 
   
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::istream & Utils::ReadLengthRestrictedString(std::istream & is,
                                                     std::string & s,
                                                     uint32_t maxLen) 
    {
      s.clear();
      if (is) {
        uint32_t  len;
        if (StreamIO::Read(is, len)) {
          if (len > 0) {
            if (len <= maxLen) {
              try {
                s.resize(len);
                if (! is.read(s.data(), len)) {
                  Syslog(LOG_ERR, "Failed to read string data");
                  s.clear();
                }
              }
              catch (...) {
                Syslog(LOG_ERR, "Exception reading string data");
              }
            }
            else {
              Syslog(LOG_ERR, "String length %u exceeds max of %u",
                     len, maxLen);
            }
          }
        }
        else {
          Syslog(LOG_ERR, "Failed to read string length");
        }
      }
      else {
        Syslog(LOG_ERR, "Invalid istream");
      }
      return is;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t
    Utils::ReadLengthRestrictedString(int fd, std::string & s,
                                      uint32_t maxLen,
                                      std::chrono::milliseconds timeOut) 
    {
      return ReadLengthRestrictedString(fd, s, maxLen, Clock::now() + timeOut);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t
    Utils::ReadLengthRestrictedString(int fd, std::string & s,
                                      uint32_t maxLen,
                                      TimePoint endTime) 
    {
      s.clear();
      ssize_t  rc = -1;
      if (fd >= 0) {
        uint32_t  len;
        if (WaitUntilBytesReady(fd, sizeof(len), endTime)) {
          if (DescriptorIO::Read(fd, len) == sizeof(len)) {
            if (len > 0) {
              if (len <= maxLen) {
                if (WaitUntilBytesReady(fd, len, endTime)) {
                  try {
                    s.resize(len);
                    if (DescriptorIO::Read(fd, s.data(), len)) {
                      rc = sizeof(len) + len;
                    }
                    else {
                      Syslog(LOG_ERR, "Failed to read string data");
                    }
                  }
                  catch (...) {
                    Syslog(LOG_ERR, "Exception reading string data");
                  }
                }
                else {
                  Syslog(LOG_ERR, "Timed out waiting to read string data");
                }
              }
              else {
                Syslog(LOG_ERR, "String length %u exceeds max of %u",
                       len, maxLen);
              }
            }
            else {
              rc = sizeof(len);
            }
          }
          else {
            Syslog(LOG_ERR, "Failed to read string length");
          }
        }
        else {
          Syslog(LOG_ERR, "Timed out waiting to read string length");
        }
      }
      else {
        Syslog(LOG_ERR, "Invalid descriptor %d", fd);
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ssize_t
    Utils::ReadLengthRestrictedString(boost::asio::ip::tcp::socket & sck,
                                      std::string & s, uint32_t maxLen,
                                      TimePoint endTime) 
    {
      using boost::asio::buffer, boost::asio::read;
      
      s.clear();
      ssize_t  rc = -1;
      if (sck.is_open()) {
        uint32_t  len;
        if (WaitUntilBytesReady(sck, sizeof(len), endTime)) {
          if (ASIO::Read(sck, len)) {
            if (len > 0) {
              if (len <= maxLen) {
                if (WaitUntilBytesReady(sck, len, endTime)) {
                  try {
                    s.resize(len);
                    boost::system::error_code  ec;
                    if (read(sck, buffer(s.data(), len), ec) == len) {
                      rc = sizeof(len) + len;
                    }
                    else {
                      Syslog(LOG_ERR, "Failed to read string data");
                    }
                  }
                  catch (...) {
                    Syslog(LOG_ERR, "Exception reading string data");
                  }
                }
                else {
                  Syslog(LOG_ERR, "Timed out waiting to read string data");
                }
              }
              else {
                Syslog(LOG_ERR, "String length %u exceeds max of %u",
                       len, maxLen);
              }
            }
            else {
              rc = sizeof(len);
            }
          }
          else {
            Syslog(LOG_ERR, "Failed to read string length");
          }
        }
        else {
          Syslog(LOG_ERR, "Timed out waiting to read string length");
        }
      }
      else {
        Syslog(LOG_ERR, "Invalid socket");
      }
      return rc;
    }
    
  }  // namespace Auth

}  // namespace Dwm
