//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmXChaCha20Poly1305OutBuffer.hh 10918 2020-08-15 17:04:47Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmXChaCha20Poly1305OutBuffer.hh
//!  \brief Dwm::XChaCha20Poly1305::OutBuffer class declaration
//---------------------------------------------------------------------------

#ifndef _DWMXCHACHA20POLY1305OUTBUFFER_HH_
#define _DWMXCHACHA20POLY1305OUTBUFFER_HH_

#include <iostream>
#include <string>

#include <cryptopp/cryptlib.h>
#include <cryptopp/chachapoly.h>
#include <cryptopp/osrng.h>

namespace Dwm {

  namespace XChaCha20Poly1305 {

    //------------------------------------------------------------------------
    //!  This class is a helper for encrypted stream output.  It is used as a
    //!  streambuf for an ostream.  It will buffer internally until flush()
    //!  is called on the ostream that owns the buffer (which will end up
    //!  calling our sync() member).  When our sync() member is called, we
    //!  will write a random nonce (initialization vector), the encrypted
    //!  data and the MAC to the associated ostream that was passed as the
    //!  first argument of the constructor.  In other words, message packaging
    //!  and transmission occurs whenever our sync() member is called.
    //!
    //!  This class is typically not used directly, but is instead
    //!  instantiated by Dwm::XChaCha20Poly1305::Ostream.
    //------------------------------------------------------------------------
    class OutBuffer
      : public std::streambuf
    {
    public:
      //----------------------------------------------------------------------
      //!  Construct with the given ostream @c os and encryption key @c key.
      //----------------------------------------------------------------------
      OutBuffer(std::ostream & os, const std::string & key);
      
    protected:
      int_type overflow(int_type c) override;
      int sync() override;
      std::streamsize xsputn(const char *p, std::streamsize n) override;

    private:
      std::ostream                                              & _os;
      CryptoPP::byte                                              _key[32];
      CryptoPP::byte                                              _iv[24];
      CryptoPP::AutoSeededRandomPool                              _rng;
      CryptoPP::XChaCha20Poly1305::Encryption                     _encryptor;
      std::unique_ptr<CryptoPP::AuthenticatedEncryptionFilter>    _aef;
      std::string                                                 _msgbuf;
    };
    

  }  // namespace XChaCha20Poly1305

}  // namespace Dwm

#endif  // _DWMXCHACHA20POLY1305OUTBUFFER_HH_
