//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmXChaCha20Poly1305InBuffer.cc 10899 2020-08-12 19:47:03Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2020
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmXChaCha20Poly1305InBuffer.cc
//!  \brief Dwm::XChaCha20Poly1305::InBuffer class implementation
//---------------------------------------------------------------------------

#include <cryptopp/filters.h>

#include "DwmPortability.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmXChaCha20Poly1305InBuffer.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmAuth/tags/libDwmAuth-0.3.5/src/DwmXChaCha20Poly1305InBuffer.cc 10899 $");

namespace Dwm {

  namespace XChaCha20Poly1305 {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    InBuffer::InBuffer(std::istream & is, const std::string & key)
        : _is(is)
    {
      if (sizeof(_key) <= key.size()) {
        memcpy(_key, key.data(), sizeof(_key));
      }
      else {
        throw std::logic_error("Key not long enough!");
      }
      
      _buffer = nullptr;
      setg(0, 0, 0);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    InBuffer::int_type InBuffer::underflow()
    {
      int_type  rc = traits_type::eof();
      if (gptr() < egptr()) {
        rc = traits_type::to_int_type(*gptr());
      }
      else if (Reload() > 0) {
        rc = traits_type::to_int_type(*gptr());
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    int InBuffer::Reload()
    {
      using namespace CryptoPP;
      using ADF = CryptoPP::AuthenticatedDecryptionFilter;
      
      int  rc = -1;
      CryptoPP::byte  iv[24];
      std::string     cipherText;
      if (LoadIVAndCipherText(iv, sizeof(iv), cipherText)) {
        size_t  bufLen = cipherText.size() - 16;   // MAC is 16 bytes, at end
        _buffer = std::make_unique<char_type[]>(bufLen);
        if (_buffer != nullptr) {
          _decryptor.SetKeyWithIV(_key, sizeof(_key), iv, sizeof(iv));
          ADF  adf(_decryptor,
                   new ArraySink((CryptoPP::byte *)_buffer.get(), bufLen));
          try {
            StringSource(cipherText, true, new Redirector(adf));
            rc = bufLen;
            setg(_buffer.get(), _buffer.get(),
                 _buffer.get() + bufLen);
          }
          catch (std::exception & ex) {
            Syslog(LOG_ERR, "Exception: %s", ex.what());
          }
        }
        else {
          Syslog(LOG_ERR, "Failed to allocate _buffer");
        }
      }
      else {
        Syslog(LOG_ERR, "Failed to read message");
      }
      if (rc < 0) {
        setg(0, 0, 0);
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool InBuffer::LoadIVAndCipherText(CryptoPP::byte *iv, size_t ivlen,
                                       std::string & cipherText)
    {
      bool  rc = false;
      if (_is.read((char *)iv, ivlen)) {
        uint64_t  msgLen;
        if (_is.read((char *)&msgLen, sizeof(msgLen))) {
          msgLen = be64toh(msgLen);
          try {
            cipherText.resize(msgLen);
            if (_is.read(cipherText.data(), msgLen)) {
              rc = true;
            }
          }
          catch (...) {
          }
        }
      }
      return rc;
    }
    
    
  }  // namespace XChaCha20Poly1305

}  // namespace Dwm
