//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmAuthPublicKeysFile.cc 10617 2020-05-02 21:18:04Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmAuthPublicKeysFile.cc
//!  \brief Dwm::Auth::PublicKeysFile implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <fcntl.h>
  #include <unistd.h>
}

#include <cstring>
#include <fstream>
#include <regex>

#include "DwmSvnTag.hh"
#include "DwmAuthPublicKeysFile.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmAuth/tags/libDwmAuth-0.3.5/src/DwmAuthPublicKeysFile.cc 10617 $");

using namespace std;

namespace Dwm {

  namespace Auth {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    PublicKeysFile::PublicKeysFile()
        : _keys()
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PublicKeysFile::Load(const string & fileName)
    {
      bool  rc = false;
      _keys.clear();
      ifstream  is(fileName.c_str());
      if (is) {
        regex   rgx("([^ \\t]+)[ \\t]+([^ \\t]+)[ \\t]+([^ \\t]+)",
                    regex::ECMAScript|regex::optimize);
        string  s;
        while (getline(is, s, '\n')) {
          if (! s.empty()) {
            smatch  sm;
            if (regex_match(s, sm, rgx)) {
              if ((sm.size() == 4) && (sm[2].str() == "dwm-ed25519")) {
                Ed25519::PublicKey  pk;
                if (pk.FromBase64String(sm[3].str())) {
                  if (pk.IsValid()) {
                    _keys[sm[1].str()] = pk;
                  }
                }
              }
            }
          }
        }
        is.close();
      }
      return (! _keys.empty());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PublicKeysFile::AddKey(const string & name,
                                const Ed25519::PublicKey & key)
    {
      bool  rc = false;
      auto  i = _keys.find(name);
      if (i == _keys.end()) {
        _keys[name] = key;
        rc = true;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PublicKeysFile::AddKey(const std::string & name,
                                const std::string & base64str)
    {
      bool  rc = false;
      Ed25519::PublicKey  key;
      if (key.FromBase64String(base64str)) {
        if (key.IsValid()) {
          rc = AddKey(name, key);
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PublicKeysFile::RemoveKey(const std::string & name)
    {
      bool  rc = false;
      auto  i = _keys.find(name);
      if (i != _keys.end()) {
        _keys.erase(i);
        rc = true;
      }
      return rc;
    }

#ifdef __linux__
    //------------------------------------------------------------------------
    //!  Linux can't acquire lock on open(), so we need to explicitly lock
    //!  after open().  We'll use fcntl().
    //------------------------------------------------------------------------
    static bool WriteLockFile(int fd)
    {
      bool  rc = false;
      if (fd >= 0) {
        struct flock  lock;
        lock.l_type = F_WRLCK;
        lock.l_whence = SEEK_SET;
        lock.l_start = SEEK_SET;
        lock.l_len = 0;
        if (fcntl(fd, F_SETLKW, &lock) != -1) {
          rc = true;
        }
      }
      return rc;
    }
#endif
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PublicKeysFile::Save(const string & fileName, int perms) const
    {
      bool  rc = false;

#ifndef __linux__
      int  fd = open(fileName.c_str(), O_EXLOCK|O_CREAT|O_WRONLY|O_TRUNC,
                     perms);
#else
      int  fd = open(fileName.c_str(), O_CREAT|O_WRONLY|O_TRUNC,
                     perms);
      if (! WriteLockFile(fd)) {
        close(fd);
        fd = -1;
      }
#endif
      if (fd >= 0) {
        const char  *ws = " ";
        const char  *keyType = "dwm-ed25519";
        const char   nl = '\n';
        auto  i = _keys.begin();
        for ( ; i != _keys.end(); ++i) {
          if (write(fd, i->first.c_str(), i->first.size())
              != i->first.size()) {
            break;
          }
          if (write(fd, ws, strlen(ws)) != strlen(ws)) {
            break;
          }
          if (write(fd, keyType, strlen(keyType)) != strlen(keyType)) {
            break;
          }
          if (write(fd, ws, strlen(ws)) != strlen(ws)) {
            break;
          }
          string  pk64str = i->second.ToBase64String();
          if (write(fd, pk64str.c_str(), pk64str.size())
              != pk64str.size()) {
            break;
          }
          if (write(fd, &nl, 1) != 1) {
            break;
          }
        }
        rc = (i == _keys.end());
        close(fd);
      }
      
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PublicKeysFile::Find(const string & name,
                              Ed25519::PublicKey & key) const
    {
      bool  rc = false;
      auto  it = _keys.find(name);
      if (it != _keys.end()) {
        key = it->second;
        rc = true;
      }
      return rc;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const PublicKeysFile::Map & PublicKeysFile::Keys() const
    {
      return _keys;
    }
    
    
  }  // namespace Auth

}  // namespace Dwm
