//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmAuth.cc 10617 2020-05-02 21:18:04Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmAuth.cc
//!  \brief Simple authentication implementations
//---------------------------------------------------------------------------

extern "C" {
  #include <unistd.h>
}

#include <iomanip>
#include <sstream>

#include <cryptopp/base64.h>
#include <cryptopp/hex.h>

#include "DwmIO.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmAuthCountedString.hh"
#include "DwmAuth.hh"

static const Dwm::SvnTag  svntag("$DwmPath: dwm/libDwmAuth/tags/libDwmAuth-0.3.3/src/DwmAuth.cc 10617 $");

//----------------------------------------------------------------------------
//!  Maximum length of string we'll accept during authentication.  This is
//!  much bigger than anything we should see, but it's here to prevent
//!  a client or server from presenting a monster-sized public key.
//----------------------------------------------------------------------------
#define MAX_BINARY_STRING_LEN  65535

using namespace std;

namespace Dwm {

  namespace Auth {

#if 0
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string GetRandomString(size_t sz)
    {
#ifdef __linux__
      CryptoPP::NonblockingRng  rng;
#else
      CryptoPP::BlockingRng  rng;
#endif
      uint8_t  buf[sz];
      rng.GenerateBlock(buf, sz);
      string  s;
      s.assign((const char *)buf, sz);
      return s;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static bool WriteBinaryString(int fd, const string & s)
    {
      bool  rc = false;
      if (fd >= 0) {
        uint32_t  len = s.size();
        if (IO::Write(fd, len) > 0) {
          if (write(fd, s.data(), s.size()) == s.size()) {
            rc = true;
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static bool ReadBinaryString(int fd, string & s)
    {
      bool  rc = false;
      if (fd >= 0) {
        uint32_t  len;
        if ((IO::Read(fd, len) > 0) && (len < MAX_BINARY_STRING_LEN)) {
          char  buf[len];
          if (read(fd, buf, len) == len) {
            s.assign(buf, len);
            rc = true;
          }
        }
      }
      return rc;
    }
#endif
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string HexEncodedString(const string & s)
    {
      ostringstream  os;
      os << hex << setfill('0');
      for (size_t i = 0; i < s.size(); ++i) {
        os << setw(2) << (uint16_t)s[i];
      }
      return os.str();
    }

#if 0
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static bool ValidatePublicKey(const PKCrypto::PublicKey & key)
    {
      CryptoPP::AutoSeededRandomPool  rng;
      return key.Validate(rng, 2);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AuthenticateClient(int fd, const PKCrypto::KeyPair & myKeys,
                            const string & secret)
    {
      //  Send our public key to the client.
      if (IO::Write(fd, myKeys.Public()) <= 0) {
        Syslog(LOG_ERR, "Failed to send public key to client");
        return false;
      }
      //  Read the client's public key.
      PKCrypto::PublicKey  clientPubKey;
      size_t  bytesRead = IO::Read(fd, clientPubKey);
      if (bytesRead != IO::StreamedLength(clientPubKey)) {
        Syslog(LOG_ERR, "Failed to read public key from client");
        return false;
      }
      //  Validate the client's public key
      if (! ValidatePublicKey(clientPubKey)) {
        Syslog(LOG_ERR, "Invalid public key from client");
        return false;
      }
      //  Send an IV to client so they can use it when decrypting
      CountedString  iv = GetRandomString(CryptoPP::AES::BLOCKSIZE);
      if (iv.Write(fd) != iv.StreamedLength()) {
        syslog(LOG_ERR, "Failed to send IV to client");
        return false;
      }
      //  Encrypt random text with the ecrypted version of our shared
      //  secret key and then with the client's public key.  This is our
      //  'challenge'.
      string  challengePlain = GetRandomString(16);
      string  challengeCipher = SymCrypto::Encrypt(secret, iv.Value(),
                                                   challengePlain);
      CountedString  challengeToSend = 
        PKCrypto::Encrypt(clientPubKey, challengeCipher);
      //  Send the challenge to the client.
      if (challengeToSend.Write(fd) != challengeToSend.StreamedLength()) {
        Syslog(LOG_ERR, "Failed to send challenge to client");
        return false;
      }
      //  Read the client's challenge response.
      CountedString  responseCipher;
      if (responseCipher.Read(fd) < 0) {
        Syslog(LOG_ERR, "Failed to read challenge response from client");
        return false;
      }
      //  Decrypt the client's challenge response.
      string  responsePlain =
        PKCrypto::Decrypt(myKeys.Private(), responseCipher);
      //  Check that the client's challenge response is equal
      //  to our random challenge.
      if (responsePlain != challengePlain) {
        Syslog(LOG_ERR, "Incorrect challenge response");
        return false;
      }

      return true;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool AuthenticateServer(int fd, const PKCrypto::KeyPair & myKeys,
                            const std::string & secret)
    {
      bool  rc = false;
      
      PKCrypto::PublicKey  servPubKey;
      if (IO::Read(fd, servPubKey) > 0) {
        if (IO::Write(fd, myKeys.Public()) > 0) {
          //  Read IV from server so we can decrypt challenge
          CountedString  iv;
          if (iv.Read(fd) > 0) {
            CountedString  chrecvd;
            //  Read the challenge from server.  The challenge is a random
            //  string encrypted with the shared secret key and then encrypted
            //  with our public key.
            if (chrecvd.Read(fd) > 0) {
              //  Decrypt with our private key.
              string  chcipher = PKCrypto::Decrypt(myKeys.Private(), chrecvd);
              //  Decrypt with our shared secret and the IV from the server.
              string  chplain = SymCrypto::Decrypt(secret, iv, chcipher);
              //  Now encrypt the challenge with the server's public key.
              CountedString  chresp = PKCrypto::Encrypt(servPubKey, chplain);
              //  And send the response to the server.
              if (chresp.Write(fd) == chresp.StreamedLength()) {
                rc = true;
              }
              else {
                Syslog(LOG_ERR, "Failed to send challenge response to server");
              }
            }
            else {
              Syslog(LOG_ERR, "Failed to read challenge from server");
            }
          }
          else {
            Syslog(LOG_ERR, "Failed to read IV from server");
          }
        }
        else {
          Syslog(LOG_ERR, "Failed to send public key to server");
        }
      }
      else {
        Syslog(LOG_ERR, "Failed to read public key from server");
      }
      return rc;
    }
#endif
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string Base64Encode(const string & s) 
    {
      string  sb;
      CryptoPP::StringSource
        ss((uint8_t *)(s.c_str()), s.size(), true,
           new CryptoPP::Base64Encoder(new CryptoPP::StringSink(sb), false));
      return sb;
    }
    
  }  // namespace Auth

}  // namespace Dwm
