//===========================================================================
// @(#) $DwmPath: dwm/libDwmAuth/tags/libDwmAuth-0.3.2/src/DwmAuthSymCryptoMessage.cc 9848 $
// @(#) $Id: DwmAuthSymCryptoMessage.cc 9848 2017-12-18 22:25:38Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmAuthSymCryptoMessage.cc
//!  \brief Dwm::Auth::SymCrypto::Message implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <string.h>  // for memset()
  #include <unistd.h>  // for read() and write()
}

#include <sstream>
#include <cryptopp/osrng.h>

#include "DwmIO.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmAuthSymCrypto.hh"
#include "DwmAuthSymCryptoMessage.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmAuth/tags/libDwmAuth-0.3.2/src/DwmAuthSymCryptoMessage.cc 9848 $");

using namespace std;

namespace Dwm {

  namespace Auth {

    namespace SymCrypto {

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      Message::Message(const string & secret, const string & value)
          : _secret(secret), _msg()
      {
        if (! value.empty()) {
#ifdef __linux__
          CryptoPP::NonblockingRng  rng;
#else
          CryptoPP::BlockingRng  rng;
#endif
          rng.GenerateBlock(_iv, CryptoPP::AES::BLOCKSIZE);
          string  ivstr((const char *)_iv, CryptoPP::AES::BLOCKSIZE);
          _msg = Encrypt(_secret, ivstr, value);
        }
      }
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      Message::~Message()
      {
        _secret.assign(_secret.size(), 0);
        memset(_iv, 0, sizeof(_iv));
      }
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      string Message::Value() const
      {
        string  iv((const char *)_iv, sizeof(_iv));
        string  val = Decrypt(_secret, iv, _msg);
        return val;
      }
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      const string & Message::Value(const string & value)
      {
        string  iv((const char *)_iv, sizeof(_iv));
        _msg = Encrypt(_secret, iv, value);
        return value;
      }
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      istream & Message::Read(istream & is)
      {
        if (is) {
          if (is.read((char *)_iv, CryptoPP::AES::BLOCKSIZE)) {
            ReadMsgValue(is);
          }
        }
        return is;
      }
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      ostream & Message::Write(ostream & os) const
      {
        if (os) {
          if (os.write((const char *)_iv, CryptoPP::AES::BLOCKSIZE)) {
            WriteMsgValue(os);
          }
        }
        return os;
      }
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      ssize_t Message::Read(int fd)
      {
        ssize_t  rc = -1;
        if (fd >= 0) {
          ssize_t  bytesRead = read(fd, _iv, sizeof(_iv));
          if (bytesRead == sizeof(_iv)) {
            rc = bytesRead;
            bytesRead = ReadMsgValue(fd);
            if (bytesRead > 0) {
              rc += bytesRead;
            }
            else {
              rc = -1;
            }
          }
        }
        return rc;
      }
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      ssize_t Message::Write(int fd) const
      {
        ssize_t  rc = -1;
        if (fd >= 0) {
#if 1
          ostringstream  oss;
          oss.write((const char *)_iv, sizeof(_iv));
          WriteMsgValue(oss);
          string   s = oss.str();
          ssize_t  bytesWritten = write(fd, s.c_str(), s.size());
          if (bytesWritten == s.size()) {
            rc = bytesWritten;
          }
          else {
            rc = -1;
          }
#else
          ssize_t  bytesWritten = write(fd, _iv, sizeof(_iv));
          if (bytesWritten == sizeof(_iv)) {
            rc = bytesWritten;
            bytesWritten = WriteMsgValue(fd);
            if (bytesWritten > 0) {
              rc += bytesWritten;
            }
            else {
              rc = -1;
            }
          }
#endif
        }
        return rc;
      }
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      size_t Message::Read(FILE *f)
      {
        size_t  rc = 0;
        if (f) {
          if (fread(_iv, sizeof(_iv), 1, f) == 1) {
            if (ReadMsgValue(f)) {
              rc = 1;
            }
          }
        }
        return rc;
      }
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      size_t Message::Write(FILE *f) const
      {
        size_t  rc = 0;
        if (f) {
          if (fwrite(_iv, sizeof(_iv), 1, f) == 1) {
            if (WriteMsgValue(f)) {
              rc = 1;
            }
          }
        }
        return rc;
      }
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      uint32_t Message::StreamedLength() const
      {
        return sizeof(_iv) + sizeof(uint32_t) + _msg.size();
      }

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      ssize_t Message::ReadMsgValue(int fd)
      {
        return IO::Read(fd, _msg);
      }
          
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      ssize_t Message::WriteMsgValue(int fd) const
      {
        return IO::Write(fd, _msg);
      }
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      istream & Message::ReadMsgValue(istream & is)
      {
        return IO::Read(is, _msg);
      }
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      ostream & Message::WriteMsgValue(ostream & os) const
      {
        return IO::Write(os, _msg);
      }
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      size_t Message::ReadMsgValue(FILE *f)
      {
        return IO::Read(f, _msg);
      }
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      size_t Message::WriteMsgValue(FILE *f) const
      {
        return IO::Write(f, _msg);
      }
      
      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      ssize_t Message::SendTo(Socket & s, int flags,
                              const Ipv4Address & dstAddr, uint16_t dstPort)
      {
        ssize_t  rc = -1;
        std::ostringstream  os;
        if (Write(os)) {
          rc = s.SendTo(os.str(), 0, dstAddr, dstPort);
        }
        return rc;
      }

      //----------------------------------------------------------------------
      //!  
      //----------------------------------------------------------------------
      ssize_t Message::RecvFrom(Socket & s, int flags,
                                Ipv4Address & srcAddr, uint16_t & srcPort)
      {
        ssize_t  rc = -1;
        std::string  rs;
        rc = s.RecvFrom(rs, 0, srcAddr, srcPort);
        if (rc > 0) {
          istringstream  is(rs);
          if (! Read(is)) {
            rc = -1;
          }
        }
        return rc;
      }
      
      
    }  // namespace SymCrypto

  }  // namespace Auth

}  // namespace Dwm
