//===========================================================================
// @(#) $DwmPath: dwm/libDwmAuth/tags/libDwmAuth-0.2.1/tests/TestDwmAuthPKCrypto.cc 9039 $
// @(#) $Id: TestDwmAuthPKCrypto.cc 9039 2017-04-12 07:47:51Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file TestDwmAuthPKCrypto.cc
//!  \brief Unit tests for Dwm::Auth::PKCrypto
//---------------------------------------------------------------------------

extern "C" {
  #include <fcntl.h>
  #include <unistd.h>
}

#include <sstream>
#include <cryptopp/base64.h>
#include <cryptopp/files.h>

#include "DwmSvnTag.hh"
#include "DwmUnitAssert.hh"
#include "DwmAuthPKCrypto.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmAuth/tags/libDwmAuth-0.2.1/tests/TestDwmAuthPKCrypto.cc 9039 $");

using namespace std;
using namespace Dwm;
using namespace Dwm::Auth;

static void FooSaveKeys(const PKCrypto::KeyPair & keyPair)
{
  CryptoPP::FileSink  pubsink("./foo.pub1");
  CryptoPP::Base64Encoder  pubenc(&pubsink, false);
  keyPair.Public().Save(pubenc);

  CryptoPP::FileSink  privsink("./foo.priv1");
  CryptoPP::Base64Encoder  privenc(&privsink, false);
  keyPair.Private().Save(privenc);

  ofstream  ospub("./foo.pub");
  ospub << keyPair.Public().ToBase64String() << '\n';
  ospub.close();
  ofstream  ospriv("./foo.priv");
  ospriv << keyPair.Private().ToBase64String() << '\n';
  ospriv.close();
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void TestDescriptors(const PKCrypto::KeyPair & keyPair)
{
  string  plainstr = "Oh, and one more thing. Bite me, Yoko. Bite me hard.";
  string  cipherstr = PKCrypto::Encrypt(keyPair.Public(), plainstr);
  UnitAssert(PKCrypto::Decrypt(keyPair.Private(), cipherstr) == plainstr);

  int  fd = open("TestDescriptorsPubKey.tmp", O_WRONLY|O_CREAT, 0600);
  UnitAssert(keyPair.Public().Write(fd));
  close(fd);
  fd = open("TestDescriptorsPrivKey.tmp", O_WRONLY|O_CREAT, 0600);
  UnitAssert(keyPair.Private().Write(fd));
  close(fd);

  fd = open("TestDescriptorsPrivKey.tmp", O_RDONLY);
  PKCrypto::PrivateKey  privKey;
  UnitAssert(privKey.Read(fd));
  close(fd);
  UnitAssert(PKCrypto::Decrypt(privKey, cipherstr) == plainstr);

  fd = open("TestDescriptorsPubKey.tmp", O_RDONLY);
  PKCrypto::PublicKey  pubKey;
  UnitAssert(pubKey.Read(fd));
  close(fd);
  cipherstr = PKCrypto::Encrypt(pubKey, plainstr);
  UnitAssert(PKCrypto::Decrypt(privKey, cipherstr) == plainstr);
  remove("TestDescriptorsPrivKey.tmp");
  remove("TestDescriptorsPubKey.tmp");

  //  FooSaveKeys(keyPair);
  
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void TestIOStreams(const PKCrypto::KeyPair & keyPair)
{
  string  plainstr = "Dear Woz: Please start mincing words.";
  string  cipherstr = PKCrypto::Encrypt(keyPair.Public(), plainstr);
  UnitAssert(PKCrypto::Decrypt(keyPair.Private(), cipherstr) == plainstr);
  
  ostringstream  ospub, ospriv;
  UnitAssert(keyPair.Public().Write(ospub));
  UnitAssert(keyPair.Private().Write(ospriv));

  istringstream  ispriv(ospriv.str());
  PKCrypto::PrivateKey  privKey;
  UnitAssert(privKey.Read(ispriv));
  UnitAssert(PKCrypto::Decrypt(privKey, cipherstr) == plainstr);

  istringstream  ispub(ospub.str());
  PKCrypto::PublicKey  pubKey;
  UnitAssert(pubKey.Read(ispub));
  cipherstr = PKCrypto::Encrypt(pubKey, plainstr);
  UnitAssert(PKCrypto::Decrypt(privKey, cipherstr) == plainstr);
  return;  
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void TestFiles(const PKCrypto::KeyPair & keyPair)
{
  string  plainstr = "Maybe we shouldn't have sued our own customers.";
  string  cipherstr = PKCrypto::Encrypt(keyPair.Public(), plainstr);
  UnitAssert(PKCrypto::Decrypt(keyPair.Private(), cipherstr) == plainstr);

  FILE  *f = fopen("TestFilesPubKey.tmp", "wb");
  UnitAssert(keyPair.Public().Write(f));
  fclose(f);
  f = fopen("TestFilesPrivKey.tmp", "wb");
  UnitAssert(keyPair.Private().Write(f));
  fclose(f);

  f = fopen("TestFilesPrivKey.tmp", "rb");
  PKCrypto::PrivateKey  privKey;
  UnitAssert(privKey.Read(f));
  fclose(f);
  UnitAssert(PKCrypto::Decrypt(privKey, cipherstr) == plainstr);

  f = fopen("TestFilesPubKey.tmp", "rb");
  PKCrypto::PublicKey  pubKey;
  UnitAssert(pubKey.Read(f));
  fclose(f);
  cipherstr = PKCrypto::Encrypt(pubKey, plainstr);
  UnitAssert(PKCrypto::Decrypt(privKey, cipherstr) == plainstr);
  remove("TestFilesPubKey.tmp");
  remove("TestFilesPrivKey.tmp");
  return;  
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  PKCrypto::KeyPair  keyPair = PKCrypto::CreateKeyPair(1024);
  string  plainstr = "Dude, where's my car?!";
  string  cipherstr = PKCrypto::Encrypt(keyPair.Public(), plainstr);
  UnitAssert(PKCrypto::Decrypt(keyPair.Private(), cipherstr) == plainstr);

  TestIOStreams(keyPair);
  TestDescriptors(keyPair);
  TestFiles(keyPair);
  
  int  rc;
  
  if (Assertions::Total().Failed()) {
    Assertions::Print(std::cerr, true);
    rc = 1;
  }
  else {
    std::cout << Assertions::Total() << " passed" << std::endl;
    rc = 0;
  }

  return rc;
}
