//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmAuthPeerAuthenticator.cc 10617 2020-05-02 21:18:04Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmAuthPeerAuthenticator.cc
//!  \brief Dwm::Auth::PeerAuthenticator class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <sys/un.h>
  #include <netinet/in.h>
  #include <netinet/tcp.h>
  #include <arpa/inet.h>
  #include <pwd.h>
  #include <unistd.h>
}

#include <cassert>
#include <sstream>
#include <cryptopp/cryptlib.h>
#include <cryptopp/rsa.h>
#include <cryptopp/pssr.h>
#include <cryptopp/secblock.h>

#include "DwmIO.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmAuthPeerAuthenticator.hh"
#include "DwmAuthSymCrypto.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmAuth/tags/libDwmAuth-0.2.1/src/DwmAuthPeerAuthenticator.cc 10617 $");

using namespace std;
using CryptoPP::PSS;
using CryptoPP::SHA256;
using CryptoPP::RSASS;
using CryptoPP::StringSource;
using CryptoPP::StringSink;
using CryptoPP::SignerFilter;

namespace Dwm {

  namespace Auth {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static string to_str(const CryptoPP::SecByteBlock & sbb)
    {
      return string((const char *)sbb.BytePtr(), sbb.SizeInBytes());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static CryptoPP::SecByteBlock to_SecByteBlock(const string & s)
    {
      return CryptoPP::SecByteBlock((const uint8_t *)s.c_str(), s.size());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    PeerAuthenticator::PeerAuthenticator(const string & myPrivKeyPath,
                                         const string & pubKeysPath)
        : _pubKeysPath(pubKeysPath)
    {
      std::string  privKeyPath = myPrivKeyPath;
      
      struct passwd  *pw = 0;
      if (privKeyPath.empty() || _pubKeysPath.empty()) {
        uid_t   uid = getuid();
        pw = getpwuid(uid);
      }
      if (privKeyPath.empty()) {
        if (pw) {
          privKeyPath = pw->pw_dir;
          privKeyPath += "/.dwmauth/id_ed25519";
        }
      }
      if (_pubKeysPath.empty()) {
        if (pw) {
          _pubKeysPath = pw->pw_dir;
          _pubKeysPath += "/.dwmauth/known_services";
        }
      }
      assert(_myKeys.Load(privKeyPath, _myId));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    PeerAuthenticator::PeerAuthenticator(const std::string & myId,
                                         const Ed25519::KeyPair & keyPair,
                                         const std::string & pubKeysPath)
        : _myId(myId), _myKeys(keyPair), _pubKeysPath(pubKeysPath)
    {
      if (_pubKeysPath.empty()) {
        uid_t           uid = getuid();
        struct passwd  *pw = getpwuid(uid);
        if (pw) {
          _pubKeysPath = pw->pw_dir;
          _pubKeysPath += "/.dwmauth/known_services";
        }
      }
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PeerAuthenticator::ECExchange(int fd,
                                       Auth::ECDHAgreement & agreement)
    {
      bool  rc = false;
      if (fd >= 0) {
        string  mypubEC = to_str(agreement.Public());
        if (IO::Write(fd, mypubEC) == IO::StreamedLength(mypubEC)) {
          string  theirPubECDH;
          if (IO::Read(fd, theirPubECDH) > 0) {
            agreement.TheirPublic(to_SecByteBlock(theirPubECDH));
            // CryptoPP::SecByteBlock  agreedKeyBB;
            if (agreement.Agree()) {
              rc = true;
            }
            else {
              Syslog(LOG_ERR, "Failed to reach agreement");
            }
          }
          else {
            Syslog(LOG_ERR, "Read(%d) failed: %m", fd);
          }
        }
        else {
          Syslog(LOG_ERR, "Write(%d) failed: %m", fd);
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string PeerAuthenticator
    ::BuildIDAndSignature(const ECDHAgreement & agreement)
    {
      string  agreedKeyStr(to_str(agreement.AgreedKey()));
      string  keystr(agreedKeyStr.substr(0, 16));
      string  ivstr(agreedKeyStr.substr(16, 16));
      string  myIdCrypted = SymCrypto::Encrypt(keystr, ivstr, _myId);
      string  msgToSign(to_str(agreement.Public()) + myIdCrypted);
      CryptoPP::ByteQueue  bt;
      _myKeys.Private().Save(bt, true);

      CryptoPP::ed25519::Signer  signer(bt);
      string  mySignature;
      CryptoPP::AutoSeededRandomPool  rng;
      StringSource ss(msgToSign, true,
                      new SignerFilter(rng, signer,
                                       new StringSink(mySignature)));
      ostringstream  os;
      IO::Write(os, myIdCrypted);
      IO::Write(os, mySignature);
      return os.str();
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PeerAuthenticator
    ::IDAndSigExchange(int fd,
                       const ECDHAgreement & agreement,
                       string & theirId,
                       string & theirIdCrypted,
                       string & theirSig)
    {
      bool  rc = false;
      if (fd >= 0) {
        string idAndSig = BuildIDAndSignature(agreement);
        if (IO::Write(fd, idAndSig)) {
          string  s;
          if (IO::Read(fd, s) > 0) {
            string  keystr((to_str(agreement.AgreedKey())).substr(0, 16));
            string  ivstr((to_str(agreement.AgreedKey())).substr(16, 16));
            istringstream  is(s);
            if (IO::Read(is, theirIdCrypted)) {
              theirId = SymCrypto::Decrypt(keystr, ivstr, theirIdCrypted);
              if (IO::Read(is, theirSig)) {
                rc = true;
              }
              else {
                Syslog(LOG_ERR, "Failed to read signature");
              }
            }
            else {
              Syslog(LOG_ERR, "Failed to read encrypted ID");
            }
          }
          else {
            Syslog(LOG_ERR, "Failed to read ID and signature from %d", fd);
          }
        }
        else {
          Syslog(LOG_ERR, "Failed to write ID and signature to %d", fd);
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PeerAuthenticator::Authenticate(int fd, string & theirId,
                                         string & agreedKey)
    {
      bool  rc = false;
      theirId.clear();
      agreedKey.clear();
      if (fd >= 0) {
        struct sockaddr_in  sockAddr;
        socklen_t           sockLen = sizeof(sockAddr);
        getpeername(fd, (struct sockaddr *)&sockAddr, &sockLen);

        int tcpNoDelay = 1;
        setsockopt(fd, IPPROTO_TCP, TCP_NODELAY, (char *)&tcpNoDelay,
                   sizeof(tcpNoDelay));
        Auth::ECDHAgreement  agreement;
        if (ECExchange(fd, agreement)) {
          string  theirIdCrypted, theirSig;
          if (IDAndSigExchange(fd, agreement, theirId,
                               theirIdCrypted, theirSig)) {
            Auth::PublicKeysFile  pubKeysFile;
            pubKeysFile.Load(_pubKeysPath);
            Ed25519::PublicKey  theirPubKey;
            if (pubKeysFile.Find(theirId, theirPubKey)) {
              string  msgToVerify =
                to_str(agreement.TheirPublic()) + theirIdCrypted;
              CryptoPP::ByteQueue  bt;
              theirPubKey.Save(bt);
              CryptoPP::ed25519::Verifier  verifier(bt);
              if (verifier.VerifyMessage((const uint8_t *)msgToVerify.c_str(),
                                         msgToVerify.size(),
                                         (const uint8_t *)theirSig.c_str(),
                                         theirSig.size())) {
                agreedKey = to_str(agreement.AgreedKey());
                rc = true;
              }
              else {
                Syslog(LOG_ERR, "Failed to verify message from %s",
                       theirId.c_str());
              }
            }
            else {
              Syslog(LOG_ERR, "Failed to find %s in %s",
                     theirId.c_str(), _pubKeysPath.c_str());
            }
          }
          else {
            Syslog(LOG_ERR, "ID and signature exhange with %s failed",
                   inet_ntoa(sockAddr.sin_addr));
          }
        }
        else {
          Syslog(LOG_ERR, "ECDH exhange with %s failed",
                 inet_ntoa(sockAddr.sin_addr));
        }
      }
      return rc;
    }
    

  }  // namespace Auth

}  // namespace Dwm
