//===========================================================================
// @(#) $DwmPath: dwm/libDwmAuth/tags/libDwmAuth-0.2.0/include/DwmAuthPKCrypto.hh 9849 $
// @(#) $Id: DwmAuthPKCrypto.hh 9849 2017-12-18 23:28:41Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmAuthPKCrypto.hh
//!  \brief Public key crypto definitions
//---------------------------------------------------------------------------

#ifndef _DWMAUTHPKCRYPTO_HH_
#define _DWMAUTHPKCRYPTO_HH_

#include <string>
#include <utility>

#include <cryptopp/cryptlib.h>
#include <cryptopp/aes.h>
#include <cryptopp/filters.h>
#include <cryptopp/gcm.h>
#include <cryptopp/osrng.h>
#include <cryptopp/rsa.h>

#include "DwmReadable.hh"
#include "DwmWritable.hh"

namespace Dwm {

  namespace Auth {

    //------------------------------------------------------------------------
    //!  Public key crypto utilities.
    //------------------------------------------------------------------------
    class PKCrypto
    {
    public:
      //----------------------------------------------------------------------
      //!  Encapsulates a readable and writable public key.
      //----------------------------------------------------------------------
      class PublicKey
        : public CryptoPP::RSA::PublicKey,
          public Dwm::Readable, public Dwm::Writable
      {
      public:
        PublicKey()
            : CryptoPP::RSA::PublicKey()
        {}
        
        PublicKey(const CryptoPP::RSA::PublicKey & key)
            : CryptoPP::RSA::PublicKey(key)
        {}

        bool IsValid() const;
        
        std::istream & Read(std::istream & is);
        ssize_t Read(int fd);
        size_t Read(FILE * f);
        
        std::ostream & Write(std::ostream & os) const;
        ssize_t Write(int fd) const;
        size_t Write(FILE *f) const;
        uint32_t StreamedLength() const;
        
        std::string ToString() const;
        std::string ToBase64String() const;
        bool FromString(const std::string & s);
        bool FromBase64String(const std::string & s);
        
        //--------------------------------------------------------------------
        //!  operator ==
        //--------------------------------------------------------------------
        bool operator == (const PublicKey & pk) const;
      };
      
      //----------------------------------------------------------------------
      //!  Encapsulates a readable and writable private key.
      //----------------------------------------------------------------------
      class PrivateKey
        : public CryptoPP::RSA::PrivateKey,
          public Dwm::Readable, public Dwm::Writable
      {
      public:
        PrivateKey()
            : CryptoPP::RSA::PrivateKey()
        {}
        
        PrivateKey(const CryptoPP::RSA::PrivateKey & key)
            : CryptoPP::RSA::PrivateKey(key)
        {}

        bool IsValid() const;

        std::istream & Read(std::istream & is);
        ssize_t Read(int fd);
        size_t Read(FILE * f);
        
        std::ostream & Write(std::ostream & os) const;
        ssize_t Write(int fd) const;
        size_t Write(FILE *f) const;
        uint32_t StreamedLength() const;
        
        std::string ToString() const;
        std::string ToBase64String() const;
        bool FromString(const std::string & s);
        bool FromBase64String(const std::string & s);

        //--------------------------------------------------------------------
        //!  operator ==
        //--------------------------------------------------------------------
        bool operator == (const PrivateKey & pk) const;
      };
      
      //----------------------------------------------------------------------
      //!  Encapsulates a private/public key pair.
      //----------------------------------------------------------------------
      class KeyPair
        : public std::pair<PrivateKey,PublicKey>
      {
      public:
        //--------------------------------------------------------------------
        //!  Default constructor.
        //--------------------------------------------------------------------
        KeyPair()
            : std::pair<PrivateKey,PublicKey>()
        {}
        
        //--------------------------------------------------------------------
        //!  Construct from a private and public key.
        //--------------------------------------------------------------------
        KeyPair(const PrivateKey & privKey, const PublicKey & publicKey)
            : std::pair<PrivateKey,PublicKey>(privKey,publicKey)
        {}

        //--------------------------------------------------------------------
        //!  operator =
        //--------------------------------------------------------------------
        KeyPair & operator = (const KeyPair & keys)
        {
          if (this != &keys) {
            this->first = keys.first;
            this->second = keys.second;
          }
          return *this;
        }
        
        //--------------------------------------------------------------------
        //!  Returns a const reference to the private key.
        //--------------------------------------------------------------------
        const PrivateKey & Private() const
        {
          return first;
        }
        
        //--------------------------------------------------------------------
        //!  Returns a mutable reference to the private key.
        //--------------------------------------------------------------------
        PrivateKey & Private()
        {
          return first;
        }
        
        //--------------------------------------------------------------------
        //!  Returns a const reference to the public key.
        //--------------------------------------------------------------------
        const PublicKey & Public() const
        {
          return second;
        }
        
        //--------------------------------------------------------------------
        //!  Returns a mutable reference to the public key.
        //--------------------------------------------------------------------
        PublicKey & Public()
        {
          return second;
        }

        bool Load(const std::string & privKeyPath, std::string & name);
      };
      
      //----------------------------------------------------------------------
      //!  Generate a private/public key pair.  The private key is the
      //!  first key, the public key is the second key.
      //----------------------------------------------------------------------
      static KeyPair CreateKeyPair(unsigned int keySize);
      
      //----------------------------------------------------------------------
      //!  Encrypts the given @c plainstr with the given @c publicKey and
      //!  returns the encrypted string.
      //----------------------------------------------------------------------
      static std::string
      Encrypt(const PublicKey & publicKey, const std::string & plainstr);
      
      //----------------------------------------------------------------------
      //!  Decrypts the given @c cipherstr using the given @c privateKey
      //!  and returns the decrypted string.
      //----------------------------------------------------------------------
      static std::string
      Decrypt(const PrivateKey & privateKey, const std::string & cipherstr);

    };
    
  }  // namespace Auth

}  // namespace Dwm

#endif  // _DWMAUTHPKCRYPTO_HH_
