//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmAuthPeerAuthenticator.cc 9028 2017-04-12 01:18:31Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmAuthPeerAuthenticator.cc
//!  \brief Dwm::Auth::PeerAuthenticator class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <sys/socket.h>
  #include <sys/un.h>
  #include <netinet/in.h>
  #include <arpa/inet.h>
  #include <pwd.h>
  #include <unistd.h>
}

#include <sstream>
#include <cryptopp/cryptlib.h>
#include <cryptopp/rsa.h>
#include <cryptopp/pssr.h>
#include <cryptopp/secblock.h>

#include "DwmIO.hh"
#include "DwmSvnTag.hh"
#include "DwmSysLogger.hh"
#include "DwmAuthPeerAuthenticator.hh"
#include "DwmAuthSymCrypto.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmAuth/tags/libDwmAuth-0.1.3/src/DwmAuthPeerAuthenticator.cc 9028 $");

using namespace std;
using CryptoPP::PSS;
using CryptoPP::SHA256;
using CryptoPP::RSASS;
using CryptoPP::StringSource;
using CryptoPP::StringSink;
using CryptoPP::SignerFilter;

namespace Dwm {

  namespace Auth {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static string to_str(const CryptoPP::SecByteBlock & sbb)
    {
      return string((const char *)sbb.BytePtr(), sbb.SizeInBytes());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static CryptoPP::SecByteBlock to_SecByteBlock(const string & s)
    {
      return CryptoPP::SecByteBlock((const byte *)s.c_str(), s.size());
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    PeerAuthenticator::PeerAuthenticator(const string & myPrivKeyPath,
                                         const string & pubKeysPath)
        : _myPrivKeyPath(myPrivKeyPath), _pubKeysPath(pubKeysPath)
    {
      struct passwd  *pw = 0;
      if (_myPrivKeyPath.empty() || _pubKeysPath.empty()) {
        uid_t   uid = getuid();
        pw = getpwuid(uid);
      }
      if (_myPrivKeyPath.empty()) {
        if (pw) {
          _myPrivKeyPath = pw->pw_dir;
          _myPrivKeyPath += "/.dwmauth/id_rsa";
        }
      }
      if (_pubKeysPath.empty()) {
        if (pw) {
          _pubKeysPath = pw->pw_dir;
          _pubKeysPath += "/.dwmauth/known_services";
        }
      }
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PeerAuthenticator::ECExchange(int fd,
                                       Auth::ECDHAgreement & agreement)
    {
      bool  rc = false;
      if (fd >= 0) {
        string  &&mypubEC = to_str(agreement.Public());
        if (IO::Write(fd, mypubEC) == IO::StreamedLength(mypubEC)) {
          string  theirPubECDH;
          if (IO::Read(fd, theirPubECDH) > 0) {
            agreement.TheirPublic(to_SecByteBlock(theirPubECDH));
            CryptoPP::SecByteBlock  agreedKeyBB;
            if (agreement.Agree()) {
              rc = true;
            }
          }
        }
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    string PeerAuthenticator
    ::BuildIDAndSignature(const string & myId,
                          const PKCrypto::PrivateKey & myPrivRSAKey,
                          const ECDHAgreement & agreement)
    {
      string  agreedKeyStr(to_str(agreement.AgreedKey()));
      string  keystr(agreedKeyStr.substr(0, 16));
      string  ivstr(agreedKeyStr.substr(16, 16));
      string  myIdCrypted = SymCrypto::Encrypt(keystr, ivstr, myId);
      string  msgToSign(to_str(agreement.Public()) + myIdCrypted);
      RSASS<PSS,SHA256>::Signer  signer(myPrivRSAKey);
      string  mySignature;
      CryptoPP::AutoSeededRandomPool  rng;
      StringSource ss(msgToSign, true,
                      new SignerFilter(rng, signer,
                                       new StringSink(mySignature)));
      ostringstream  os;
      IO::Write(os, myIdCrypted);
      IO::Write(os, mySignature);
      return os.str();
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PeerAuthenticator
    ::IDAndSigExchange(int fd, const string & myId,
                       const PKCrypto::PrivateKey & myPrivRSAKey,
                       const ECDHAgreement & agreement,
                       string & theirId,
                       string & theirIdCrypted,
                       string & theirSig)
    {
      bool  rc = false;
      if (fd >= 0) {
        string idAndSig = BuildIDAndSignature(myId, myPrivRSAKey, agreement);
        if (IO::Write(fd, idAndSig)) {
          string  s;
          if (IO::Read(fd, s) > 0) {
            string  keystr((to_str(agreement.AgreedKey())).substr(0, 16));
            string  ivstr((to_str(agreement.AgreedKey())).substr(16, 16));
            istringstream  is(s);
            if (IO::Read(is, theirIdCrypted)) {
              theirId = SymCrypto::Decrypt(keystr, ivstr, theirIdCrypted);
              if (IO::Read(is, theirSig)) {
                rc = true;
              }
            }
          }
        }
      }
      return rc;
    }

#if 0
    bool PeerAuthenticator::VerifySignature()
    {
      Auth::PublicKeysFile  pubKeysFile;
      pubKeysFile.Load(_pubKeysPath);
      PublicKeysFile::Map::const_iterator  pki =
        pubKeysFile.Find(theirId);
      if (pki != pubKeysFile.Keys().end()) {
        string  msgToVerify = theirPubECDH + theirIdCrypted;
        RSASS<PSS,SHA256>::Verifier  verifier(pki->second);
        if (verifier.VerifyMessage((const byte *)msgToVerify.c_str(),
                                   msgToVerify.size(),
                                   (const byte *)theirSig.c_str(),
                                   theirSig.size())) {
        }
      }
    }
#endif
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool PeerAuthenticator::Authenticate(int fd, string & theirId,
                                         string & agreedKey)
    {
      bool  rc = false;
      if (fd >= 0) {
        struct sockaddr_in  sockAddr;
        socklen_t           sockLen = sizeof(sockAddr);
        getpeername(fd, (struct sockaddr *)&sockAddr, &sockLen);
        
        Auth::ECDHAgreement  agreement;
        if (ECExchange(fd, agreement)) {
          PKCrypto::KeyPair  myKeys;
          string             myId;
          if (myKeys.Load(_myPrivKeyPath, myId)) {
            string  theirIdCrypted, theirSig;
            if (IDAndSigExchange(fd, myId, myKeys.Private(),
                                 agreement, theirId,
                                 theirIdCrypted, theirSig)) {
              Auth::PublicKeysFile  pubKeysFile;
              pubKeysFile.Load(_pubKeysPath);
              PublicKeysFile::Map::const_iterator  pki =
                pubKeysFile.Find(theirId);
              if (pki != pubKeysFile.Keys().end()) {
                string  msgToVerify =
                  to_str(agreement.TheirPublic()) + theirIdCrypted;
                RSASS<PSS,SHA256>::Verifier  verifier(pki->second);
                if (verifier.VerifyMessage((const byte *)msgToVerify.c_str(),
                                           msgToVerify.size(),
                                           (const byte *)theirSig.c_str(),
                                           theirSig.size())) {
                  agreedKey = to_str(agreement.AgreedKey());
                  rc = true;
                }
                else {
                  Syslog(LOG_ERR, "Failed to verify message from %s",
                         theirId.c_str());
                }
              }
              else {
                Syslog(LOG_ERR, "Failed to find %s in %s",
                       theirId.c_str(), _pubKeysPath.c_str());
              }
            }
            else {
              Syslog(LOG_ERR, "ID and signature exhange with %s failed",
                     inet_ntoa(sockAddr.sin_addr));
            }
          }
          else {
            Syslog(LOG_ERR, "Failed to load RSA key from %s",
                   _myPrivKeyPath.c_str());
          }
        }
        else {
          Syslog(LOG_ERR, "ECDH exhange with %s failed",
                 inet_ntoa(sockAddr.sin_addr));
        }
      }
      return rc;
    }
    

  }  // namespace Auth

}  // namespace Dwm
