//===========================================================================
// @(#) $Name:$
// @(#) $Id: DwmAuthECDHAgreement.hh 9036 2017-04-12 07:08:21Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmAuthECDHAgreement.hh
//!  \brief Dwm::Auth::ECDHAgreement class definition
//---------------------------------------------------------------------------

#ifndef _DWMAUTHECDHAGREEMENT_HH_
#define _DWMAUTHECDHAGREEMENT_HH_

#include <cryptopp/eccrypto.h>

namespace Dwm {

  namespace Auth {

    //------------------------------------------------------------------------
    //!  Encapsulates information used in an ECDH key agreement.  This isn't
    //!  typically used directly but is a helper class for PeerAuthenticator.
    //------------------------------------------------------------------------
    class ECDHAgreement
    {
    public:
      //----------------------------------------------------------------------
      //!  constructor.  This will create our public and private parts of
      //!  our ECDH key.
      //----------------------------------------------------------------------
      ECDHAgreement();

      //----------------------------------------------------------------------
      //!  Returns a const reference to the public half of our ECDH key.
      //----------------------------------------------------------------------
      const CryptoPP::SecByteBlock & Public() const;

      //----------------------------------------------------------------------
      //!  Returns a const reference to the private half of our ECDH key.
      //----------------------------------------------------------------------
      const CryptoPP::SecByteBlock & Private() const;

      //----------------------------------------------------------------------
      //!  Returns a const reference to the other side's public key.  This
      //!  will be empty until it is set with TheirPublic(SecByteBlock &),
      //!  typically after it has been read from a socket.
      //----------------------------------------------------------------------
      const CryptoPP::SecByteBlock & TheirPublic() const;

      //----------------------------------------------------------------------
      //!  Saves the other side's public half of their ECDH key.
      //----------------------------------------------------------------------
      const CryptoPP::SecByteBlock &
      TheirPublic(const CryptoPP::SecByteBlock & theirPub);

      //----------------------------------------------------------------------
      //!  Once we have our private and public keys and the other side's
      //!  public key has been saved with TheirPublic(SecByteBlock &), we
      //!  can call this function.  If we agree on a shared key, this
      //!  member returns true and we can fetch the shared key with
      //!  AgreedKey().
      //----------------------------------------------------------------------
      bool Agree();

      //----------------------------------------------------------------------
      //!  If we stored the public half of the other side's ECDH key with
      //!  TheirPublic(SecByteBlock &) and then called Agree(), we can
      //!  fetch the agreed shared key with this memner.
      //----------------------------------------------------------------------
      const CryptoPP::SecByteBlock & AgreedKey() const;
      
    private:
      CryptoPP::ECDH<CryptoPP::ECP>::Domain  _dh;
      CryptoPP::SecByteBlock                 _mypriv;
      CryptoPP::SecByteBlock                 _mypub;
      CryptoPP::SecByteBlock                 _theirpub;
      CryptoPP::SecByteBlock                 _agreedKey;
    };
    
  }  // namespace Auth

}  // namespace Dwm

#endif  // _DWMAUTHECDHAGREEMENT_HH_
