//===========================================================================
// @(#) $Name:$
// @(#) $Id: dwmauth.cc 9057 2017-04-12 21:25:46Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2017
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file dwmauth.cc
//!  \brief tiny dwmauth key generator and checker
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/types.h>
  #include <fcntl.h>
  #include <limits.h>
  #include <pwd.h>
  #include <unistd.h>
}

#include <fstream>
#include <iostream>

#include "DwmOptArgs.hh"
#include "DwmSvnTag.hh"
#include "DwmAuthPublicKeysFile.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwmAuth/tags/libDwmAuth-0.1.3/apps/dwmauth/dwmauth.cc 9057 $");

using namespace std;
using namespace Dwm;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static string GetDefaultUserName()
{
  string  username;
  uid_t   uid = getuid();
  struct passwd  *pw = getpwuid(uid);
  if (pw) {
    username = pw->pw_name;
    char   hn[255];
    if (gethostname(hn, sizeof(hn)) == 0) {
      username += '@';
      username += hn;
    }
  }
  return username;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static string GetDefaultDirectory()
{
  string  dirName;
  uid_t   uid = getuid();
  struct passwd  *pw = getpwuid(uid);
  if (pw) {
    dirName = pw->pw_dir;
    dirName += "/.dwmauth";
  }
  return dirName;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void Usage(const char *argv0)
{
  cerr << "Usage: " << argv0 << " keygen [-C comment] [-f private_keyfile]\n"
       << "       " << argv0 << " keycheck [-f private_keyfile]\n";
  return;
}

#ifdef __linux__
//----------------------------------------------------------------------------
//!  Linux can't acquire lock on open(), so we need to explicitly lock
//!  after open().  We'll use fcntl().
//----------------------------------------------------------------------------
static bool WriteLockFile(int fd)
{
  bool  rc = false;
  if (fd >= 0) {
    struct flock  lock = {
      .l_start = SEEK_SET,
      .l_len = 0,
      .l_type = F_WRLCK,
      .l_whence = SEEK_SET,
    };
    if (fcntl(fd, F_SETLKW, &lock) != -1) {
      rc = true;
    }
  }
  return rc;
}
#endif

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool GenerateKeys(int argc, char *argv[])
{
  OptArgs  optargs;
  optargs.AddOptArg("C:", "comment", false, "", "comment");
  optargs.AddOptArg("f:", "keyfile", false, "", "output keyfile");
  int  nextArg = optargs.Parse(argc, argv);
  
  bool  rc = false;
  string  un(optargs.Get<string>('C'));
  if (un.empty()) {
    un = GetDefaultUserName();
  }
  string  privFile(optargs.Get<string>('f'));
  if (privFile.empty()) {
    privFile = GetDefaultDirectory() + "/id_rsa";
  }
  
  Auth::PKCrypto::KeyPair  &&kp = Auth::PKCrypto::CreateKeyPair(2048);
#ifndef __linux__
  int  fd = open(privFile.c_str(), O_WRONLY|O_CREAT|O_TRUNC|O_EXLOCK, 0600);
#else
  int  fd = open(privFile.c_str(), O_WRONLY|O_CREAT|O_TRUNC, 0600);
  if (fd >= 0) {
    if (! WriteLockFile(fd)) {
      close(fd);
      fd = -1;
    }
  }
#endif
  if (fd >= 0) {
    if (kp.Private().Write(fd) > 0) {
      close(fd);
      string  pubFile(privFile + ".pub");
      ofstream  os(pubFile.c_str());
      if (os) {
        string  &&pub64 = kp.Public().ToBase64String();
        if (os << un << ' ' << pub64 << '\n') {
          rc = true;
        }
        os.close();
      }
      else {
        cerr << "Failed to open " << pubFile << ": " << strerror(errno)
             << '\n';
      }
    }
  }
  else {
    cerr << "Failed to open " << privFile << ": " << strerror(errno) << '\n';
  }
  
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool CheckKeys(int argc, char *argv[])
{
  OptArgs  optargs;
  optargs.AddOptArg("f:", "keyfile", false, "", "input keyfile");
  int  nextArg = optargs.Parse(argc, argv);
  
  bool  rc = false;
  string  privFile(optargs.Get<string>('f'));
  if (privFile.empty()) {
    privFile = GetDefaultDirectory() + "/id_rsa";
  }
  
  Auth::PKCrypto::KeyPair  kp;
  int  fd = open(privFile.c_str(), O_RDONLY);
  if (fd >= 0) {
    if (kp.Private().Read(fd) > 0) {
      close(fd);
      if (kp.Private().IsValid()) {
        string  pubFile(privFile + ".pub");
        Auth::PublicKeysFile  ak;
        if (ak.Load(pubFile)) {
          if (ak.Keys().size() == 1) {
            kp.second = ak.Keys().begin()->second;
            if (kp.Public().IsValid()) {
              string  es = Auth::PKCrypto::Encrypt(kp.Public(),
                                                   "SomeTextToEncrypt");
              string  ds = Auth::PKCrypto::Decrypt(kp.Private(), es);
              if (ds == "SomeTextToEncrypt") {
                rc = true;
              }
              else {
                cerr << "Encrypt/decrypt failed!\n";
              }
            }
            else {
              cerr << "Public key for " << ak.Keys().begin()->first
                   << " not valid!\n";
            }
          }
          else {
            cerr << "No valid public keys in " << pubFile << '\n';
          }
        }
        else {
          cerr << "Failed to load public key from " << pubFile << '\n';
        }
      }
      else {
        cerr << "Private key in " << privFile << " is invalid!\n";
      }
    }
    else {
      cerr << "Failed to read private key from " << privFile << '\n';
    }
  }
  else {
    cerr << "Failed to open " << privFile << ": " << strerror(errno) << '\n';
  }
  
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  bool  rc = false;
  
  if (argc > 1) {
    if (string(argv[1]) == "keygen") {
      rc = GenerateKeys(argc-1, &argv[1]);
    }
    else if (string(argv[1]) == "keycheck") {
      rc = CheckKeys(argc-1, &argv[1]);
    }
    else {
      Usage(argv[0]);
      rc = false;
    }
  }
  else {
    Usage(argv[0]);
  }
  
  return (rc ? 0 : 1);
}
