%code requires
{
  #include <iostream>
  #include <string>
  #include <utility>
  #include <set>

  #include "DwmDebControl.hh"
  #include "DwmDebPkgDepend.hh"
  #include "DwmDebPkgVersion.hh"
    
  using std::pair, std::string, std::set;
}

%{
  #include <cstdio>
  #include <cstdlib>
    
  extern "C" {
    extern void dwmdebctrlerror(const char *arg, ...);
    extern FILE *dwmdebctrlin;
  }

  #include <mutex>
  #include <string>

  #include "DwmSvnTag.hh"
  #include "DwmDebControl.hh"
  #include "DwmDebPkgDepend.hh"

  static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwm/tags/libDwm-0.9.4/apps/mkdebcontrol/DwmDebControlParser.y 11491 $");
  
  using namespace std;
  
  string            g_debPath;
  Dwm::DebControl  *g_debctrl = nullptr;
  static mutex      g_mtx;
  
%}

%define api.prefix {dwmdebctrl}

%union {
  string                  *stringVal;
  pair<string,string>     *stringPairVal;
  Dwm::DebPkgDepend       *dwmDebPkgDependVal;
  Dwm::DebPkgVersion      *dwmDebPkgVersionVal;
  set<Dwm::DebPkgDepend>  *dwmDebPkgDependSetVal;
}

%code provides
{
  // Tell Flex the expected prototype of yylex.
  #define YY_DECL                             \
    int dwmdebctrllex ()

  // Declare the scanner.
  YY_DECL;
}

%token DEPENDS PREDEPENDS
%token<stringVal> FIELDNAME STRING UNKNOWNFIELDNAME

%type<stringVal>             Values VersionOperator
%type<stringPairVal>         Field
%type<dwmDebPkgVersionVal>   PackageVersion
%type<dwmDebPkgDependVal>    DependPackage
%type<dwmDebPkgDependSetVal> Depends PreDepends DependPackages

%%

Fields: Field
{
  g_debctrl->Add(*$1);
  //  std::cout << $1->first << " " << $1->second  << '\n';
  delete $1;
}
| Depends
{
  g_debctrl->AddDepends(*$1);
  delete $1;
}
| PreDepends
{
  g_debctrl->AddDepends(*$1);
  delete $1;
}
| Fields Field
{
  g_debctrl->Add(*$2);
  //  std::cout << $2->first << " " << $2->second << '\n';
  delete $2;
}
| Fields Depends
{
  g_debctrl->AddDepends(*$2);
  delete $2;
}
| Fields PreDepends
{
  g_debctrl->AddPreDepends(*$2);
  delete $2;
};

Field: FIELDNAME Values
{
  $$ = new std::pair<std::string,std::string>(*$1,*$2);
  delete $1;
  delete $2;
}
| UNKNOWNFIELDNAME Values
{
  $$ = new std::pair<std::string,std::string>(*$1,*$2);
  delete $1;
  delete $2;
};

Values: STRING
{
  $$ = new std::string();
  (*$$) = *$1;
  delete $1;
}
| Values STRING
{
  (*$$) += *$2;
  delete $2;
};

Depends: DEPENDS DependPackages
{
  $$ = $2;
};

PreDepends: PREDEPENDS DependPackages
{
  $$ = $2;
};

DependPackages: DependPackage
{
  $$ = new std::set<Dwm::DebPkgDepend>();
  $$->insert(*$1);
  delete $1;
}
| DependPackages ',' DependPackage
{
  $$->insert(*$3);
  delete $3;
};

DependPackage: STRING
{
  $$ = new Dwm::DebPkgDepend(*$1);
  delete $1;
}
| STRING '(' VersionOperator PackageVersion ')'
{
    $$ = new Dwm::DebPkgDepend(*$1, *$3, *$4);
    delete $1;
    delete $3;
    delete $4;
};

VersionOperator: '>' '>'
{
  $$ = new std::string(">>");
}
| '<' '<'
{
  $$ = new std::string("<<");
}
| '>' '='
{
  $$ = new std::string(">=");
}
| '<' '='
{
  $$ = new std::string("<=");
}
| '='
{
  $$ = new std::string("=");
};

PackageVersion: STRING ':' STRING
{
  $$ = new Dwm::DebPkgVersion(atoi($1->c_str()), *$3);
  delete $1;
  delete $3;
}
| STRING
{
  $$ = new Dwm::DebPkgVersion(*$1);
  delete $1;
};

%%

namespace Dwm {

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  DebControl::DebControl()
      : _entries(), _predepends(), _depends()
  {}
  
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  bool DebControl::Parse(const string & path)
  {
    lock_guard<mutex>  lck(g_mtx);
      
    bool  rc = false;
    dwmdebctrlin = fopen(path.c_str(), "r");
    if (dwmdebctrlin) {
      g_debPath = path;
      g_debctrl = this;
      rc = (0 == dwmdebctrlparse());
      fclose(dwmdebctrlin);
    }
    return rc;
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  bool DebControl::HasRequiredEntries() const
  {
    static const vector<string>  requiredFields = {
      "Package:",
      "Version:",
      "Architecture:",
      "Maintainer:",
      "Description:"
    };
    bool  rc = true;
    for (const auto & rf : requiredFields) {
      if (_entries.find(rf) == _entries.end()) {
        rc = false;
        break;
      }
    }
    return rc;
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  void DebControl::Add(const std::pair<std::string,std::string> & entry)
  {
    _entries[entry.first] = entry.second;
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  void DebControl::RemoveDepend(const DebPkgDepend & dep)
  {
    _depends.erase(dep);
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  void DebControl::RemovePreDepend(const DebPkgDepend & dep)
  {
    _predepends.erase(dep);
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  void DebControl::AddDepend(const DebPkgDepend & dep)
  {
    _depends.insert(dep);
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  void DebControl::AddPreDepend(const DebPkgDepend & dep)
  {
    _predepends.insert(dep);
  }
  
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  void DebControl::AddDepends(const std::set<DebPkgDepend> & depends)
  {
    for (const auto & d : depends) {
      _depends.insert(d);
    }
    return;
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  void DebControl::AddPreDepends(const std::set<DebPkgDepend> & depends)
  {
    for (const auto & d : depends) {
      _predepends.insert(d);
    }
    return;
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  std::ostream & operator << (std::ostream & os,
                              const DebControl & debctrl)
  {
    for (const auto & e : debctrl._entries) {
      os << e.first << ' ' << e.second << '\n';
    }
    if (! debctrl._predepends.empty()) {
      os << "Pre-Depends: ";
      string sep;
      for (const auto & predep : debctrl._predepends) {
        os << sep << predep;
        sep = ", ";
      }
      os << '\n';
    }

    if (! debctrl._depends.empty()) {
      os << "Depends: ";
      string sep;
      for (const auto & dep : debctrl._depends) {
        os << sep << dep;
        sep = ", ";
      }
      os << '\n';
    }
    return os;
  }
  
}  // namespace Dwm
