//===========================================================================
// @(#) $DwmPath: dwm/libDwm/tags/libDwm-0.8.8/apps/dwmspath/dwmspath.cc 9859 $
// @(#) $Id: dwmspath.cc 9859 2017-12-19 09:09:55Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file dwmspath.cc
//!  \brief Simple application to sanitize path(s) given on command line by
//!  removing redundancy ('/./' and '//') and replacing '../'.  Will yield
//!  an absolute path.  This is much like realpath().  One significant
//!  difference: I don't require the path to exist and don't look at the
//!  filesystem at all.  It's purely string manipulations.
//---------------------------------------------------------------------------

extern "C" {
  #include <unistd.h>
}

#include <algorithm>
#include <cstdlib>
#include <iostream>
#include <iterator>
#include <regex>

#include "DwmOptArgs.hh"
#include "DwmSvnTag.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwm/tags/libDwm-0.8.8/apps/dwmspath/dwmspath.cc 9859 $");

using namespace std;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static string ReplaceRedundant(const string & s)
{
  string rc = s;
  regex  rgx("(//|/\\./|/\\.$)");
  while (regex_search(rc, rgx) && (rc.length() > 1)) {
    rc = regex_replace(rc, rgx, "/");
  }
  rgx = "(^\\./|/\\.$|/$)";
  while (regex_search(rc, rgx) && (rc.length() > 1)) {
    rc = regex_replace(rc, rgx, "");
  }
  
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void ReplaceDotDots(string & s)
{
  regex   rgx("([^/]+/)\\.\\.[/]{0,1}");
  smatch  m;
  while (regex_search(s, m, rgx)) {
    if ((m.size() > 0) && m[0] != "../..") {
      string  ms = m[0];
      string::size_type  idx = s.find(ms);
      s.replace(idx, ms.length(), "");
    }
    else {
      break;
    }
  }
  while (s.substr(0,3) == "/..") {
    s = s.substr(3);
  }
  if (s.empty()) {
    s = "/";
  }
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void PrependCwd(string & s)
{
  char  buf[PATH_MAX];
  if (getcwd(buf, PATH_MAX) == buf) {
    s = string(buf) + "/" + s;
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void SanitizePath(string & s)
{
  if (s[0] != '/') {
    PrependCwd(s);
  }
  if (s.length() > 1) {
    s = ReplaceRedundant(s);
  }
  ReplaceDotDots(s);
  if (s.length() > 1) {
    s = ReplaceRedundant(s);
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  Dwm::OptArgs  optargs;
  optargs.AddOptArg("r", "realpath", false, "0", "Use realpath()");
  optargs.AddNormalArg("path(s)", true);
  int  nextArg = optargs.Parse(argc, argv);

  if (nextArg == argc) {
    optargs.Usage(argv[0]);
    return 1;
  }

  for (int arg = nextArg; arg < argc; ++arg) {
    if (optargs.Get<bool>('r')) {
      char  *rc = realpath(argv[arg], 0);
      if (rc) {
        cout << rc << '\n';
        free(rc);
        rc = 0;
      }
    }
    else {
      string  s = argv[arg];
      SanitizePath(s);
      cout << s << '\n';
    }
  }
  return 0;
}
