//===========================================================================
// @(#) $DwmPath: dwm/libDwm/tags/libDwm-0.8.1/apps/mkfbsdmnfst/mkfbsdmnfst.cc 10190 $
// @(#) $Id: mkfbsdmnfst.cc 10190 2019-11-24 00:42:35Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file mkfbsdmnfst.cc
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

//  I can set:
//    name
//    version
//    origin
//    comment
//    desc
//    arch
//    www
//    maintainer
//    prefix
//    files
//
//  I may need:
//    conflict
//    flatsize
//    deps

extern "C" {
  #include <fcntl.h>
  #include <fts.h>
  #include <libgen.h>
  #include <openssl/sha.h>
  #include <sys/types.h>
  #include <sys/stat.h>
  #include <sys/utsname.h>
  #include <unistd.h>
  #include <sqlite3.h>
}
#include <cstdlib>
#include <fstream>
#include <iomanip>
#include <iostream>
#include <map>
#include <regex>
#include <set>
#include <sstream>
#include <string>
#include <vector>

#include "DwmFreeBSDPkgManifest.hh"
#include "DwmOptArgs.hh"
#include "DwmSvnTag.hh"

using namespace std;
using Dwm::FreeBSDPkg::Manifest;

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwm/tags/libDwm-0.8.1/apps/mkfbsdmnfst/mkfbsdmnfst.cc 10190 $");

static Dwm::OptArgs  g_optargs;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void GetSharedLibs(const string & filename, set<string> & libs)
{
  string  lddcmd("ldd " + filename);
  FILE    *lddpipe = popen(lddcmd.c_str(), "r");
  if (lddpipe) {
    regex   rgx(".+[ \\t]+[=][>][ \\t]+([^ \\t]+)[ \\t]+",
                regex::ECMAScript|regex::optimize);
    smatch  sm;
    char    line[4096];
    while (fgets(line, 4096, lddpipe) != NULL) {
      string  s(line);
      if (regex_search(s, sm, rgx)) {
        if (sm.size() == 2) {
          string  lib(sm[1].str());
          libs.insert(lib);
        }
      }
    }
    pclose(lddpipe);
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void GetPackageDeps(const set<string> & libs, set<string> & packages)
{
  string  queryPreamble("select packages.name, packages.version, packages.id,"
                        " files.package_id, files.path"
                        " from packages, files where"
                        " packages.id = files.package_id"
                        " and files.path = ");
  sqlite3  *ppdb;
  if (sqlite3_open_v2("/var/db/pkg/local.sqlite", &ppdb,
                      SQLITE_OPEN_READONLY, 0)
      == SQLITE_OK) {
    for (auto lib : libs) {
      string  qrystr(queryPreamble + '"' + lib + '"');
      sqlite3_stmt *ppStmt;
      if (sqlite3_prepare(ppdb, qrystr.c_str(), -1, &ppStmt, 0) == SQLITE_OK) {
        while (sqlite3_step(ppStmt) == SQLITE_ROW) {
          string  pkgName((const char *)sqlite3_column_text(ppStmt, 0));
          pkgName += '-';
          pkgName += (const char *)sqlite3_column_text(ppStmt, 1);
          packages.insert(pkgName);
        }
      }
      sqlite3_finalize(ppStmt);
    }
    sqlite3_close_v2(ppdb);
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static void GetPackageInfo(const set<string> & pkgs,
                           vector<Manifest::Dependency> & pkginfo)
{
  regex  rgx("([^ \\t]+)[ \\t]+([^ \\t]+)[ \\t]+([^ \\t]+)\\n",
             regex::ECMAScript|regex::optimize);
  smatch  sm;
  for (auto pkg : pkgs) {
    string  querycmd("pkg query \"%n %o %v\" " + pkg);
    FILE    *querypipe = popen(querycmd.c_str(), "r");
    if (querypipe) {
      char    line[4096];
      if (fgets(line, 4096, querypipe) != NULL) {
        string  s(line);
        if (regex_search(s, sm, rgx)) {
          if (sm.size() == 4) {
            Manifest::Dependency  dep(sm[1].str(), sm[2].str(), sm[3].str());
            if (find_if(pkginfo.begin(), pkginfo.end(),
                        [&] (Manifest::Dependency const & item)
                        { return dep.Name() == item.Name(); })
                == pkginfo.end()) {
              pkginfo.push_back(dep);
            }
          }
        }
      }
      pclose(querypipe);
    }
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
vector<string> GetFiles(const string & dirName)
{
  regex              excludeRegex("^[/][#]*\\+(DESC|DISPLAY|MANIFEST|PRE_DEINSTALL|POST_DEINSTALL|PRE_INSTALL|POST_INSTALL)[~#]+");
  vector<string>     filenames;
  string             filename;
  string::size_type  idx;
  char  *dirs[2] = { strdup(dirName.c_str()), 0 };
  FTS  *fts = fts_open(&dirs[0], FTS_PHYSICAL|FTS_NOCHDIR, 0);
  if (fts) {
    FTSENT  *ftsent;
    while ((ftsent = fts_read(fts))) {
      switch (ftsent->fts_info) {
        case FTS_F:
        case FTS_SL:
          filename = ftsent->fts_path;
          idx = filename.find(dirName);
          if (idx == 0) {
            filename = filename.substr(dirName.length());
          }
          if (filename.front() != '/') {
            filename = "/" + filename;
          }
          if (! regex_match(filename, excludeRegex)) {
            filenames.push_back(filename);
          }
          break;
        default:
          break;
      }
    }
    fts_close(fts);
  }
  free(dirs[0]);
  return filenames;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static string GetSHA256(const string & filename)
{
  SHA_CTX       ctx;
  unsigned char md[SHA_DIGEST_LENGTH];
  int fd = open(filename.c_str(), O_RDONLY);
  if (fd >= 0) {
    SHA1_Init(&ctx);
    uint8_t  buf[65536];
    ssize_t  bytesRead;
    while ((bytesRead = read(fd, buf, 65536)) > 0) {
      SHA1_Update(&ctx, buf, bytesRead);
    }
    close(fd);
    SHA1_Final(&(md[0]), &ctx);
  }

  ostringstream  os;
  os << setfill('0') << hex;
  for (int i = 0; i < SHA_DIGEST_LENGTH; ++i) {
    os << setw(2) << (uint16_t)md[i];
  }
  return os.str();
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
vector<Manifest::File> GetManifestFiles(const string & dirName)
{
  vector<Manifest::File>  rc;
  vector<string>  filenames = GetFiles(dirName);
  for (auto f : filenames) {
    Manifest::File  mf(f /*, GetSHA256(dirName + f) */);
    rc.push_back(mf);
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static string GetArchName()
{
  string  rc;
  struct utsname  utsn;
  
  if (uname(&utsn) == 0) {
    rc = utsn.machine;
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static string EscapeNewlinesAndQuotes(const string & s)
{
  string  rc;
  regex   rgx("\\\\");
  rc = regex_replace(s, rgx, "\\\\");
  rgx = "\"";
  rc = regex_replace(rc, rgx, "\\\"");
  rgx ="\\n";
  rc = regex_replace(rc, rgx, "\\n");
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
static string GetEscapedFileContents(const string & path)
{
  string    rc;
  ifstream  is(path.c_str());
  if (is) {
    rc = string((istreambuf_iterator<char>(is)), istreambuf_iterator<char>());
    rc = EscapeNewlinesAndQuotes(rc);
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  Special files are those which we won't include in the manifest files
//!  list, but will use to populate other fields in the manifest.
//----------------------------------------------------------------------------
static bool HandleSpecialFile(const string & dirName, Manifest & manifest,
                              const Manifest::File & mf)
{
  bool  rc = false;
  typedef const string & (Manifest::*FieldSetFn)(const string & value);
  static const map<string,FieldSetFn>  fieldSetters = {
    { "/+DESC",           &Manifest::Description },
    { "/+PRE_INSTALL",    &Manifest::PreInstall },
    { "/+POST_INSTALL",   &Manifest::PostInstall },
    { "/+PRE_DEINSTALL",  &Manifest::PreDeinstall },
    { "/+POST_DEINSTALL", &Manifest::PostDeinstall }
  };
  auto  fs = fieldSetters.find(mf.Path());
  if (fs != fieldSetters.end()) {
    (manifest.*(fs->second))(GetEscapedFileContents(dirName + mf.Path()));
    rc = true;
  }
  else if (mf.Path() == "/+MANIFEST") {
    rc = true;
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  Check for either mismatched package dependencies or missing dependencies
//!  by looking for shared libraries in files in bin and sbin directories.
//!  Patch up the dependencies if the version is mismatched, add them if
//!  they're missing.
//----------------------------------------------------------------------------
static void CheckPackageDependencies(const string & dirName,
                                     Manifest & manifest)
{
  bool    rc = true;
  regex   rgx("/bin/|/sbin/|/lib/lib.+\\.so\\.",
              regex::ECMAScript|regex::optimize);
  smatch  sm;

  set<string>  sharedLibs;
  for (auto file : manifest.Files()) {
    if (regex_search(file.Path(), sm, rgx)) {
      string stagingPath = dirName + file.Path();
      GetSharedLibs(stagingPath, sharedLibs);
    }
  }
  if (! sharedLibs.empty()) {
    set<string>  packageDeps;
    GetPackageDeps(sharedLibs, packageDeps);
    if (! packageDeps.empty()) {
      vector<Manifest::Dependency>  dependencies;
      GetPackageInfo(packageDeps, dependencies);
      if (! dependencies.empty()) {
        for (auto dep : dependencies) {
          auto  it = find_if(manifest.Dependencies().begin(),
                      manifest.Dependencies().end(),
                      [&] (const Manifest::Dependency & mdep) 
                      {
                        return ((dep.Name() == mdep.Name())
                                && ((dep.Version() != mdep.Version())
                                    || (dep.Origin() != mdep.Origin())));
                      });
          if (it != manifest.Dependencies().end()) {
            if (dep.Version() != it->Version()) {
              cerr << "Dependency version mismatch, " << dep.Name()
                   << " version corrected to " << dep.Version() << '\n';
            }
            if (dep.Origin() != it->Origin()) {
              cerr << "Dependency origin mismatch, " << dep.Name()
                   << " origin corrected to " << dep.Origin() << '\n';
            }
            *it = dep;
          }
          if (find_if(manifest.Dependencies().begin(),
                      manifest.Dependencies().end(),
                      [&] (const Manifest::Dependency & mdep) 
                      {
                        return (dep.Name() == mdep.Name());
                      })
              == manifest.Dependencies().end()) {
            cerr << "Added dependency " << dep.Name()
                 << " version " << dep.Version() << '\n';
            manifest.Dependencies().push_back(dep);
          }
        }
      }
    }
  }
  return;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
bool PopulateManifest(const string & dirName, Manifest & manifest)
{
  //  All of the fields I want to set in a Manifest object can be set
  //  with a member function with the same signature.  So I can use a
  //  map of command line options to member functions in order to set
  //  the fields.
  typedef const string & (Manifest::*FieldSetFn)(const string & value);
  static const map<char,FieldSetFn>  fieldSetters = {
    { 'n', &Manifest::Name },
    { 'v', &Manifest::Version },
    { 'o', &Manifest::Origin },
    { 'c', &Manifest::Comment },
    { 'd', &Manifest::Description },
    { 'w', &Manifest::WWW },
    { 'm', &Manifest::Maintainer },
    { 'p', &Manifest::Prefix }
  };
  
  bool  rc = false;
  vector<Manifest::File>  manifestFiles = GetManifestFiles(dirName);
  if (! manifestFiles.empty()) {
    for (auto it : fieldSetters) {
      if (! g_optargs.Get<string>(it.first).empty()) {
        //  Weird syntax is due to calling a pointer to member function.
        //  (manifest.*(it.second)) is the pointer to member function.
        (manifest.*(it.second))(g_optargs.Get<string>(it.first));
      }
    }
    set<string> sharedLibDeps;
    
    for (auto mfit : manifestFiles) {
      //  Only add files that are not already in the manifest.
      if (find_if(manifest.Files().begin(), manifest.Files().end(),
                  [&mfit](Manifest::File const & item)
                  { return item.Path() == mfit.Path(); })
          == manifest.Files().end()) {
        if (! HandleSpecialFile(dirName, manifest, mfit)) {
          mfit.Group(g_optargs.Get<string>('g'));
          mfit.User(g_optargs.Get<string>('u'));
          manifest.Files().push_back(mfit);
        }
      }
    }
    if ((! manifest.Files().empty())
        && (! manifest.Name().empty())
        && (! manifest.Version().empty())) {
      rc = true;
    }
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  g_optargs.AddOptArg("n:", "name", false, "",
                      "name of package (e.g. 'libDwm')");
  g_optargs.AddOptArg("v:", "version", false, "", "version (e.g. '1.5.2')");
  g_optargs.AddOptArg("o:", "origin", false, "",
                      "origin (e.g. 'devel/libDwm')");
  g_optargs.AddOptArg("c:", "comment", false, "", "comment");
  g_optargs.AddOptArg("d:", "desc", false, "", "description");
  g_optargs.AddOptArg("g:", "group", false, "wheel",
                      "group ID of files (default is 'wheel')");
  g_optargs.AddOptArg("w:", "www", false, "", "software's official web site");
  g_optargs.AddOptArg("m:", "maintainer", false, "",
                      "maintainer's email address");
  g_optargs.AddOptArg("p:", "prefix", false, "",
                      "path where files will be installed");
  g_optargs.AddOptArg("r:", "read", false, "", "read manifest file first");
  g_optargs.AddOptArg("u:", "user", false, "root",
                      "owner of files (default is 'root')");
  g_optargs.AddNormalArg("directory", true);
  
  int nextArg = g_optargs.Parse(argc, argv);
  if (nextArg > (argc - 1)) {
    g_optargs.Usage(argv[0]);
    return 1;
  }

  Dwm::FreeBSDPkg::Manifest  manifest;

  if (! g_optargs.Get<string>('r').empty()) {
    manifest.Parse(g_optargs.Get<string>('r').c_str());
  }
  
  struct stat  statbuf;
  if (stat(argv[nextArg], &statbuf) == 0) {
    if (statbuf.st_mode & S_IFDIR) {
      //  Add files from directory argv[nextArg] to the manifest.
      if (PopulateManifest(argv[nextArg], manifest)) {
        //  check for missing/mismatched dependencies.
        CheckPackageDependencies(argv[nextArg], manifest);
        //  Check for missing files.
        vector<Manifest::File>  missingFiles =
          manifest.MissingFiles(argv[nextArg]);
        if (missingFiles.empty()) {
          //  No missing files.  Emit the manifest.
          cout << manifest;
          return 0;
        }
        else {
          cerr << "Missing files:\n";
          for (auto mfit : missingFiles) {
            cerr << "  " << mfit.Path() << '\n';
          }
          return 1;
        }
      }
    }
    else {
      cerr << argv[nextArg] << " is not a directory!\n";
      return 1;
    }
  }
  else {
    cerr << "Failed to stat " << argv[nextArg] << ": " << strerror(errno)
         << '\n';
    return 1;
  }

}

