//===========================================================================
// @(#) $DwmPath: dwm/libDwm/tags/libDwm-0.7.1/apps/mcping/mcping.cc 9488 $
// @(#) $Id: mcping.cc 9488 2017-06-09 22:40:32Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file mcping2.cc
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

#include "DwmHostPinger.hh"
#include "DwmLocalInterfaces.hh"
#include "DwmOptArgs.hh"
#include "DwmSignal.hh"
#include "DwmSvnTag.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwm/tags/libDwm-0.7.1/apps/mcping/mcping.cc 9488 $");

static Dwm::OptArgs      g_optargs;
static uint64_t          g_responses = 0;
static Dwm::HostPinger  *g_pinger = nullptr;
static Dwm::Signal       g_sigInt(SIGINT);

using namespace std;

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
void HandleSigInt(int signum)
{
  cerr << "Stopping sender..." << flush;
  g_pinger->StopSender();
  cerr << " sender stopped\n";
  g_sigInt.PopHandler();
}

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
string GetLikelyInterface()
{
  string  rc;
  map<string,Dwm::LocalInterface>  interfaces;
  GetLocalInterfaces(interfaces);
  for (auto iface : interfaces) {
    if (iface.second.IsUp()
        && iface.second.IsRunning()
        && iface.second.Addr() != Dwm::Ipv4Address("127.0.0.1")
        && iface.second.Addr() != Dwm::Ipv4Address("255.255.255.255")
        && iface.second.Addr() != Dwm::Ipv4Address("0.0.0.0")) {
      rc = iface.second.Name();
      break;
    }
  }
  return rc;
}

//----------------------------------------------------------------------------
//!  Implementation of the HostPinger::Receiver interface.  Simply prints
//!  round trip times of received packets.
//----------------------------------------------------------------------------
class PingReceiver
  : public Dwm::HostPinger::Receiver
{
public:
  PingReceiver(bool quiet)
      : _quiet(quiet)
  {}
  
  void operator () (const Dwm::PingDestination & destination,
                    const Dwm::TimeValue & recvTime,
                    const Dwm::TimeValue & rtt)
  {
    if (rtt.Secs() != 0xFFFFFFFF) {
      ++g_responses;
      if (! _quiet) {
        if ((! destination.Hostname().empty()) && (! g_optargs.Get<bool>('n'))) {
          cout << destination.Hostname();
        }
        else {
          cout << destination.Address();
        }
        if (destination.Port()) {
          cout << ":" << destination.Port();
        }
        cout << " " << setw(6)
             << (double)(rtt.Secs() * 1000.0) + (rtt.Usecs() / 1000.0)
             << " ms\n";
        cout.flush();
      }
    }
  }
private:
  bool _quiet;
};

//----------------------------------------------------------------------------
//!  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  int  exitCode = 1;
  
  g_optargs.AddOptArg("c:", "count", false, "0", "count");
  g_optargs.AddOptArg("i:", "interface", false, "",
                      "network interface on which to listen");
  g_optargs.AddOptArg("n", "nonames", false, "0", "don't show host names");
  g_optargs.AddOptArg("p:", "pps", false, "1", "packet rate (packets/sec)");
  g_optargs.AddOptArg("q", "quiet", false, "false", "quiet");
  g_optargs.AddOptArg("t:", "timeout", false, "5", "timeout (seconds)");
  g_optargs.AddNormalArg("host(s)", true);
  
  int  nextArg = g_optargs.Parse(argc, argv);
  if (argc <= nextArg) {
    g_optargs.Usage(argv[0]);
    exit(1);
  }

  string  interface = g_optargs.Get<string>('i');
  if (interface.empty()) {
    interface = GetLikelyInterface();
  }

  Dwm::PingSequencesInFlight::TimeoutSecs(g_optargs.Get<uint32_t>('t'));
  
  g_sigInt.PushHandler(HandleSigInt);

  vector<Dwm::PingDestination>  destinations;
  Dwm::HostPinger  pinger(getpid() & 0xFFFF, interface);
  for (int arg = nextArg; arg < argc; ++arg) {
    Dwm::PingDestination  dst(argv[arg]);
    destinations.push_back(dst);
    pinger.AddDestination(dst);
  }
  pinger.PacketRate(g_optargs.Get<int>('p'));
  pinger.SetReceiver(new PingReceiver(g_optargs.Get<bool>('q')));
  g_pinger = &pinger;
  pinger.Start(g_optargs.Get<int>('c'));
  
  if (g_optargs.Get<int>('c')) {
    while (! pinger.SenderDone()) {
      usleep(250000);
    }
    if (pinger.Outstanding()) {
      cerr << "Waiting up to "
           << Dwm::PingSequencesInFlight::TimeoutSecs()
           << " second"
           << ((Dwm::PingSequencesInFlight::TimeoutSecs() > 1) ? "s" : "")
           << " for outstanding packets...\n";
      while (pinger.Outstanding()) {
        usleep(250000);
      }
    }
    pinger.Stop();
  }
  else {
    g_sigInt.Wait();
    pinger.StopSender();
    if (pinger.Outstanding()) {
      cerr << "Waiting up to "
           << Dwm::PingSequencesInFlight::TimeoutSecs()
           << " second"
           << ((Dwm::PingSequencesInFlight::TimeoutSecs() > 1) ? "s" : "")
           << " for outstanding packets...\n";
      while (pinger.Outstanding()) {
        usleep(250000);
      }
    }
    pinger.Stop();
  }

  for (int i = 0; i < destinations.size(); ++i) {
    cout << endl
         << "--- " << destinations[i];
    string  hn = destinations[i].Hostname();
    if (! hn.empty()) {
      cout << " (" << hn << ')';
    }
    cout << " ping statistics ---"
         << endl
         << pinger.PacketsSent(destinations[i]) 
         << " packets sent, " 
         << pinger.PacketsReceived(destinations[i]) 
         << " packets received, "
         << pinger.PacketLoss(destinations[i])
         << "% packet loss"
         << endl;
    if (pinger.PacketsReceived(destinations[i]) > 0)
      exitCode = 0;
  }
  return exitCode;
}
