//===========================================================================
// @(#) $Name$
// @(#) $Id: DwmFileLogger.cc 8388 2016-04-17 03:23:45Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2007,2015
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmFileLogger.cc
//!  \brief NOT YET DOCUMENTED
//---------------------------------------------------------------------------

extern "C" {
  #include <sys/param.h>
  #include <sys/stat.h>
  #include <stdarg.h>
  #include <unistd.h>
}

#include <cstdlib>
#include <cstring>
#include <sstream>

#include "DwmSvnTag.hh"
#include "DwmFileLogger.hh"
#include "DwmPathUtils.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwm/tags/libDwm-0.6.8/src/DwmFileLogger.cc 8388 $");

using namespace std;

namespace Dwm {
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  FileLogger::FileLogger()
      : _isOpen(false), _logOptions(0), _minimumPriority(LOG_DEBUG),
        _ident(), _showPriorities(false), _showFunction(false), 
        _showFileLocation(false), _hostName(), _filePrefix(), _file(0),
        _rolloverSize(k_defaultRolloverSize), 
        _numRolloverFiles(k_defaultNumRolloverFiles),
        _currentFilePosition(0), _bufferingType(e_notBuffered),
        _priorityTags(), _priorityNames(), _mutex()
  {
  }

  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  FileLogger::~FileLogger()
  {
    this->Close();
  }

  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  bool FileLogger::Lock() const
  {
    _mutex.lock();
    return(true);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  bool FileLogger::Unlock() const
  {
    _mutex.unlock();
    return(true);
  }

  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  bool FileLogger::Open(const string & ident, int logopt,
                        const string & filename,
                        BufferingType bufferingType)
  {
    this->Lock();
    InitializePriorities();
    bool  rc = this->OpenNoLock(ident, logopt, filename, bufferingType);
    this->Unlock();
    return(rc);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  bool FileLogger::OpenNoLock(const string & ident, int logopt,
                              const string & filename,
                              BufferingType bufferingType)
  {
    bool  rc = false;

    //  We can't portably check all useful options in one shot;
    //  FreeBSD and Linux have LOG_PERROR, and Solaris has LOG_NOWAIT.
    int  knownLogOpts = LOG_PID | LOG_CONS | LOG_NDELAY;

#ifdef LOG_NOWAIT
    knownLogOpts |= LOG_NOWAIT;
#endif
#ifdef LOG_PERROR
    knownLogOpts |= LOG_PERROR;
#endif

    if (logopt & (~knownLogOpts)) {
      //  unknown option
      return(rc);
    }

    if (_isOpen) {
      //  we're already open; close first
      if (! CloseNoLock()) {
        //  failed to close
        return(rc);
      }
    }

    //  open the current logfile.
    rc = OpenFile(filename);

    if (rc) {
      _ident = ident;
      _logOptions = logopt;
      _filePrefix = filename;
      _bufferingType = bufferingType;
      setvbuf(_file, NULL, bufferingType, 0);
      char  buf[MAXHOSTNAMELEN];
      memset(buf,0,MAXHOSTNAMELEN);
      if (gethostname(buf, MAXHOSTNAMELEN-1) == 0) {
        _hostName = buf;
        string::size_type idx = _hostName.find_first_of(".");
        if (idx != _hostName.npos) {
          _hostName.erase(idx);
        }
      }
      else {
        _hostName = "";
      }
    }

    return(rc);
  }

  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  bool FileLogger::OpenFile(const string & filename)
  {
    bool  rc = false;
    _file = fopen(filename.c_str(),"w");
    if (_file) {
      rc = true;
      _isOpen = true;
      _currentFilePosition = 0;
    }

    return(rc);
  }
    
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  bool FileLogger::IsOpen() const
  {
    Lock();
    bool  rc = _isOpen;
    Unlock();
    return(rc);
  }

  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  bool FileLogger::Close()
  {
    Lock();
    bool  rc = CloseNoLock();
    Unlock();
    return(rc);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  bool FileLogger::CloseNoLock()
  {
    bool  rc = false;

    if ((! _isOpen) || (! _file))
      return(rc);
    
    if (fclose(_file) == 0) {
      rc = true;
      _file = (FILE *)0;
    }

    if (rc) {
      _isOpen = false;
      _ident = "";
      _logOptions = 0;
    }
    
    return(rc);
  }
      
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  bool FileLogger::Log(int priority, const char * message, ...)
  {
    va_list     vaList;
    va_start(vaList,message);
    bool  rc = FileLogger::VaLog(priority,message,vaList);
    va_end(vaList);
    return(rc);
  }

  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  bool FileLogger::Log(const string & filename, int lineno, 
                       const string & function, int priority,
                       const char * message, ...)
  {
    va_list     vaList;
    va_start(vaList,message);
    bool  rc = FileLogger::VaLog(filename, lineno, function, 
                                 priority, message, vaList);
    va_end(vaList);
    return(rc);
  }
    
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  bool FileLogger::VaLog(const string & filename, int lineno, 
                         const string & function, int priority,
                         const string & message, va_list & vaList)
  {
    ostringstream  fmt;
    fmt << message;
    if (FileLogger::_showFunction) {
      fmt << " (" << function << ")";
    }
    if (FileLogger::_showFileLocation) {
      fmt << " {" << PathUtils::Basename(filename) << ":" << lineno << "}";
    }
    return(FileLogger::VaLog(priority, fmt.str(), vaList));
  }
    
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  bool FileLogger::VaLog(int priority, const string & message, 
                         va_list & vaList)
  {
    bool  rc = false;

    Lock();
    
    if (! _isOpen) {
      Unlock();
      return(rc);
    }

    if (priority > _minimumPriority) {
      Unlock();
      return(true);
    }

    static const char *monthNames[] = { "Jan", "Feb", "Mar", "Apr",
                                        "May", "Jun", "Jul", "Aug",
                                        "Sep", "Oct", "Nov", "Dec" };
    time_t      ts;
    struct tm  *tmPtr;
    string      fmtString("");

    time(&ts);
    tmPtr = localtime(&ts);
    fprintf(_file,"%s %d %02d:%02d:%02d %s %s",
            monthNames[tmPtr->tm_mon],tmPtr->tm_mday,tmPtr->tm_hour,
            tmPtr->tm_min,tmPtr->tm_sec,_hostName.c_str(),
            _ident.c_str());
    if (_logOptions & LOG_PID) {
      fprintf(_file,"[%u]",(uint32_t)getpid());
    }
    
    fmtString = string(": ");
    if (_showPriorities) {
      fmtString += PriorityTag(priority) + " ";
    }
    fmtString += message;
    vfprintf(_file,fmtString.c_str(),vaList);
    if (fprintf(_file,"\n") > 0) {
      rc = true;
    }
    
    //  special code for LOG_PERROR
#ifdef LOG_PERROR
    if (_logOptions & LOG_PERROR) {
      fprintf(stderr,"%s %d %02d:%02d:%02d ",
              monthNames[tmPtr->tm_mon],tmPtr->tm_mday,tmPtr->tm_hour,
              tmPtr->tm_min,tmPtr->tm_sec);
      vfprintf(stderr,fmtString.c_str(),vaList);
      fprintf(stderr,"\n");
    }
#endif
    
    _currentFilePosition = ftell(_file);
    if (_rolloverSize) {
      if (_currentFilePosition > _rolloverSize) {
        rc = RollFilesOver();
      }
    }

    Unlock();
    
    return(rc);
  }

  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  bool FileLogger::Flush()
  {
    bool  rc = false;

    Lock();
    if (_isOpen && _file) {
      if (fflush(_file) == 0) {
        rc = true;
      }
    }

    Unlock();
    
    return(rc);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  int FileLogger::MinimumPriority() const
  {
    Lock();
    int  rc = _minimumPriority;
    Unlock();
    return(rc);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  int FileLogger::MinimumPriority(int minimumPriority)
  {
    Lock();
    _minimumPriority = minimumPriority;
    Unlock();
    return(minimumPriority);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  bool FileLogger::ShowPriorities() const
  {
    Lock();
    bool  rc = _showPriorities;
    Unlock();
    return(rc);
  }

  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  bool FileLogger::ShowPriorities(bool showPriorities)
  {
    Lock();
    _showPriorities = showPriorities;
    Unlock();
    return(showPriorities);
  }

  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  bool FileLogger::ShowFunction() const
  {
    Lock();
    bool  rc = _showFunction;
    Unlock();
    return(rc);
  }

  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  bool FileLogger::ShowFunction(bool showFunction)
  {
    Lock();
    _showFunction = showFunction;
    Unlock();
    return(showFunction);
  }

  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  bool FileLogger::ShowFileLocation() const
  {
    Lock();
    bool  rc = _showFileLocation;
    Unlock();
    return(rc);
  }

  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  bool FileLogger::ShowFileLocation(bool showFileLocation)
  {
    Lock();
    _showFileLocation = showFileLocation;
    Unlock();
    return(showFileLocation);
  }

  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  uint32_t FileLogger::RolloverSize() const
  {
    Lock();
    uint32_t  rc = _rolloverSize;
    Unlock();
    return(rc);
  }

  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  uint32_t FileLogger::RolloverSize(uint32_t rolloverSize)
  {
    Lock();
    _rolloverSize = rolloverSize;
    Unlock();
    return(rolloverSize);
  }

  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  uint32_t FileLogger::NumRolloverFiles() const
  {
    Lock();
    uint32_t  rc = _numRolloverFiles;
    Unlock();
    return(rc);
  }

  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  uint32_t FileLogger::NumRolloverFiles(uint32_t numRolloverFiles)
  {
    Lock();
    _numRolloverFiles = numRolloverFiles;
    Unlock();
    return(numRolloverFiles);
  }

  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  bool FileLogger::RollFilesOver()
  {
    int      logopt = _logOptions;
    string   ident = _ident;
    string   filePrefix = _filePrefix;
    BufferingType  bufferingType = _bufferingType;
    
    if (_isOpen) {
      if (! CloseNoLock())
        return(false);
    }

    bool         rc = true;
    char         buf[16];
    string       srcFile, dstFile;
    struct stat  statbuf;
    
    for (uint32_t i = _numRolloverFiles; i > 1; --i) {
      memset(buf,0,16);
      snprintf(buf,sizeof(buf),"%u",i - 1);
      srcFile = filePrefix + "." + buf;
      if (stat(srcFile.c_str(),&statbuf) == 0) {
        //  srcFile exists.
        memset(buf,0,16);
        snprintf(buf,sizeof(buf),"%u",i);
        dstFile = filePrefix + "." + buf;
        if (rename(srcFile.c_str(),dstFile.c_str()) != 0) {
          rc = false;
          break;
        }
      }
    }

    if (_numRolloverFiles > 0) {
      dstFile = filePrefix + ".0";
      if (stat(filePrefix.c_str(),&statbuf) == 0) {
        if (rename(filePrefix.c_str(),dstFile.c_str()) != 0) {
          rc = false;
        }
      }
    }
    
    if (! rc)
      return(rc);

    rc = OpenNoLock(ident,logopt,filePrefix,bufferingType);
    
    return(rc);
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  bool FileLogger::
  SetPriorities(const map<int,pair<string,string> > & priorities)
  {
    bool  rc = false;
    if (Lock()) {
      _priorityTags.clear();
      _priorityNames.clear();
      map<int, pair<string,string> >::const_iterator  i;
      for (auto & i : priorities) {
        _priorityTags[i.first] = i.second.first;
        _priorityNames[i.second.second] = i.first;
      }
      Unlock();
      rc = true;
    }
    return(rc);
  }
  
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  std::string FileLogger::PriorityTag(int priority)
  {
    string  rc("");
    auto  i = _priorityTags.find(priority);
    if (i != _priorityTags.end()) {
      rc = i->second;
    }
    return(rc);
  }
    
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  void FileLogger::InitializePriorities()
  {
    if (_priorityTags.empty()) {
      _priorityTags[LOG_EMERG]   = "[M]";
      _priorityTags[LOG_ALERT]   = "[A]";
      _priorityTags[LOG_CRIT]    = "[C]";
      _priorityTags[LOG_ERR]     = "[E]";
      _priorityTags[LOG_WARNING] = "[W]";
      _priorityTags[LOG_NOTICE]  = "[N]";
      _priorityTags[LOG_INFO]    = "[I]";
      _priorityTags[LOG_DEBUG]   = "[D]";
    }
    
    if (_priorityNames.empty()) {
      _priorityNames["debug"]     = LOG_DEBUG;
      _priorityNames["info"]      = LOG_INFO;
      _priorityNames["notice"]    = LOG_NOTICE;
      _priorityNames["warning"]   = LOG_WARNING;
      _priorityNames["error"]     = LOG_ERR;
      _priorityNames["critical"]  = LOG_CRIT;
      _priorityNames["alert"]     = LOG_ALERT;
      _priorityNames["emergency"] = LOG_EMERG;
    }

    return;
  }
  

}  // namespace Dwm
