//===========================================================================
// @(#) $Name$
// @(#) $Id: DwmSignal.cc 8388 2016-04-17 03:23:45Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2000-2007
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR
//  PURPOSE, OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

#include <cassert>
#include <cstring>

#include "DwmMutex.hh"
#include "DwmSignal.hh"
#include "DwmSysLogger.hh"
#include "DwmSvnTag.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwm/tags/libDwm-0.6.6/src/DwmSignal.cc 8388 $");

namespace Dwm {

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  void Signal::FillNames()
  {
    static Pthread::Mutex  mtx;
    mtx.Lock();
    
    if (Signal::_names.empty()) {
      Signal::_names[0] = "Unknown";
      Signal::_names[SIGHUP] = "SIGHUP";
      Signal::_names[SIGINT] = "SIGINT";
      Signal::_names[SIGQUIT] = "SIGQUIT";
      Signal::_names[SIGILL] = "SIGILL";
      #ifdef SIGTRAP
        Signal::_names[SIGTRAP] = "SIGTRAP";
      #endif
      Signal::_names[SIGABRT] = "SIGABRT";
      #ifdef SIGEMT
        Signal::_names[SIGEMT] = "SIGEMT";
      #endif
      Signal::_names[SIGFPE] = "SIGFPE";
      Signal::_names[SIGKILL] = "SIGKILL";
      #ifdef SIGBUS
        Signal::_names[SIGBUS] = "SIGBUS";
      #endif
      Signal::_names[SIGSEGV] = "SIGSEGV";
      #ifdef SIGSYS
        Signal::_names[SIGSYS] = "SIGSYS";
      #endif
      Signal::_names[SIGPIPE] = "SIGPIPE";
      Signal::_names[SIGALRM] = "SIGALRM";
      Signal::_names[SIGTERM] = "SIGTERM";
      #ifdef SIGURG
        Signal::_names[SIGURG] = "SIGURG";
      #endif
      Signal::_names[SIGSTOP] = "SIGSTOP";
      Signal::_names[SIGTSTP] = "SIGTSTP";
      Signal::_names[SIGCONT] = "SIGCONT";
      Signal::_names[SIGCHLD] = "SIGCHLD";
      Signal::_names[SIGTTIN] = "SIGTTIN";
      Signal::_names[SIGTTOU] = "SIGTTOU";
      #ifdef SIGIO
        Signal::_names[SIGIO] = "SIGIO";
      #endif
      #ifdef SIGXCPU
        Signal::_names[SIGXCPU] = "SIGXCPU";
      #endif
      #ifdef SIGXFSZ
        Signal::_names[SIGXFSZ] = "SIGXFSZ";
      #endif
      #ifdef SIGVTALRM
        Signal::_names[SIGVTALRM] = "SIGVTALRM";
      #endif
      #ifdef SIGPROF
        Signal::_names[SIGPROF] = "SIGPROF";
      #endif
      #ifdef SIGWINCH
        Signal::_names[SIGWINCH] = "SIGWINCH";
      #endif
      #ifdef SIGINFO
        Signal::_names[SIGINFO] = "SIGINFO";
      #endif
        
      Signal::_names[SIGUSR1] = "SIGUSR1";
      Signal::_names[SIGUSR2] = "SIGUSR2";
    }
    mtx.Unlock();
  }
  
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  Signal::Signal()
  : _sigNum(-1)
  {
    if (Signal::_names.empty())
      Signal::FillNames();
  }
  
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  Signal::Signal(int sigNum)
    : _sigNum(sigNum)
  {
    if (Signal::_names.empty())
      Signal::FillNames();
  }
  
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  bool Signal::Block()
  {
    bool      rc = false;
    sigset_t  blockSet;
    
    sigemptyset(&blockSet);
    sigaddset(&blockSet,this->_sigNum);

    if (sigprocmask(SIG_BLOCK,&blockSet,NULL) == 0)
      rc = true;
    
    return(rc);
  }
  
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  bool Signal::Unblock()
  {
    bool      rc = false;
    sigset_t  blockSet;
    
    sigemptyset(&blockSet);
    sigaddset(&blockSet,this->_sigNum);

    if (sigprocmask(SIG_UNBLOCK,&blockSet,NULL) == 0) {
      rc = true;
    }
    else {
      Syslog(LOG_ERR, "sigprocmask(SIG_UNBLOCK,%p,NULL) failed", &blockSet);
    }
    
    return(rc);
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  bool Signal::IsBlocked()
  {
    bool      rc = false;
    sigset_t  blockSet;
    
    sigemptyset(&blockSet);
    if (sigprocmask(SIG_BLOCK,NULL,&blockSet) == 0) {
      if (sigismember(&blockSet,this->_sigNum)) {
        rc = true;
      }
    }
    else {
      Syslog(LOG_ERR,"sigprocmask(SIG_BLOCK,NULL,%p) failed", &blockSet);
    }
      
    return(rc);
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  bool Signal::IsPending()
  {
    bool      rc = false;
    sigset_t  pendingSet;
    sigemptyset(&pendingSet);

    if (sigpending(&pendingSet) == 0) {
      if (sigismember(&pendingSet,this->_sigNum)) {
        rc = true;
      }
    }
    else {
      Syslog(LOG_ERR,"sigpending(%p) failed", &pendingSet);
    }

    return(rc);
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  void Signal::Wait()
  {
    sigset_t  sigSet;
    
    if (sigemptyset(&sigSet) != 0)
      ;
    if (sigaddset(&sigSet,_sigNum) != 0)
      ;
    int  sigNum;
    if (sigwait(&sigSet,&sigNum) == 0) {
      assert(sigNum == _sigNum);
    }
    return;
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  int Signal::PushHandler(void (*handler)(int))
  {
    int  rc = -1;
    
    struct sigaction  action;
    memset(&action,0,sizeof(action));
    action.sa_handler = handler;
    action.sa_flags   = 0;

    if (sigaction(this->_sigNum,&action,NULL) < 0) {
      Syslog(LOG_ERR,"sigaction(%d,%p,NULL) failed {%s:%d}",
             this->_sigNum,&action,__FILE__,__LINE__);
    }
    else {
      this->_handlers.push_front(handler);
      rc = this->_handlers.size();
    }

    return(rc);
  }
  
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  int Signal::PopHandler()
  {
    int  rc = -1;
    if (! this->_handlers.empty()) {
      this->_handlers.pop_front();
      if (this->_handlers.empty()) {
        struct sigaction  action;
        memset(&action,0,sizeof(action));
        action.sa_handler = SIG_DFL;
        if (sigaction(this->_sigNum,&action,NULL) == 0) {
          rc = 0;
        }
      }
      else {
        struct sigaction  action;
        memset(&action,0,sizeof(action));
        action.sa_handler = this->_handlers.front();
        action.sa_flags = 0;
        sigaction(this->_sigNum,&action,NULL);
      }
      rc = this->_handlers.size();
    }

    return(rc);
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  const std::string & Signal::Name() const
  {
    std::map<int,std::string>::const_iterator iter;
    iter = Signal::_names.find(this->_sigNum);
    if (iter != Signal::_names.end())
      return(iter->second);
    else
      return(Signal::_names[0]);
  }
    
  std::map<int,std::string>  Signal::_names;

}  // namespace Dwm
