//===========================================================================
// @(#) $DwmPath: dwm/libDwm/tags/libDwm-0.6.6/include/DwmLoadBalancer.hh 8401 $
// @(#) $Id: DwmLoadBalancer.hh 8401 2016-04-17 06:44:31Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2008, 2009, 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmLoadBalancer.hh
//!  \brief Dwm::LoadBalancer class template definition
//---------------------------------------------------------------------------

#ifndef _DWMLOADBALANCER_HH_
#define _DWMLOADBALANCER_HH_

extern "C" {
  #include <unistd.h>
}

#include <algorithm>
#include <iostream>
#include <queue>
#include <string>
#include <thread>
#include <vector>

#include "DwmSysLogger.hh"
#include "DwmThreadQueue.hh"

namespace Dwm {

  //--------------------------------------------------------------------------
  //!  A simple load balancer class template which balances work across
  //!  Worker objects that each run in their own thread.
  //!
  //!  A user would derive a worker class from LoadBalancer<ItemT>::Worker,
  //!  and implement the Worker::ProcessWork(ItemT) member.  The worker can
  //!  be added to a LoadBalancer via the LoadBalancer::AddWorker() member.
  //!  Work can be added to the LoadBalancer with AddWork(ItemT).  A unit of
  //!  work is encapsulated in the ItemT template parameter.  This is a
  //!  template so that any copyable type can be used as a unit of work.
  //!
  //!  Today, this class template may not be all that useful (we have
  //!  the concurrency features of C++11 and C++14).  I originally wrote
  //!  this class using pthreads, before we had C++11 and C++14.  It's
  //!  still in use in some of my code, but I may revisit this soon and
  //!  modernize it.  For now the only modernization is the migration to
  //!  std::thread.
  //--------------------------------------------------------------------------
  template <typename ItemT>
  class LoadBalancer
  {
  public:
    //------------------------------------------------------------------------
    //!  Worker class for LoadBalancer.
    //------------------------------------------------------------------------
    class Worker
    {
    public:
      //----------------------------------------------------------------------
      //!  Constructs the worker.  @c waitForNotEmptyMicroseconds is the
      //!  period of time to wait for the work queue to become non-empty
      //!  before checking again whether or not to continue.
      //----------------------------------------------------------------------
      Worker(uint32_t waitNotEmptyMicroseconds = 1000)
          : _workQueue(), _keepRunning(false), _thread(),
            _waitUsecs(waitNotEmptyMicroseconds)
      {}

      //----------------------------------------------------------------------
      //!  Destructor.  Stops the worker thread.
      //----------------------------------------------------------------------
      virtual ~Worker()
      {
        Stop();
      }
      
      //----------------------------------------------------------------------
      //!  Adds a work item for the worker.
      //----------------------------------------------------------------------
      void AddWork(ItemT item)
      {
        _workQueue.PushBack(item);
        return;
      }
      
      //----------------------------------------------------------------------
      //!  Adds work items for the worker.
      //----------------------------------------------------------------------
      template <typename InputIterator>
      void AddWork(InputIterator firstIter, InputIterator lastIter)
      {
        _workQueue.PushBack(firstIter, lastIter);
        return;
      }
      
      //----------------------------------------------------------------------
      //!  Returns the current length of the worker's work queue.
      //----------------------------------------------------------------------
      uint32_t QueueLength() const
      {
        return _workQueue.Length();
      }
      
      //----------------------------------------------------------------------
      //!  Returns true if the worker is ready for more work (has room in
      //!  its work queue and is running).
      //----------------------------------------------------------------------
      bool ReadyForWork() const
      {
        return (_keepRunning
                && ((! _workQueue.MaxLength())
                    || (_workQueue.Length() < _workQueue.MaxLength())));
      }
      
      //----------------------------------------------------------------------
      //!  Sets the maximum length of the worker's work queue.
      //----------------------------------------------------------------------
      void MaxWork(uint32_t maxItems)
      {
        _workQueue.MaxLength(maxItems);
      }
      
      //----------------------------------------------------------------------
      //!  Starts the worker.
      //----------------------------------------------------------------------
      bool Start()
      {
        bool  rc = false;
        _keepRunning = true;
        _thread = std::thread(&Worker::Run, this);
        rc = true;
        return rc;
      }
      
      //----------------------------------------------------------------------
      //!  Stops the worker.
      //----------------------------------------------------------------------
      void Stop()
      {
        _keepRunning = false;
        if (_thread.joinable()) {
          _thread.join();
        }
        return;
      }

      //----------------------------------------------------------------------
      //!  Returns true if the worker's thread is running.
      //----------------------------------------------------------------------
      bool IsRunning()
      {
        return (_thread.joinable());
      }
      
      //----------------------------------------------------------------------
      //!  Runs the worker thread.
      //----------------------------------------------------------------------
      void Run()
      {
        while (_keepRunning) {
          if (_workQueue.TimedWaitForNotEmpty(std::chrono::microseconds(_waitUsecs))) {
            std::deque<ItemT>  myCopy;
            _workQueue.Swap(myCopy);
            //  try processing in bulk
            if (! ProcessWork(myCopy)) {
              //  else fall back to one at a time.
              for (auto i : myCopy) {
                ProcessWork(i);
              }
            }
          }
        }
        return;
      }
      
      //----------------------------------------------------------------------
      //!  Pure virtual member to process a single work item.  Obviously
      //!  must be implemented by derived classes.
      //----------------------------------------------------------------------
      virtual void ProcessWork(ItemT item) = 0;

      //----------------------------------------------------------------------
      //!  Process a deque of work items.  The default implementation does
      //!  nothing and returns @c false.  If you impelement this in your
      //!  derived class, it should return @c true.
      //----------------------------------------------------------------------
      virtual bool ProcessWork(std::deque<ItemT> & items)
      {
        return false;
      }
      
    protected:
      Thread::Queue<ItemT>  _workQueue;
      bool                  _keepRunning;
      std::thread           _thread;
      uint32_t              _waitUsecs;
    };

    //------------------------------------------------------------------------
    //!  Comparison class for workers within the LoadBalancer.
    //------------------------------------------------------------------------
    class CompareWorkers
    {
    public:
      //----------------------------------------------------------------------
      //!  Returns @c true if @c a's work queus is shorter than @c b's
      //!  work queue.
      //----------------------------------------------------------------------
      bool operator () (const Worker * a,
                        const Worker * b) const
      {
        return (a->QueueLength() < b->QueueLength());
      }
    };

    //------------------------------------------------------------------------
    //!  Adds the given worker to the load balancer.
    //------------------------------------------------------------------------
    void AddWorker(Worker *worker)
    {
      _workers.push_back(worker);
      return;
    }

    //------------------------------------------------------------------------
    //!  Adds work to be done with load balancing.
    //------------------------------------------------------------------------
    void AddWork(ItemT item)
    {
      auto  w = std::min_element(_workers.begin(), _workers.end(),
                                 CompareWorkers());
      if (w != _workers.end()) {
        (*w)->AddWork(item);
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  Adds work to be done with load balancing.
    //------------------------------------------------------------------------
    template <typename InputIterator>
    void AddWork(InputIterator firstIter, InputIterator lastIter)
    {
      while (! WorkerReady()) {
        usleep(1000);
      }
      auto  w = std::min_element(_workers.begin(), _workers.end(),
                                 CompareWorkers());
      (*w)->AddWork(firstIter, lastIter);
      return;
    }

    //------------------------------------------------------------------------
    //!  Calls Worker::Stop() on all encapsulated Worker objects.
    //------------------------------------------------------------------------
    void Stop()
    {
      for (auto w : _workers) {
        w->Stop();
      }
      return;
    }

    //------------------------------------------------------------------------
    //!  Returns a const reference to the encapsulated workers.
    //------------------------------------------------------------------------
    const std::vector<Worker *> & Workers() const
    {
      return _workers;
    }
    
  private:
    std::vector<Worker *>  _workers;

    //------------------------------------------------------------------------
    //!  Returns true if there is a worker ready to do work.
    //------------------------------------------------------------------------
    bool WorkerReady()
    {
      bool  rc = false;
      for (auto w : _workers) {
        if (w->ReadyForWork()) {
          rc = true;
          break;
        }
      }
      return rc;
    }
    
  };
  
}  // namespace Dwm


#endif  // _DWMLOADBALANCER_HH_
