//===========================================================================
// @(#) $DwmPath: dwm/libDwm/tags/libDwm-0.6.19/include/DwmBZ2IO.hh 9341 $
// @(#) $Id: DwmBZ2IO.hh 9341 2017-05-17 07:14:02Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2004-2006, 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR
//  PURPOSE, OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmBZ2IO.hh
//!  \brief Dwm::BZ2IO class definition
//---------------------------------------------------------------------------

#ifndef _DWMBZ2IO_HH_
#define _DWMBZ2IO_HH_

extern "C" {
  #include <inttypes.h>
  #include <sys/types.h>
}

#include <cstdio>
#include <deque>
#include <iostream>
#include <list>
#include <map>
#include <set>
#include <string>
#include <tuple>
#include <unordered_map>
#include <unordered_set>
#include <vector>

#include "DwmPortability.hh"

#include "DwmBZ2Readable.hh"
#include "DwmBZ2Writable.hh"

namespace Dwm {

  //--------------------------------------------------------------------------
  //!  This class contains a collection of static functions for reading and
  //!  writing simple types in network byte order (MSB first) from/to bzip2
  //!  files.  It also contains functions to read and write strings from/to
  //!  bzip2 files.  It also contains function templates to read and write 
  //!  STL deques, lists, vectors, maps, multimaps, sets, multisets, 
  //!  hash_maps and hash_sets from/to bzip2 files.  We use our member 
  //!  functions to handle reading and writing simple types in these
  //!  containers, and function templates to handle reading and writing other
  //!  class types.  In the latter case, the class must implement the 
  //!  BZ2Readable and BZ2Writable interfaces, since our function templates
  //!  simply call out to them.
  //--------------------------------------------------------------------------
  class BZ2IO
  {
  public:
    //------------------------------------------------------------------------
    //!  Writes \c c to \c bzf.  Returns the number of bytes written on
    //!  success, -1 on failure.
    //------------------------------------------------------------------------
    static int BZWrite(BZFILE *bzf, char c);

    //------------------------------------------------------------------------
    //!  Writes \c c to \c bzf.  Returns the number of bytes written (1) on
    //!  success, less on failure.
    //------------------------------------------------------------------------
    static int BZWrite(BZFILE *bzf, uint8_t c);

    //------------------------------------------------------------------------
    //!  Writes \c val to \c bzf, in network byte order (MSB first).
    //!  Returns the number of bytes written (2) on success, less on
    //!  failure.
    //------------------------------------------------------------------------
    static int BZWrite(BZFILE *bzf, int16_t val);

    //------------------------------------------------------------------------
    //!  Writes \c val to \c bzf, in network byte order (MSB first).
    //!  Returns the number of bytes written (2) on success, less on
    //!  failure.
    //------------------------------------------------------------------------
    static int BZWrite(BZFILE *bzf, uint16_t val);

    //------------------------------------------------------------------------
    //!  Writes \c val to \c bzf, in network byte order (MSB first).
    //!  Returns the number of bytes written (4) on success, less on
    //!  failure.
    //------------------------------------------------------------------------
    static int BZWrite(BZFILE *bzf, int32_t val);

    //------------------------------------------------------------------------
    //!  Writes \c val to \c bzf, in network byte order (MSB first).
    //!  Returns the number of bytes written (4) on success, less on
    //!  failure.
    //------------------------------------------------------------------------
    static int BZWrite(BZFILE *bzf, uint32_t val);

    //------------------------------------------------------------------------
    //!  Writes \c val to \c bzf, in network byte order (MSB first).
    //!  Returns the number of bytes written (8) on success, less on
    //!  failure.
    //------------------------------------------------------------------------
    static int BZWrite(BZFILE *bzf, const int64_t & val);

    //------------------------------------------------------------------------
    //!  Writes \c val to \c bzf, in network byte order (MSB first).
    //!  Returns the number of bytes written (8) on success, less on
    //!  failure.
    //------------------------------------------------------------------------
    static int BZWrite(BZFILE *bzf, const uint64_t & val);

    //------------------------------------------------------------------------
    //!  Writes \c val tp \c bzf, in IEEE format (see RFC 1832
    //!  and/or ANSI/IEEE Standard 754-1985).  Returns the number of bytes
    //!  written (4) on success, less on failure.
    //------------------------------------------------------------------------
    static int BZWrite(BZFILE *bzf, float val);
    
    //------------------------------------------------------------------------
    //!  Writes \c val tp \c bzf, in IEEE format (see RFC 1832
    //!  and/or ANSI/IEEE Standard 754-1985).  Returns the number of bytes
    //!  written (8) on success, less on failure.
    //------------------------------------------------------------------------
    static int BZWrite(BZFILE *bzf, const double & val);
    
    //------------------------------------------------------------------------
    //!  Writes \c s to \c bzf.  Returns the number of bytes written on
    //!  success, -1 on failure.  Note that a 32-bit value is written first,
    //!  holding the length of the string.  The terminating NULL is also
    //!  written.  Hence, on success this will always return a value of
    //!  5 or greater.
    //------------------------------------------------------------------------
    static int BZWrite(BZFILE *bzf, const std::string & s);
    
    //------------------------------------------------------------------------
    //!  Reads \c from \c bzf.  Returns the number of bytes read (1) on 
    //!  success, less on failure.
    //------------------------------------------------------------------------
    static int BZRead(BZFILE *bzf, char & c);

    //------------------------------------------------------------------------
    //!  Reads \c from \c bzf.  Returns the number of bytes read (1) on 
    //!  success, less on failure.
    //------------------------------------------------------------------------
    static int BZRead(BZFILE *bzf, uint8_t & c);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c bzf, in network byte order (MSB first). 
    //!  Returns number of bytes read (2) on success, less on failure.
    //------------------------------------------------------------------------
    static int BZRead(BZFILE *bzf, int16_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c bzf, in network byte order (MSB first). 
    //!  Returns number of bytes read (2) on success, less on failure.
    //------------------------------------------------------------------------
    static int BZRead(BZFILE *bzf, uint16_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c bzf, in network byte order (MSB first). 
    //!  Returns number of bytes read (4) on success, less on failure.
    //------------------------------------------------------------------------
    static int BZRead(BZFILE *bzf, int32_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c bzf, in network byte order (MSB first). 
    //!  Returns number of bytes read (4) on success, less on failure.
    //------------------------------------------------------------------------
    static int BZRead(BZFILE *bzf, uint32_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c bzf, in network byte order (MSB first). 
    //!  Returns number of bytes read (8) on success, less on failure.
    //------------------------------------------------------------------------
    static int BZRead(BZFILE *bzf, int64_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c bzf, in network byte order (MSB first). 
    //!  Returns number of bytes read (8) on success, less on failure.
    //------------------------------------------------------------------------
    static int BZRead(BZFILE *bzf, uint64_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c bzf, in IEEE format (see RFC 1832 and/or 
    //!  ANSI/IEEE Standard 754-1985).  Returns number of bytes read (4)
    //!  on success, less on failure.
    //------------------------------------------------------------------------
    static int BZRead(BZFILE *bzf, float & val);
    
    //------------------------------------------------------------------------
    //!  Reads \c val from \c bzf, in IEEE format (see RFC 1832 and/or 
    //!  ANSI/IEEE Standard 754-1985).  Returns number of bytes read (8)
    //!  on success, less on failure.
    //------------------------------------------------------------------------
    static int BZRead(BZFILE *bzf, double & val);
    
    //------------------------------------------------------------------------
    //!  Reads \c s from \c bzf.  Returns the number of bytes read on success,
    //!  -1 on failure.  Since we write strings with a 32-bit unsigned length
    //!  value preceding the actual string, and always have a terminating
    //!  NULL, this always reads at least 5 bytes on success.
    //------------------------------------------------------------------------
    static int BZRead(BZFILE *bzf, std::string & s);
    
    //------------------------------------------------------------------------
    //!  Wrapper function to write a BZ2Writable object to a BZFILE pointer.
    //------------------------------------------------------------------------
    static int BZWrite(BZFILE *bzf, const BZ2Writable & val)
    {
      return(val.BZWrite(bzf));
    }

    //------------------------------------------------------------------------
    //!  Wrapper function to read a BZ2Readable object from a BZFILE pointer.
    //------------------------------------------------------------------------
    static int BZRead(BZFILE *bzf, BZ2Readable & val)
    {
      return(val.BZRead(bzf));
    }

    //------------------------------------------------------------------------
    //!  Reads a pair<_firstT,_secondT> from a BZFILE pointer.  Returns
    //!  the number of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _firstT, typename _secondT>
    static int BZRead(BZFILE *bzf, std::pair<_firstT, _secondT> & p)
    {
      int  rc = -1;
      if (bzf) {
        int  bytesRead = BZRead(bzf, p.first);
        if (bytesRead > 0) {
          rc = bytesRead;
          bytesRead = BZRead(bzf, p.second);
          if (bytesRead > 0)
            rc += bytesRead;
          else
            rc = -1;
        }
      }
      return(rc);
    }
    
    //------------------------------------------------------------------------
    //!  Writes a pair<_firstT,_secondT> to a BZFILE pointer.  Returns
    //!  the number of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _firstT, typename _secondT>
    static int BZWrite(BZFILE *bzf, const std::pair<_firstT,_secondT> & p)
    {
      int  rc = -1;
      if (bzf) {
        int  bytesWritten = BZWrite(bzf, p.first);
        if (bytesWritten > 0) {
          rc = bytesWritten;
          bytesWritten = BZWrite(bzf, p.second);
          if (bytesWritten > 0) {
            rc += bytesWritten;
          }
          else {
            rc = -1;
          }
        }
      }
      return(rc);
    }
                                
    //------------------------------------------------------------------------
    //!  Reads a map<_keyT,_valueT> from a BZFILE pointer.  Returns the
    //!  number of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Compare, typename _Alloc>
    static int BZRead(BZFILE *bzf, std::map<_keyT, _valueT, _Compare, _Alloc> & m)
    {
      return(PairAssocContBZRead<std::map<_keyT, _valueT, _Compare, _Alloc> >(bzf, m));
    }

    //------------------------------------------------------------------------
    //!  Writes a map<_keyT,_valueT> to a BZFILE pointer.  Returns the
    //!  number of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template<typename _keyT, typename _valueT, 
             typename _Compare, typename _Alloc>
    static int 
    BZWrite(BZFILE *bzf, const std::map<_keyT,_valueT,_Compare,_Alloc> & m)
    {
      return(ContainerBZWrite<std::map<_keyT,_valueT,_Compare,_Alloc> >(bzf, m));
    }
                                
    //------------------------------------------------------------------------
    //!  Reads a multimap<_keyT,_valueT> from a BZFILE pointer.  Returns the
    //!  number of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Compare, typename _Alloc>
    static int 
    BZRead(BZFILE *bzf, std::multimap<_keyT, _valueT, _Compare, _Alloc> & m)
    {
      return(PairAssocContBZRead<std::multimap<_keyT,_valueT,_Compare,_Alloc> >(bzf, m));
    }

    //------------------------------------------------------------------------
    //!  Writes a multimap<_keyT,_valueT> to a BZFILE pointer.  Returns 
    //!  the number of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Compare, typename _Alloc>
    static int 
    BZWrite(BZFILE *bzf, const std::multimap<_keyT,_valueT, _Compare, _Alloc> & m)
    {
      return(ContainerBZWrite<std::multimap<_keyT,_valueT,_Compare,_Alloc> >(bzf, m));
    }
    
    //------------------------------------------------------------------------
    //!  Reads a vector<_valueT> from a BZFILE pointer.  Returns the number
    //!  of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static int BZRead(BZFILE *bzf, std::vector<_valueT, _Alloc> & v)
    {
      return(ContainerBZRead<std::vector<_valueT, _Alloc> >(bzf, v));
    }

    //------------------------------------------------------------------------
    //!  Writes a vector<_valueT> to a BZFILE pointer.  Returns the number
    //!  of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static int BZWrite(BZFILE *bzf, const std::vector<_valueT, _Alloc> & v)
    {
      return(ContainerBZWrite<std::vector<_valueT, _Alloc> >(bzf, v));
    }

    //------------------------------------------------------------------------
    //!  Reads a deque<_valueT> from a BZFILE pointer.  Returns the number
    //!  of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static int BZRead(BZFILE *bzf, std::deque<_valueT, _Alloc> & d)
    {
      return(ContainerBZRead<std::deque<_valueT, _Alloc> >(bzf, d));
    }

    //------------------------------------------------------------------------
    //!  Writes a deque<_valueT> to a BZFILE pointer.  Returns the number
    //!  of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static int BZWrite(BZFILE *bzf, const std::deque<_valueT, _Alloc> & d)
    {
      return(ContainerBZWrite<std::deque<_valueT, _Alloc> >(bzf, d));
    }

    //------------------------------------------------------------------------
    //!  Reads a list<_valueT> from a BZFILE pointer.  Returns the number
    //!  of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static int BZRead(BZFILE *bzf, std::list<_valueT, _Alloc> & l)
    {
      return(ContainerBZRead<std::list<_valueT, _Alloc> >(bzf, l));
    }

    //------------------------------------------------------------------------
    //!  Writes a list<_valueT> to a BZFILE pointer.  Returns the number
    //!  of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static int BZWrite(BZFILE *bzf, const std::list<_valueT, _Alloc> & l)
    {
      return(ContainerBZWrite<std::list<_valueT, _Alloc> >(bzf, l));
    }

    //------------------------------------------------------------------------
    //!  Reads a set<_valueT> from a BZFILE pointer.  Returns the number
    //!  of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Compare, typename _Alloc>
    static int BZRead(BZFILE *bzf, std::set<_valueT, _Compare, _Alloc> & l)
    {
      return(ContainerBZRead<std::set<_valueT, _Compare, _Alloc> >(bzf, l));
    }

    //------------------------------------------------------------------------
    //!  Writes a set<_valueT> to a BZFILE pointer.  Returns the number
    //!  of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Compare, typename _Alloc>
    static int BZWrite(BZFILE *bzf, const std::set<_valueT, _Compare, _Alloc> & l)
    {
      return(ContainerBZWrite<std::set<_valueT, _Compare, _Alloc> >(bzf, l));
    }

    //------------------------------------------------------------------------
    //!  Reads a multiset<_valueT> from a BZFILE pointer.  Returns the number
    //!  of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Compare, typename _Alloc>
    static int BZRead(BZFILE *bzf, std::multiset<_valueT, _Compare, _Alloc> & l)
    {
      return(ContainerBZRead<std::multiset<_valueT, _Compare, _Alloc> >(bzf, l));
    }

    //------------------------------------------------------------------------
    //!  Writes a multiset<_valueT> to a BZFILE pointer.  Returns the number
    //!  of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Compare, typename _Alloc>
    static int
    BZWrite(BZFILE *bzf, const std::multiset<_valueT, _Compare, _Alloc> & l)
    {
      return(ContainerBZWrite<std::multiset<_valueT, _Compare, _Alloc> >(bzf, l));
    }

    //========================================================================
    //  gcc-4.3.x has variadic template support.
    //========================================================================

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    template <typename T, typename... Args>
    static int BZWrite(BZFILE *bzf, const std::tuple<T, Args...> & t)
    {
      return(BZWriteTuple<std::tuple<T, Args...> >(bzf, t));
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    template <typename T, typename... Args>
    static int BZRead(BZFILE *bzf, std::tuple<T, Args...> & t)
    {
      return(BZReadTuple<std::tuple<T, Args...> >(bzf, t));
    }

    //------------------------------------------------------------------------
    //!  T must be a tuple, n is the number of elements.
    //------------------------------------------------------------------------
    template <typename T>
    static int BZWriteTuple(BZFILE *bzf, const T & t)
    {
      return(TupleBZ2IOHelper<T,std::tuple_size<T>::value-1>::Write(bzf, t));
    }

    //------------------------------------------------------------------------
    //!  T must be a tuple, n is the number of elements.
    //------------------------------------------------------------------------
    template <typename T>
    static int BZReadTuple(BZFILE *bzf, T & t)
    {
      return(TupleBZ2IOHelper<T,std::tuple_size<T>::value-1>::Read(bzf, t));
    }

    //------------------------------------------------------------------------
    //!  Reads an unordered_map from a BZFILE pointer.  Returns the number
    //!  of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template<typename _keyT, typename _valueT, typename _Hash,
             typename _Pred, typename _Alloc>
    static int
    BZRead(BZFILE *bzf, 
           std::unordered_map<_keyT,_valueT,_Hash,_Pred,_Alloc> & hm)
    {
      return(PairAssocContBZRead<std::unordered_map<_keyT,_valueT,_Hash,_Pred,_Alloc> >(bzf, hm));
    }

    //------------------------------------------------------------------------
    //!  Writes an unordered_map to a BZFILE pointer.  Returns the number
    //!  of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template<typename _keyT, typename _valueT, typename _Hash,
             typename _Pred, typename _Alloc>
    static int 
    BZWrite(BZFILE *bzf, 
            const std::unordered_map<_keyT,_valueT,_Hash,_Pred,_Alloc> & hm)
    {
      return(ContainerBZWrite<std::unordered_map<_keyT,_valueT,_Hash,_Pred,_Alloc> >(bzf, hm));
    }

    //------------------------------------------------------------------------
    //!  Reads an unordered_set from a BZFILE pointer.  Returns the number
    //!  of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template<typename _valueT, typename _Hash,
             typename _Pred, typename _Alloc>
    static int
    BZRead(BZFILE *bzf, 
           std::unordered_set<_valueT,_Hash,_Pred,_Alloc> & hm)
    {
      return(ContainerBZRead<std::unordered_set<_valueT,_Hash,_Pred,_Alloc> >(bzf, hm));
    }

    //------------------------------------------------------------------------
    //!  Writes an unordered_set to a BZFILE pointer.  Returns the number
    //!  of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template<typename _valueT, typename _Hash,
             typename _Pred, typename _Alloc>
    static int 
    BZWrite(BZFILE *bzf, 
            const std::unordered_set<_valueT,_Hash,_Pred,_Alloc> & hm)
    {
      return(ContainerBZWrite<std::unordered_set<_valueT,_Hash,_Pred,_Alloc> >(bzf, hm));
    }

    //------------------------------------------------------------------------
    //!  Reads an unordered_multimap from a BZFILE pointer.  Returns the
    //!  number of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template<typename _keyT, typename _valueT, typename _Hash,
             typename _Pred, typename _Alloc>
    static int
    BZRead(BZFILE *bzf, 
           std::unordered_multimap<_keyT,_valueT,_Hash,_Pred,_Alloc> & hm)
    {
      return(PairAssocContBZRead<std::unordered_multimap<_keyT,_valueT,_Hash,_Pred,_Alloc> >(bzf, hm));
    }

    //------------------------------------------------------------------------
    //!  Writes an unordered_multimap to a BZFILE pointer.  Returns the number
    //!  of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template<typename _keyT, typename _valueT, typename _Hash,
             typename _Pred, typename _Alloc>
    static int 
    BZWrite(BZFILE *bzf, 
            const std::unordered_multimap<_keyT,_valueT,_Hash,_Pred,_Alloc> & hm)
    {
      return(ContainerBZWrite<std::unordered_multimap<_keyT,_valueT,_Hash,_Pred,_Alloc> >(bzf, hm));
    }

    //------------------------------------------------------------------------
    //!  Reads an unordered_multiset from a BZFILE pointer.  Returns the
    //!  number of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template<typename _valueT, typename _Hash,
             typename _Pred, typename _Alloc>
    static int
    BZRead(BZFILE *bzf, 
           std::unordered_multiset<_valueT,_Hash,_Pred,_Alloc> & hm)
    {
      return(ContainerBZRead<std::unordered_multiset<_valueT,_Hash,_Pred,_Alloc> >(bzf, hm));
    }

    //------------------------------------------------------------------------
    //!  Writes an unordered_multiset to a BZFILE pointer.  Returns the number
    //!  of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template<typename _valueT, typename _Hash,
             typename _Pred, typename _Alloc>
    static int 
    BZWrite(BZFILE *bzf, 
            const std::unordered_multiset<_valueT,_Hash,_Pred,_Alloc> & hm)
    {
      return(ContainerBZWrite<std::unordered_multiset<_valueT,_Hash,_Pred,_Alloc> >(bzf, hm));
    }

  private:
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    template <typename _inputIteratorT>
    static int BZWrite(BZFILE *bzf, _inputIteratorT f, _inputIteratorT l)
    {
      int  rc = 0;
      if (bzf) {
        for ( ; f != l; ++f) {
          int  bytesWritten = BZWrite(bzf, *f);
          if (bytesWritten > 0) {
            rc += bytesWritten;
          }
          else {
            rc = -1;
            break;
          }
        }
      }
      return(rc);
    }

    //------------------------------------------------------------------------
    //!  Reads a container from a BZFILE pointer.  Returns the number of
    //!  bytes read on success, -1 on failure.
    //!  We use this for deques, lists, vectors, sets and multisets.
    //------------------------------------------------------------------------
    template <typename _containerT>
    static int ContainerBZRead(BZFILE *bzf, _containerT & c)
    {
      if (! c.empty())
        c.clear();
      int  rc = -1;
      if (bzf) {
        uint32_t  numEntries;
        int  bytesRead = BZRead(bzf, numEntries);
        if (bytesRead == sizeof(numEntries)) {
          rc = bytesRead;
          for (uint32_t i = 0; i < numEntries; ++i) {
            typename _containerT::value_type  val;
            bytesRead = BZRead(bzf, val);
            if (bytesRead > 0) {
              rc += bytesRead;
              c.insert(c.end(), std::move(val));
            }
            else {
              rc = -1;
              break;
            }
          }
        }
      }
      return(rc);
    }

    //------------------------------------------------------------------------
    //!  Writes a container to a BZFILE pointer.  Returns the number of
    //!  bytes written on success, -1 on failure.
    //!  We use this for all containers.  
    //------------------------------------------------------------------------
    template <typename _containerT>
    static int ContainerBZWrite(BZFILE *bzf, const _containerT & c)
    {
      int  rc = -1;
      if (bzf) {
        uint32_t  numEntries = c.size();
        uint32_t  bytesWritten = BZWrite(bzf, numEntries);
        if (bytesWritten == sizeof(numEntries)) {
          rc = bytesWritten;
          if (numEntries) {
            bytesWritten = 
              BZWrite<typename _containerT::const_iterator>(bzf,
                                                            c.begin(), 
                                                            c.end());
            if (bytesWritten > 0)
              rc += bytesWritten;
            else
              rc = -1;
          }
        }
      }
      return(rc);
    }

    //------------------------------------------------------------------------
    //!  Reads a PairAssociative container from a BZFILE pointer.  Returns
    //!  the number of bytes read on success, -1 on failure.
    //!  We use this for map, multimap and hash_map.
    //------------------------------------------------------------------------
    template <typename _containerT>
    static int PairAssocContBZRead(BZFILE *bzf, _containerT & m)
    {
      int  rc = -1;
      if (! m.empty())
        m.clear();
      if (bzf) {
        uint32_t  numEntries;
        int  bytesRead = BZRead(bzf, numEntries);
        if (bytesRead == sizeof(numEntries)) {
          rc = bytesRead;
          for (uint32_t i = 0; i < numEntries; ++i) {
            typename _containerT::key_type  key;
            bytesRead = BZRead(bzf, key);
            if (bytesRead > 0) {
              rc += bytesRead;
              typename _containerT::mapped_type  val;
              bytesRead = BZRead(bzf, val);
              if (bytesRead > 0) {
                rc += bytesRead;
                m.insert(typename _containerT::value_type(std::move(key),
                                                          std::move(val)));
              }
              else {
                rc = -1;
                break;
              }
            }
            else {
              rc = -1;
              break;
            }
          }
        }
      }
      return(rc);
    }

    //------------------------------------------------------------------------
    //!  Declare tuple BZ2IO helper class template.  elt is the last element
    //!  index (size of the tuple minus 1).
    //------------------------------------------------------------------------
    template <typename T, size_t elt>
    class TupleBZ2IOHelper;
    
    //------------------------------------------------------------------------
    //!  Specialization for a tuple with one element.
    //------------------------------------------------------------------------
    template <typename T>
    class TupleBZ2IOHelper<T, 0>
    {
    public:
      //----------------------------------------------------------------------
      //!  Write a tuple \c t to an ostream \c os.
      //----------------------------------------------------------------------
      static int Write(BZFILE *bzf, const T & t)
      {
        return(BZ2IO::BZWrite(bzf, std::get<0>(t)));
      }
      
      //----------------------------------------------------------------------
      //!  Read a tuple \c t from an istream \c is.
      //----------------------------------------------------------------------
      static int Read(BZFILE *bzf, T & t)
      {
        return(BZ2IO::BZRead(bzf, std::get<0>(t)));
      }
    };
    
    //------------------------------------------------------------------------
    //!  The recursive tuple BZ2IO helper template.
    //------------------------------------------------------------------------
    template <typename T, size_t elt>
    class TupleBZ2IOHelper
    {
    public:
      //----------------------------------------------------------------------
      //!  Write a tuple \c t to an ostream \c os.
      //----------------------------------------------------------------------
      static int Write(BZFILE *bzf, const T & t)
      {
        int  rc = TupleBZ2IOHelper<T,elt-1>::Write(bzf, t);
        if (rc > 0) {
          int  rcr = BZ2IO::BZWrite(bzf, std::get<elt>(t));
          if (rcr > 0) {
            rc += rcr;
          }
          else {
            rc = -1;
          }
        }
        return(rc);
      }
      
      //----------------------------------------------------------------------
      //!  Read a tuple \c t from an istream \c is.
      //----------------------------------------------------------------------
      static int Read(BZFILE *bzf, T & t)
      {
        int  rc = TupleBZ2IOHelper<T,elt-1>::Read(bzf, t);
        if (rc > 0) {
          int  rcr = BZ2IO::BZRead(bzf, std::get<elt>(t));
          if (rcr > 0) {
            rc += rcr;
          }
          else {
            rc = -1;
          }
        }
        return(rc);
      }

    };

  };

}  // namespace Dwm

#endif  // _DWMBZ2IO_HH_

//---------------------------- emacs settings -----------------------------
//  Local Variables:
//  mode: C++
//  tab-width: 2
//  indent-tabs-mode: nil
//  c-basic-offset: 2
//  End:
//-------------------------------------------------------------------------
