//===========================================================================
// @(#) $DwmPath: dwm/libDwm/tags/libDwm-0.6.13/include/DwmIO.hh 9341 $
// @(#) $Id: DwmIO.hh 9341 2017-05-17 07:14:02Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2004, 2005, 2006, 2007, 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR
//  PURPOSE, OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmIO.hh
//!  \brief Dwm::IO class definition
//---------------------------------------------------------------------------

#ifndef _DWMIO_HH_
#define _DWMIO_HH_

extern "C" {
  #include <inttypes.h>
  #include <sys/types.h>
}

#include <cstdio>
#include <deque>
#include <iostream>
#include <list>
#include <map>
#include <set>
#include <string>
#include <tuple>
#include <unordered_map>
#include <unordered_set>
#include <vector>

#include "DwmPortability.hh"

#include "DwmReadable.hh"
#include "DwmWritable.hh"

namespace Dwm {

  //--------------------------------------------------------------------------
  //!  This class contains a collection of static functions for reading and
  //!  writing simple types, in network byte order (MSB first).  It also
  //!  contains functions to read and write strings.  It also contains
  //!  template functions to read and write STL deques, lists, vectors, maps, 
  //!  multimaps, sets and multisets.  If you have an older version of gcc,
  //!  there are functions for hash_maps and hash_sets.  For gcc-4.2 and
  //!  later, there are functions for tuple, unordered_map and unordered_set.
  //!  We use our member functions to handle reading and writing simple
  //!  types in the containers, and template functions to handle reading 
  //!  and writing other class types.  For a user-defined class, the class
  //!  must implement the Readable and Writable interfaces, since our 
  //!  template functions simply call out to them.  Since templates are
  //!  static polymorphism, you don't need to inherit from Readable and
  //!  Writable, but you must implement the interfaces.
  //--------------------------------------------------------------------------
  class IO
  {
  public:
    //------------------------------------------------------------------------
    //!  Returns the number of bytes that would be written if we called
    //!  Write() for a char.  Should always be 1.
    //------------------------------------------------------------------------
    static uint32_t StreamedLength(char c)
    {
      return(sizeof(c));
    }

    //------------------------------------------------------------------------
    //!  Writes \c c to \c os.  Returns \c os.
    //------------------------------------------------------------------------
    static std::ostream & Write(std::ostream & os, char c);

    //------------------------------------------------------------------------
    //!  Returns the number of bytes that would be written if we called
    //!  Write() for a uint8_t.  Should always be 1.
    //------------------------------------------------------------------------
    static uint32_t StreamedLength(uint8_t c)
    {
      return(sizeof(c));
    }

    //------------------------------------------------------------------------
    //!  Writes \c c to \c os.  Returns \c os.
    //------------------------------------------------------------------------
    static std::ostream & Write(std::ostream & os, uint8_t c);

    //------------------------------------------------------------------------
    //!  Returns the number of bytes that would be written if we called
    //!  one of the Write() members for a bool.
    //------------------------------------------------------------------------
    static uint32_t StreamedLength(bool b)
    {
      return(1);
    }
    
    //------------------------------------------------------------------------
    //!  Writes a bool \c b to an ostream \c os.  Returns \c os.
    //------------------------------------------------------------------------
    static std::ostream & Write(std::ostream & os, bool b);
    
    //------------------------------------------------------------------------
    //!  Returns the number of bytes that would be written if we called
    //!  one of the Write() members for an \c int16_t.
    //------------------------------------------------------------------------
    static uint32_t StreamedLength(int16_t val)
    {
      return(sizeof(val));
    }
    
    //------------------------------------------------------------------------
    //!  Writes \c val to \c os, in network byte order (MSB first).
    //!  Returns \c os.
    //------------------------------------------------------------------------
    static std::ostream & Write(std::ostream & os, int16_t val);

    //------------------------------------------------------------------------
    //!  Returns the number of bytes that would be written if we called
    //!  one of the Write() members for a \c uint16_t.
    //------------------------------------------------------------------------
    static uint32_t StreamedLength(uint16_t val)
    {
      return(sizeof(val));
    }
    
    //------------------------------------------------------------------------
    //!  Writes \c val to \c os, in network byte order (MSB first).
    //!  Returns \c os.
    //------------------------------------------------------------------------
    static std::ostream & Write(std::ostream & os, uint16_t val);

    //------------------------------------------------------------------------
    //!  Returns the number of bytes that would be written if we called
    //!  one of the Write() members for an \c int32_t.
    //------------------------------------------------------------------------
    static uint32_t StreamedLength(int32_t val)
    {
      return(sizeof(val));
    }
    
    //------------------------------------------------------------------------
    //!  Writes \c val to \c os, in network byte order (MSB first).
    //!  Returns \c os.
    //------------------------------------------------------------------------
    static std::ostream & Write(std::ostream & os, int32_t val);

    //------------------------------------------------------------------------
    //!  Returns the number of bytes that would be written if we called
    //!  one of the Write() members for a \c uint32_t.
    //------------------------------------------------------------------------
    static uint32_t StreamedLength(uint32_t val)
    {
      return(sizeof(val));
    }
    
    //------------------------------------------------------------------------
    //!  Writes \c val to \c os, in network byte order (MSB first).
    //!  Returns \c os.
    //------------------------------------------------------------------------
    static std::ostream & Write(std::ostream & os, uint32_t val);

    //------------------------------------------------------------------------
    //!  Returns the number of bytes that would be written if we called
    //!  one of the Write() members for an \c int64_t.
    //------------------------------------------------------------------------
    static uint32_t StreamedLength(int64_t val)
    {
      return(sizeof(val));
    }
    
    //------------------------------------------------------------------------
    //!  Writes \c val to \c os, in network byte order (MSB first).
    //!  Returns \c os.
    //------------------------------------------------------------------------
    static std::ostream & Write(std::ostream & os, const int64_t & val);

    //------------------------------------------------------------------------
    //!  Returns the number of bytes that would be written if we called
    //!  one of the Write() members for a \c uint64_t.
    //------------------------------------------------------------------------
    static uint32_t StreamedLength(uint64_t val)
    {
      return(sizeof(val));
    }
    
    //------------------------------------------------------------------------
    //!  Writes \c val to \c os, in network byte order (MSB first).
    //!  Returns \c os.
    //------------------------------------------------------------------------
    static std::ostream & Write(std::ostream & os, const uint64_t & val);

    //------------------------------------------------------------------------
    //!  Returns the number of bytes that should be written if we call one
    //!  of the Write() members for a float.
    //------------------------------------------------------------------------
    static uint32_t StreamedLength(float val)
    {
      return(4);
    }
    
    //------------------------------------------------------------------------
    //!  Writes \c val to \c os, in IEEE format (see RFC 1832
    //!  and/or ANSI/IEEE Standard 754-1985).  Returns \c os.
    //------------------------------------------------------------------------
    static std::ostream & Write(std::ostream & os, float val);

    //------------------------------------------------------------------------
    //!  Returns the number of bytes that should be written if we call one
    //!  of the Write() members for a double.
    //------------------------------------------------------------------------
    static uint32_t StreamedLength(double val)
    {
      return(8);
    }

    //------------------------------------------------------------------------
    //!  Writes \c val to \c os, in IEEE format (see RFC 1832
    //!  and/or ANSI/IEEE Standard 754-1985).  Returns \c os.
    //------------------------------------------------------------------------
    static std::ostream & Write(std::ostream & os, const double & val);

    //------------------------------------------------------------------------
    //!  Returns the number of bytes that should be written if we call one
    //!  of the Write() members for a string.
    //------------------------------------------------------------------------
    static uint32_t StreamedLength(const std::string & s)
    {
      return(sizeof(uint32_t) + s.size());
    }

    //------------------------------------------------------------------------
    //!  Writes \c s to \c os.  Note that the length of \c s is written
    //!  first, as an unsigned 32-bit number in network byte order
    //!  (MSB first).  Hence at least 5 bytes will always be written;
    //!  4 for the length and 1 for the terminating NULL character of
    //!  an empty string.
    //------------------------------------------------------------------------
    static std::ostream & Write(std::ostream & os, const std::string & s);
    
    //------------------------------------------------------------------------
    //!  Reads \c c from \c is.  Returns \c is.
    //------------------------------------------------------------------------
    static std::istream & Read(std::istream & is, char & c);

    //------------------------------------------------------------------------
    //!  Reads \c c from \c is.  Returns \c is.
    //------------------------------------------------------------------------
    static std::istream & Read(std::istream & is, uint8_t & c);

    //------------------------------------------------------------------------
    //!  Reads \c b from \c is.  Returns \c is.
    //------------------------------------------------------------------------
    static std::istream & Read(std::istream & is, bool & b);
    
    //------------------------------------------------------------------------
    //!  Reads \c val from \c is, in network byte order (MSB first).
    //!  Returns \c is.
    //------------------------------------------------------------------------
    static std::istream & Read(std::istream & is, int16_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c is, in network byte order (MSB first).
    //!  Returns \c is.
    //------------------------------------------------------------------------
    static std::istream & Read(std::istream & is, uint16_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c is, in network byte order (MSB first).
    //!  Returns \c is.
    //------------------------------------------------------------------------
    static std::istream & Read(std::istream & is, int32_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c is, in network byte order (MSB first).
    //!  Returns \c is.
    //------------------------------------------------------------------------
    static std::istream & Read(std::istream & is, uint32_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c is, in network byte order (MSB first).
    //!  Returns \c is.
    //------------------------------------------------------------------------
    static std::istream & Read(std::istream & is, int64_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c is, in network byte order (MSB first).
    //!  Returns \c is.
    //------------------------------------------------------------------------
    static std::istream & Read(std::istream & is, uint64_t & val);
    
    //------------------------------------------------------------------------
    //!  Reads \c val from \c is, in IEEE format (see RFC 1832
    //!  and/or ANSI/IEEE Standard 754-1985).  Returns \c is.
    //------------------------------------------------------------------------
    static std::istream & Read(std::istream & is, float & val);
    
    //------------------------------------------------------------------------
    //!  Reads \c val from \c is, in IEEE format (see RFC 1832
    //!  and/or ANSI/IEEE Standard 754-1985).  Returns \c is.
    //------------------------------------------------------------------------
    static std::istream & Read(std::istream & is, double & val);
    
    //------------------------------------------------------------------------
    //!  Reads string \c s from \c is.  Since we write strings with a 
    //!  32-bit length value preceding, and always write the terminating
    //!  NULL, this function will always read at least 5 bytes on success.
    //------------------------------------------------------------------------
    static std::istream & Read(std::istream & is, std::string & s);
    
    //------------------------------------------------------------------------
    //!  Writes \c c to \c fd.  Returns the number of bytes written on
    //!  success, -1 on failure.
    //------------------------------------------------------------------------
    static ssize_t Write(int fd, char c);

    //------------------------------------------------------------------------
    //!  Writes \c c to \c fd.  Returns the number of bytes written (1) on
    //!  success, less on failure.
    //------------------------------------------------------------------------
    static ssize_t Write(int fd, uint8_t c);

    //------------------------------------------------------------------------
    //!  Writes \c b to \c fd.  Returns the number of bytes written (1) on
    //!  success, less on failure.
    //------------------------------------------------------------------------
    static ssize_t Write(int fd, bool b);
    
    //------------------------------------------------------------------------
    //!  Writes \c val to \c fd, in network byte order (MSB first).
    //!  Returns the number of bytes written (2) on success, less on
    //!  failure.
    //------------------------------------------------------------------------
    static ssize_t Write(int fd, int16_t val);

    //------------------------------------------------------------------------
    //!  Writes \c val to \c fd, in network byte order (MSB first).
    //!  Returns the number of bytes written (2) on success, less on
    //!  failure.
    //------------------------------------------------------------------------
    static ssize_t Write(int fd, uint16_t val);

    //------------------------------------------------------------------------
    //!  Writes \c val to \c fd, in network byte order (MSB first).
    //!  Returns the number of bytes written (4) on success, less on
    //!  failure.
    //------------------------------------------------------------------------
    static ssize_t Write(int fd, int32_t val);

    //------------------------------------------------------------------------
    //!  Writes \c val to \c fd, in network byte order (MSB first).
    //!  Returns the number of bytes written (4) on success, less on
    //!  failure.
    //------------------------------------------------------------------------
    static ssize_t Write(int fd, uint32_t val);

    //------------------------------------------------------------------------
    //!  Writes \c val to \c fd, in network byte order (MSB first).
    //!  Returns the number of bytes written (8) on success, less on
    //!  failure.
    //------------------------------------------------------------------------
    static ssize_t Write(int fd, const int64_t & val);

    //------------------------------------------------------------------------
    //!  Writes \c val to \c fd, in network byte order (MSB first).
    //!  Returns the number of bytes written (8) on success, less on
    //!  failure.
    //------------------------------------------------------------------------
    static ssize_t Write(int fd, const uint64_t & val);

    //------------------------------------------------------------------------
    //!  Writes \c val to \c fd, in IEEE format (see RFC 1832
    //!  and/or ANSI/IEEE Standard 754-1985).  Returns the number of bytes
    //!  written (4) on success, less on failure.
    //------------------------------------------------------------------------
    static ssize_t Write(int fd, float val);
    
    //------------------------------------------------------------------------
    //!  Writes \c val to \c fd, in IEEE format (see RFC 1832
    //!  and/or ANSI/IEEE Standard 754-1985).  Returns the number of bytes
    //!  written (8) on success, less on failure.
    //------------------------------------------------------------------------
    static ssize_t Write(int fd, const double & val);
    
    //------------------------------------------------------------------------
    //!  Writes \c s to \c fd.  Returns the number of bytes written on
    //!  success, -1 on failure.  Note that a 32-bit value is written first,
    //!  holding the length of the string.  The terminating NULL is also
    //!  written.  Hence, on success this will always return a value of
    //!  5 or greater.
    //------------------------------------------------------------------------
    static ssize_t Write(int fd, const std::string & s);
    
    //------------------------------------------------------------------------
    //!  Reads \c c from \c fd.  Returns the number of bytes read (1) on 
    //!  success, less on failure.
    //------------------------------------------------------------------------
    static ssize_t Read(int fd, char & c);

    //------------------------------------------------------------------------
    //!  Reads \c c from \c fd.  Returns the number of bytes read (1) on 
    //!  success, less on failure.
    //------------------------------------------------------------------------
    static ssize_t Read(int fd, uint8_t & c);

    //------------------------------------------------------------------------
    //!  Reads \c b from \c fd.  Returns the number of bytes read (1) on
    //!  success, less on failure.
    //------------------------------------------------------------------------
    static ssize_t Read(int fd, bool & b);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c fd, in network byte order (MSB first). 
    //!  Returns number of bytes read (2) on success, less on failure.
    //------------------------------------------------------------------------
    static ssize_t Read(int fd, int16_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c fd, in network byte order (MSB first). 
    //!  Returns number of bytes read (2) on success, less on failure.
    //------------------------------------------------------------------------
    static ssize_t Read(int fd, uint16_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c fd, in network byte order (MSB first). 
    //!  Returns number of bytes read (4) on success, less on failure.
    //------------------------------------------------------------------------
    static ssize_t Read(int fd, int32_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c fd, in network byte order (MSB first). 
    //!  Returns number of bytes read (4) on success, less on failure.
    //------------------------------------------------------------------------
    static ssize_t Read(int fd, uint32_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c fd, in network byte order (MSB first). 
    //!  Returns number of bytes read (8) on success, less on failure.
    //------------------------------------------------------------------------
    static ssize_t Read(int fd, int64_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c fd, in network byte order (MSB first). 
    //!  Returns number of bytes read (8) on success, less on failure.
    //------------------------------------------------------------------------
    static ssize_t Read(int fd, uint64_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c fd, in IEEE format (see RFC 1832 and/or 
    //!  ANSI/IEEE Standard 754-1985).  Returns number of bytes read (4)
    //!  on success, less on failure.
    //------------------------------------------------------------------------
    static ssize_t Read(int fd, float & val);
    
    //------------------------------------------------------------------------
    //!  Reads \c val from \c fd, in IEEE format (see RFC 1832 and/or 
    //!  ANSI/IEEE Standard 754-1985).  Returns number of bytes read (8)
    //!  on success, less on failure.
    //------------------------------------------------------------------------
    static ssize_t Read(int fd, double & val);
    
    //------------------------------------------------------------------------
    //!  Reads \c s from \c fd.  Returns the number of bytes read on success,
    //!  -1 on failure.  Since we write strings with a 32-bit unsigned length
    //!  value preceding the actual string, and always have a terminating
    //!  NULL, this always reads at least 5 bytes on success.
    //------------------------------------------------------------------------
    static ssize_t Read(int fd, std::string & s);
    
    //------------------------------------------------------------------------
    //!  Writes \c c to \c f.  Returns the number of bytes written (1) on 
    //!  success.
    //------------------------------------------------------------------------
    static size_t Write(FILE * f, char c);

    //------------------------------------------------------------------------
    //!  Writes \c c to \c f.  Returns the number of bytes written (1) on
    //!  success.
    //------------------------------------------------------------------------
    static size_t Write(FILE * f, uint8_t c);

    //------------------------------------------------------------------------
    //!  Writes \c b to \c f.  Returns the number of bytes written (1) on
    //!  success.
    //------------------------------------------------------------------------
    static size_t Write(FILE * f, bool b);

    //------------------------------------------------------------------------
    //!  Writes \c val to \c f, in network byte order (MSB first).  Returns
    //!  the number of bytes written (2) on success.
    //------------------------------------------------------------------------
    static size_t Write(FILE * f, int16_t val);

    //------------------------------------------------------------------------
    //!  Writes \c val to \c f, in network byte order (MSB first).  Returns
    //!  the number of bytes written (2) on success.
    //------------------------------------------------------------------------
    static size_t Write(FILE * f, uint16_t val);

    //------------------------------------------------------------------------
    //!  Writes \c val to \c f, in network byte order (MSB first).  Returns
    //!  the number of bytes written (4) on success.
    //------------------------------------------------------------------------
    static size_t Write(FILE * f, int32_t val);

    //------------------------------------------------------------------------
    //!  Writes \c val to \c f, in network byte order (MSB first).  Returns
    //!  the number of bytes written (4) on success.
    //------------------------------------------------------------------------
    static size_t Write(FILE * f, uint32_t val);

    //------------------------------------------------------------------------
    //!  Writes \c val to \c f, in network byte order (MSB first).  Returns
    //!  the number of bytes written (8) on success.
    //------------------------------------------------------------------------
    static size_t Write(FILE * f, const int64_t & val);

    //------------------------------------------------------------------------
    //!  Writes \c val to \c f, in network byte order (MSB first).  Returns
    //!  the number of bytes written (8) on success.
    //------------------------------------------------------------------------
    static size_t Write(FILE * f, const uint64_t & val);

    //------------------------------------------------------------------------
    //!  Writes \c val to \c f, in IEEE format (see RFC 1832 and/or 
    //!  ANSI/IEEE Standard 754-1985).  Returns the number of bytes 
    //!  written (4 on success).
    //------------------------------------------------------------------------
    static size_t Write(FILE * f, float val);
    
    //------------------------------------------------------------------------
    //!  Writes \c val to \c f, in IEEE format (see RFC 1832 and/or 
    //!  ANSI/IEEE Standard 754-1985).  Returns the number of bytes 
    //!  written (8 on success).
    //------------------------------------------------------------------------
    static size_t Write(FILE * f, const double & val);
    
    //------------------------------------------------------------------------
    //!  Writes \c s to \c f.  Returns the number of bytes written.  Note
    //!  that we first write a 32-bit length value, then the string itself
    //!  (with terminating NULL).  Hence a successful write will always
    //!  return a value of 4 + \c s.length() + 1.
    //------------------------------------------------------------------------
    static size_t Write(FILE * f, const std::string & s);
    
    //------------------------------------------------------------------------
    //!  Reads \c c from \c f.  Returns the number of bytes read (1 on
    //!  success, 0 on failure).
    //------------------------------------------------------------------------
    static size_t Read(FILE * f, char & c);

    //------------------------------------------------------------------------
    //!  Reads \c c from \c f.  Returns the number of bytes read (1 on
    //!  success).
    //------------------------------------------------------------------------
    static size_t Read(FILE * f, uint8_t & c);

    //------------------------------------------------------------------------
    //!  Reads \c b from \c f.  Returns the number of bytes read (1 on
    //!  success).
    //------------------------------------------------------------------------
    static size_t Read(FILE * f, bool & b);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c f, in network byte order (MSB first).
    //!  Returns the number of bytes read (2 on success).
    //------------------------------------------------------------------------
    static size_t Read(FILE * f, int16_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c f, in network byte order (MSB first).
    //!  Returns the number of bytes read (2 on success).
    //------------------------------------------------------------------------
    static size_t Read(FILE * f, uint16_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c f, in network byte order (MSB first).
    //!  Returns the number of bytes read (4 on success).
    //------------------------------------------------------------------------
    static size_t Read(FILE * f, int32_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c f, in network byte order (MSB first).
    //!  Returns the number of bytes read (4 on success).
    //------------------------------------------------------------------------
    static size_t Read(FILE * f, uint32_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c f, in network byte order (MSB first).
    //!  Returns the number of bytes read (8 on success).
    //------------------------------------------------------------------------
    static size_t Read(FILE * f, int64_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c f, in network byte order (MSB first).
    //!  Returns the number of bytes read (8 on success).
    //------------------------------------------------------------------------
    static size_t Read(FILE * f, uint64_t & val);

    //------------------------------------------------------------------------
    //!  Reads \c val from \c f, in IEEE format (see RFC 1832 and/or
    //!  ANSI/IEEE Standard 754-1985).  Returns the number of bytes read
    //! (4 on success).
    //------------------------------------------------------------------------
    static size_t Read(FILE * f, float & val);
    
    //------------------------------------------------------------------------
    //!  Reads \c val from \c f, in IEEE format (see RFC 1832 and/or
    //!  ANSI/IEEE Standard 754-1985).  Returns the number of bytes read
    //! (8 on success).
    //------------------------------------------------------------------------
    static size_t Read(FILE * f, double & val);
    
    //------------------------------------------------------------------------
    //!  Reads \c s from \c f.  Since we write strings with a 32-bit length
    //!  value preceding, and always write the terminating NULL, this
    //!  function will always return a value of 5 or greater on success.
    //------------------------------------------------------------------------
    static size_t Read(FILE * f, std::string & s);

    //------------------------------------------------------------------------
    //!  Returns the number of bytes that should be written if we call one
    //!  of the Write() members for a Writable \c val.
    //------------------------------------------------------------------------
    static uint32_t StreamedLength(const Writable & val)
    {
      return(val.StreamedLength());
    }
    
    //------------------------------------------------------------------------
    //!  Wrapper function to write a Writable object to an ostream.
    //------------------------------------------------------------------------
    static std::ostream & Write(std::ostream & os, const Writable & val)
    {
      return(val.Write(os));
    }

    //------------------------------------------------------------------------
    //!  Wrapper function to write a Writable object to a descriptor.
    //------------------------------------------------------------------------
    static ssize_t Write(int fd, const Writable & val)
    {
      return(val.Write(fd));
    }

    //------------------------------------------------------------------------
    //!  Wrapper function to write a Writable object to a FILE pointer.
    //------------------------------------------------------------------------
    static size_t Write(FILE * f, const Writable & val)
    {
      return(val.Write(f));
    }

    //------------------------------------------------------------------------
    //!  Wrapper function to read a Readable object from an istream.
    //------------------------------------------------------------------------
    static std::istream & Read(std::istream & is, Readable & val)
    {
      return(val.Read(is));
    }
    
    //------------------------------------------------------------------------
    //!  Wrapper function to read a Readable object from a descriptor.
    //------------------------------------------------------------------------
    static ssize_t Read(int fd, Readable & val)
    {
      return(val.Read(fd));
    }
    
    //------------------------------------------------------------------------
    //!  Wrapper function to read a Readable object from a FILE pointer.
    //------------------------------------------------------------------------
    static size_t Read(FILE *f, Readable & val)
    {
      return(val.Read(f));
    }
    
    //------------------------------------------------------------------------
    //!  Returns the number of bytes that should be written if we call one
    //!  of the Write() members for a pair<_firstT, _secondT>
    //------------------------------------------------------------------------
    template <typename _firstT, typename _secondT>
    static uint32_t StreamedLength(const std::pair<_firstT, _secondT> & p)
    {
      return(StreamedLength(p.first) + StreamedLength(p.second));
    }
    
    //------------------------------------------------------------------------
    //!  Reads a pair<_firstT,_secondT> from an istream.  Returns the istream.
    //------------------------------------------------------------------------
    template <typename _firstT, typename _secondT>
    static std::istream & Read(std::istream & is,
                               std::pair<_firstT, _secondT> & p)
    {
      if (is) {
        if (Read(is, p.first))
          Read(is, p.second);
      }
      return(is);
    }
    
    //------------------------------------------------------------------------
    //!  Writes a pair<_firstT,_secondT> to an ostream.  Returns the ostream.
    //------------------------------------------------------------------------
    template <typename _firstT, typename _secondT>
    static std::ostream & Write(std::ostream & os,
                                const std::pair<_firstT,_secondT> & p)
    {
      if (os) {
        if (Write(os, p.first)) {
          Write(os, p.second);
        }
      }
      return(os);
    }

    //------------------------------------------------------------------------
    //!  Reads a pair<_firstT,_secondT> from a FILE pointer.  Returns 1 on
    //!  success, 0 on failure.
    //------------------------------------------------------------------------
    template <typename _firstT, typename _secondT>
    static size_t Read(FILE *f, std::pair<_firstT,_secondT> & p)
    {
      size_t  rc = 0;
      if (f)
        if (Read(f, p.first) > 0)
          if (Read(f, p.second) > 0)
            rc = 1;
      return(rc);
    }
    
    //------------------------------------------------------------------------
    //!  Writes a pair<_firstT,_secondT> to a FILE pointer.  Returns 1
    //!  on success, 0 on failure.
    //------------------------------------------------------------------------
    template <typename _firstT, typename _secondT>
    static size_t Write(FILE *f, const std::pair<_firstT,_secondT> & p)
    {
      size_t  rc = 0;
      if (f)
        if (Write(f, p.first) > 0)
          if (Write(f, p.second) > 0)
            rc = 1;
      return(rc);
    }

    //------------------------------------------------------------------------
    //!  Reads a pair<_firstT,_secondT> from a file descriptor.  Returns
    //!  the number of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _firstT, typename _secondT>
    static ssize_t Read(int fd, std::pair<_firstT, _secondT> & p)
    {
      ssize_t  rc = -1;
      if (fd >= 0) {
        ssize_t  bytesRead = Read(fd, p.first);
        if (bytesRead > 0) {
          rc = bytesRead;
          bytesRead = Read(fd, p.second);
          if (bytesRead > 0)
            rc += bytesRead;
          else
            rc = -1;
        }
      }
      return(rc);
    }
    
    //------------------------------------------------------------------------
    //!  Writes a pair<_firstT,_secondT> to a file descriptor.  Returns
    //!  the number of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _firstT, typename _secondT>
    static ssize_t Write(int fd, const std::pair<_firstT,_secondT> & p)
    {
      ssize_t  rc = -1;
      if (fd >= 0) {
        ssize_t  bytesWritten = Write(fd, p.first);
        if (bytesWritten > 0) {
          rc = bytesWritten;
          bytesWritten = Write(fd, p.second);
          if (bytesWritten > 0) {
            rc += bytesWritten;
          }
          else {
            rc = -1;
          }
        }
      }
      return(rc);
    }
         
    //------------------------------------------------------------------------
    //!  Returns the number of bytes that should be written if we call one
    //!  of the Write() members for a map<_keyT, _valueT>
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Compare, typename _Alloc>
    static uint32_t
    StreamedLength(const std::map<_keyT, _valueT, _Compare, _Alloc> & m)
    {
      return(ContainerStreamedLength<std::map<_keyT, _valueT, _Compare, _Alloc> >(m));
    }
    
    //------------------------------------------------------------------------
    //!  Reads a map<_keyT,_valueT> from an istream.  Returns the istream.
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Compare, typename _Alloc>
    static std::istream & Read(std::istream & is,
                               std::map<_keyT, _valueT, _Compare, _Alloc> & m)
    {
      return(PairAssocContRead<std::map<_keyT, _valueT, _Compare, _Alloc> >(is, m));
    }

    //------------------------------------------------------------------------
    //!  Writes a map<_keyT,_valueT> to an ostream.  Returns the ostream.
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Compare, typename _Alloc>
    static std::ostream & Write(std::ostream & os,
                                const std::map<_keyT,_valueT, _Compare, _Alloc> & m)
    {
      return(ContainerWrite<std::map<_keyT,_valueT,_Compare,_Alloc> >(os, m));
    }
    
    //------------------------------------------------------------------------
    //!  Reads a map<_keyT,_valueT> from a FILE pointer.  Returns 1
    //!  on success, 0 on failure.
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Compare, typename _Alloc>
    static size_t Read(FILE *f, std::map<_keyT, _valueT, _Compare, _Alloc> & m)
    {
      return(PairAssocContRead<std::map<_keyT, _valueT, _Compare, _Alloc> >(f, m));
    }

    //------------------------------------------------------------------------
    //!  Writes a map<_keyT,_valueT> to a FILE pointer.  Returns 1 on
    //!  success, 0 on failure.
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Compare, typename _Alloc>
    static size_t
    Write(FILE *f, const std::map<_keyT, _valueT, _Compare, _Alloc> & m)
    {
      return(ContainerWrite<std::map<_keyT,_valueT> >(f, m));
    }

    //------------------------------------------------------------------------
    //!  Reads a map<_keyT,_valueT> from a file descriptor.  Returns the
    //!  number of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Compare, typename _Alloc>
    static ssize_t Read(int fd, std::map<_keyT, _valueT, _Compare, _Alloc> & m)
    {
      return(PairAssocContRead<std::map<_keyT, _valueT, _Compare, _Alloc> >(fd, m));
    }

    //------------------------------------------------------------------------
    //!  Writes a map<_keyT,_valueT> to a file descriptor.  Returns the
    //!  number of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template<typename _keyT, typename _valueT, 
             typename _Compare, typename _Alloc>
    static ssize_t 
    Write(int fd, const std::map<_keyT,_valueT,_Compare,_Alloc> & m)
    {
      return(ContainerWrite<std::map<_keyT,_valueT,_Compare,_Alloc> >(fd, m));
    }
         
    //------------------------------------------------------------------------
    //!  Returns the number of bytes that should be written if we call one
    //!  of the Write() members for a multimap<_keyT, _valueT>
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Compare, typename _Alloc>
    static uint32_t
    StreamedLength(const std::multimap<_keyT,_valueT,_Compare,_Alloc> & m)
    {
      return(ContainerStreamedLength<std::multimap<_keyT,_valueT,_Compare,_Alloc> >(m));
    }

    //------------------------------------------------------------------------
    //!  Reads a multimap<_keyT,_valueT> from an istream.  Returns the 
    //!  istream.
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Compare, typename _Alloc>
    static std::istream &
    Read(std::istream & is, std::multimap<_keyT,_valueT,_Compare,_Alloc> & m)
    {
      return(PairAssocContRead<std::multimap<_keyT,_valueT,_Compare,_Alloc> >(is, m));
    }

    //------------------------------------------------------------------------
    //!  Writes a multimap<_keyT,_valueT> to an ostream.  Returns the ostream.
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Compare, typename _Alloc>
    static std::ostream & 
    Write(std::ostream & os,
          const std::multimap<_keyT,_valueT, _Compare, _Alloc> & m)
    {
      return(ContainerWrite<std::multimap<_keyT,_valueT,_Compare,_Alloc> >(os, m));
    }

    //------------------------------------------------------------------------
    //!  Reads a multimap<_keyT,_valueT> from a FILE pointer.  Returns 1
    //!  on success, 0 on failure.
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Compare, typename _Alloc>
    static size_t
    Read(FILE *f, std::multimap<_keyT, _valueT, _Compare, _Alloc> & m)
    {
      return(PairAssocContRead<std::multimap<_keyT,_valueT,_Compare,_Alloc> >(f, m));
    }

    //------------------------------------------------------------------------
    //!  Writes a multimap<_keyT,_valueT> to a FILE pointer.  Returns 1 on
    //!  success, 0 on failure.
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Compare, typename _Alloc>
    static size_t 
    Write(FILE *f, const std::multimap<_keyT,_valueT, _Compare, _Alloc> & m)
    {
      return(ContainerWrite<std::multimap<_keyT,_valueT,_Compare,_Alloc> >(f, m));
    }

    //------------------------------------------------------------------------
    //!  Reads a multimap<_keyT,_valueT> from a file descriptor.  Returns the
    //!  number of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Compare, typename _Alloc>
    static ssize_t 
    Read(int fd, std::multimap<_keyT, _valueT, _Compare, _Alloc> & m)
    {
      return(PairAssocContRead<std::multimap<_keyT,_valueT,_Compare,_Alloc> >(fd, m));
    }

    //------------------------------------------------------------------------
    //!  Writes a multimap<_keyT,_valueT> to a file descriptor.  Returns 
    //!  the number of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Compare, typename _Alloc>
    static ssize_t 
    Write(int fd, const std::multimap<_keyT,_valueT, _Compare, _Alloc> & m)
    {
      return(ContainerWrite<std::multimap<_keyT,_valueT,_Compare,_Alloc> >(fd, m));
    }
    
    //------------------------------------------------------------------------
    //!  Returns the number of bytes that should be written if we call one
    //!  of the Write() members for a vector<_valueT>
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static uint32_t StreamedLength(const std::vector<_valueT, _Alloc> & v)
    {
      return(ContainerStreamedLength<std::vector<_valueT, _Alloc> >(v));
    }

    //------------------------------------------------------------------------
    //!  Reads a vector<_valueT> from an istream.  Returns the istream.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static std::istream & Read(std::istream & is,
                               std::vector<_valueT, _Alloc> & v)
    {
      return(ContainerRead<std::vector<_valueT, _Alloc> >(is, v));
    }

    //------------------------------------------------------------------------
    //!  Writes a vector<_valueT> to an ostream.  Returns the ostream.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static std::ostream & Write(std::ostream & os,
                                const std::vector<_valueT, _Alloc> & v)
    {
      return(ContainerWrite<std::vector<_valueT, _Alloc> >(os, v));
    }

    //------------------------------------------------------------------------
    //!  Reads a vector<_valueT> from a FILE pointer.  Returns 1 on success,
    //!  0 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static size_t Read(FILE *f, std::vector<_valueT, _Alloc> & v)
    {
      return(ContainerRead<std::vector<_valueT, _Alloc> >(f, v));
    }
    
    //------------------------------------------------------------------------
    //!  Writes a vector<_valueT> to a FILE pointer.  Returns 1 on success,
    //!  0 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static size_t Write(FILE *f, const std::vector<_valueT, _Alloc> & v)
    {
      return(ContainerWrite<std::vector<_valueT, _Alloc> >(f, v));
    }

    //------------------------------------------------------------------------
    //!  Reads a vector<_valueT> from a file descriptor.  Returns the number
    //!  of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static ssize_t Read(int fd, std::vector<_valueT, _Alloc> & v)
    {
      return(ContainerRead<std::vector<_valueT, _Alloc> >(fd, v));
    }

    //------------------------------------------------------------------------
    //!  Writes a vector<_valueT> to a file descriptor.  Returns the number
    //!  of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static ssize_t Write(int fd, const std::vector<_valueT, _Alloc> & v)
    {
      return(ContainerWrite<std::vector<_valueT, _Alloc> >(fd, v));
    }

    //------------------------------------------------------------------------
    //!  Returns the number of bytes that should be written if we call one
    //!  of the Write() members for a deque<_valueT>
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static uint32_t StreamedLength(const std::deque<_valueT, _Alloc> & d)
    {
      return(ContainerStreamedLength<std::deque<_valueT, _Alloc> >(d));
    }

    //------------------------------------------------------------------------
    //!  Reads a deque<_valueT> from an istream.  Returns the istream.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static std::istream & Read(std::istream & is,
                               std::deque<_valueT, _Alloc> & d)
    {
      return(ContainerRead<std::deque<_valueT, _Alloc> >(is, d));
    }

    //------------------------------------------------------------------------
    //!  Writes a deque<_valueT> to an ostream.  Returns the ostream.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static std::ostream & Write(std::ostream & os,
                                const std::deque<_valueT, _Alloc> & d)
    {
      return(ContainerWrite<std::deque<_valueT, _Alloc> >(os, d));
    }

    //------------------------------------------------------------------------
    //!  Reads a deque<_valueT> from a FILE pointer.  Returns 1 on success,
    //!  0 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static size_t Read(FILE *f, std::deque<_valueT, _Alloc> & d)
    {
      return(ContainerRead<std::deque<_valueT, _Alloc> >(f, d));
    }
    
    //------------------------------------------------------------------------
    //!  Writes a deque<_valueT> to a FILE pointer.  Returns 1 on success,
    //!  0 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static size_t Write(FILE *f, const std::deque<_valueT, _Alloc> & d)
    {
      return(ContainerWrite<std::deque<_valueT, _Alloc> >(f, d));
    }

    //------------------------------------------------------------------------
    //!  Reads a deque<_valueT> from a file descriptor.  Returns the number
    //!  of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static ssize_t Read(int fd, std::deque<_valueT, _Alloc> & d)
    {
      return(ContainerRead<std::deque<_valueT, _Alloc> >(fd, d));
    }

    //------------------------------------------------------------------------
    //!  Writes a deque<_valueT> to a file descriptor.  Returns the number
    //!  of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static ssize_t Write(int fd, const std::deque<_valueT, _Alloc> & d)
    {
      return(ContainerWrite<std::deque<_valueT, _Alloc> >(fd, d));
    }

    //------------------------------------------------------------------------
    //!  Returns the number of bytes that should be written if we call one
    //!  of the Write() members for a list<_valueT>
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static uint32_t StreamedLength(const std::list<_valueT, _Alloc> & l)
    {
      return(ContainerStreamedLength<std::list<_valueT, _Alloc> >(l));
    }

    //------------------------------------------------------------------------
    //!  Reads a list<_valueT> from an istream.  Returns the istream.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static std::istream & Read(std::istream & is,
                               std::list<_valueT, _Alloc> & l)
    {
      return(ContainerRead<std::list<_valueT, _Alloc> >(is, l));
    }
    
    //------------------------------------------------------------------------
    //!  Writes a list<_valueT> to an ostream.  Returns the ostream.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static std::ostream & Write(std::ostream & os,
                                const std::list<_valueT, _Alloc> & l)
    {
      return(ContainerWrite<std::list<_valueT, _Alloc> >(os, l));
    }
    
    //------------------------------------------------------------------------
    //!  Reads a list<_valueT> from a FILE pointer.  Returns 1 on success,
    //!  0 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static size_t Read(FILE *f, std::list<_valueT, _Alloc> & l)
    {
      return(ContainerRead<std::list<_valueT, _Alloc> >(f, l));
    }
    
    //------------------------------------------------------------------------
    //!  Writes a list<_valueT> to a FILE pointer.  Returns 1 on success,
    //!  0 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static size_t Write(FILE *f, const std::list<_valueT, _Alloc> & l)
    {
      return(ContainerWrite<std::list<_valueT, _Alloc> >(f, l));
    }

    //------------------------------------------------------------------------
    //!  Reads a list<_valueT> from a file descriptor.  Returns the number
    //!  of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static ssize_t Read(int fd, std::list<_valueT, _Alloc> & l)
    {
      return(ContainerRead<std::list<_valueT, _Alloc> >(fd, l));
    }

    //------------------------------------------------------------------------
    //!  Writes a list<_valueT> to a file descriptor.  Returns the number
    //!  of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Alloc>
    static ssize_t Write(int fd, const std::list<_valueT, _Alloc> & l)
    {
      return(ContainerWrite<std::list<_valueT, _Alloc> >(fd, l));
    }

    //------------------------------------------------------------------------
    //!  Returns the number of bytes that should be written if we call one
    //!  of the Write() members for a set<_valueT>
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Compare, typename _Alloc>
    static uint32_t 
    StreamedLength(const std::set<_valueT, _Compare, _Alloc> & l)
    {
      return(ContainerStreamedLength<std::set<_valueT, _Compare, _Alloc> >(l));
    }

    //------------------------------------------------------------------------
    //!  Reads a set<_valueT> from an istream.  Returns the istream.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Compare, typename _Alloc>
    static std::istream & Read(std::istream & is,
                               std::set<_valueT, _Compare, _Alloc> & l)
    {
      return(ContainerRead<std::set<_valueT, _Compare, _Alloc> >(is, l));
    }
    
    //------------------------------------------------------------------------
    //!  Writes a set<_valueT> to an ostream.  Returns the ostream.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Compare, typename _Alloc>
    static std::ostream & Write(std::ostream & os,
                                const std::set<_valueT, _Compare, _Alloc> & l)
    {
      return(ContainerWrite<std::set<_valueT, _Compare, _Alloc> >(os, l));
    }
    
    //------------------------------------------------------------------------
    //!  Reads a set<_valueT> from a FILE pointer.  Returns 1 on success,
    //!  0 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Compare, typename _Alloc>
    static size_t Read(FILE *f, std::set<_valueT, _Compare, _Alloc> & l)
    {
      return(ContainerRead<std::set<_valueT, _Compare, _Alloc> >(f, l));
    }
    
    //------------------------------------------------------------------------
    //!  Writes a set<_valueT> to a FILE pointer.  Returns 1 on success,
    //!  0 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Compare, typename _Alloc>
    static size_t Write(FILE *f, const std::set<_valueT, _Compare, _Alloc> & l)
    {
      return(ContainerWrite<std::set<_valueT, _Compare, _Alloc> >(f, l));
    }

    //------------------------------------------------------------------------
    //!  Reads a set<_valueT> from a file descriptor.  Returns the number
    //!  of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Compare, typename _Alloc>
    static ssize_t Read(int fd, std::set<_valueT, _Compare, _Alloc> & l)
    {
      return(ContainerRead<std::set<_valueT, _Compare, _Alloc> >(fd, l));
    }

    //------------------------------------------------------------------------
    //!  Writes a set<_valueT> to a file descriptor.  Returns the number
    //!  of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Compare, typename _Alloc>
    static ssize_t Write(int fd, const std::set<_valueT, _Compare, _Alloc> & l)
    {
      return(ContainerWrite<std::set<_valueT, _Compare, _Alloc> >(fd, l));
    }

    //------------------------------------------------------------------------
    //!  Returns the number of bytes that should be written if we call one
    //!  of the Write() members for a multiset<_valueT>
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Compare, typename _Alloc>
    static uint32_t
    StreamedLength(const std::multiset<_valueT, _Compare, _Alloc> & l)
    {
      return(ContainerStreamedLength<std::multiset<_valueT, _Compare, _Alloc> >(l));
    }

    //------------------------------------------------------------------------
    //!  Reads a multiset<_valueT> from an istream.  Returns the istream.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Compare, typename _Alloc>
    static std::istream & Read(std::istream & is,
                               std::multiset<_valueT, _Compare, _Alloc> & l)
    {
      return(ContainerRead<std::multiset<_valueT, _Compare, _Alloc> >(is, l));
    }
    
    //------------------------------------------------------------------------
    //!  Writes a multiset<_valueT> to an ostream.  Returns the ostream.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Compare, typename _Alloc>
    static std::ostream & 
    Write(std::ostream & os, 
          const std::multiset<_valueT, _Compare, _Alloc> & l)
    {
      return(ContainerWrite<std::multiset<_valueT, _Compare, _Alloc> >(os, l));
    }
    
    //------------------------------------------------------------------------
    //!  Reads a multiset<_valueT> from a FILE pointer.  Returns 1 on success,
    //!  0 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Compare, typename _Alloc>
    static size_t Read(FILE *f, std::multiset<_valueT, _Compare, _Alloc> & l)
    {
      return(ContainerRead<std::multiset<_valueT, _Compare, _Alloc> >(f, l));
    }
    
    //------------------------------------------------------------------------
    //!  Writes a multiset<_valueT> to a FILE pointer.  Returns 1 on success,
    //!  0 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Compare, typename _Alloc>
    static size_t
    Write(FILE *f, const std::multiset<_valueT, _Compare, _Alloc> & l)
    {
      return(ContainerWrite<std::multiset<_valueT, _Compare, _Alloc> >(f, l));
    }

    //------------------------------------------------------------------------
    //!  Reads a multiset<_valueT> from a file descriptor.  Returns the number
    //!  of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Compare, typename _Alloc>
    static ssize_t Read(int fd, std::multiset<_valueT, _Compare, _Alloc> & l)
    {
      return(ContainerRead<std::multiset<_valueT, _Compare, _Alloc> >(fd, l));
    }

    //------------------------------------------------------------------------
    //!  Writes a multiset<_valueT> to a file descriptor.  Returns the number
    //!  of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Compare, typename _Alloc>
    static ssize_t
    Write(int fd, const std::multiset<_valueT, _Compare, _Alloc> & l)
    {
      return(ContainerWrite<std::multiset<_valueT, _Compare, _Alloc> >(fd, l));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    template <typename T, typename... Args>
    static uint32_t VarStreamedLength(const T & t, Args... args)
    {
      uint32_t rc = StreamedLength(t);
      if (sizeof...(Args) > 0)
        rc += VarStreamedLength(args...);
      return(rc);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    template <typename T, typename... Args>
    static std::ostream & VarWrite(std::ostream & os, 
                                const T & t, Args... args)
    {
      if (Write(os, t))
        if (sizeof...(Args) > 0)
          Write(os, args...);
      return(os);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    template <typename T, typename... Args>
    static std::istream & VarRead(std::istream & is, 
                                  const T & t, Args... args)
    {
      if (Read(is, t))
        if (sizeof...(Args) > 0)
          Read(is, args...);
      return(is);
    }

    //------------------------------------------------------------------------
    //!  Returns the number of bytes that should be written if we call one
    //!  of the Write() members for a tuple.
    //------------------------------------------------------------------------
    template <typename T, typename... Args>
    static uint32_t StreamedLength(const std::tuple<T, Args...> & t)
    {
      return(TupleStreamedLength<std::tuple<T, Args...> >(t));
    }
    
    //------------------------------------------------------------------------
    //!  Reads a tuple from an istream.  Returns the istream.
    //------------------------------------------------------------------------
    template <typename T, typename... Args>
    static std::istream & Read(std::istream & is, 
                               std::tuple<T, Args...> & t)
    {
      return(ReadTuple<std::tuple<T, Args...> >(is, t));
    }

    //------------------------------------------------------------------------
    //!  Writes a tuple to an ostream.  Returns the ostream.
    //------------------------------------------------------------------------
    template <typename T, typename... Args>
    static std::ostream & Write(std::ostream & os, 
                                const std::tuple<T, Args...> & t)
    {
      return(WriteTuple<std::tuple<T, Args...> >(os, t));
    }

    //------------------------------------------------------------------------
    //!  Reads a tuple from a file descriptor.  Returns the number of
    //!  bytes read.
    //------------------------------------------------------------------------
    template <typename T, typename... Args>
    static ssize_t Read(int fd, std::tuple<T, Args...> & t)
    {
      return(ReadTuple<std::tuple<T, Args...> >(fd, t));
    }

    //------------------------------------------------------------------------
    //!  Writes a tuple to a file descriptor.  Returns the number of
    //!  bytes written.
    //------------------------------------------------------------------------
    template <typename T, typename... Args>
    static ssize_t Write(int fd, const std::tuple<T, Args...> & t)
    {
      return(WriteTuple<std::tuple<T, Args...> >(fd, t));
    }

    //------------------------------------------------------------------------
    //!  Reads a tuple from a FILE.  Returns 1 on success, 0 on failure.
    //------------------------------------------------------------------------
    template <typename T, typename... Args>
    static size_t Read(FILE *f, std::tuple<T, Args...> & t)
    {
      return(ReadTuple<std::tuple<T, Args...> >(f, t));
    }

    //------------------------------------------------------------------------
    //!  Writes a tuple to a FILE.  Returns 1 on success, 0 on failure.
    //------------------------------------------------------------------------
    template <typename T, typename... Args>
    static size_t Write(FILE *f, const std::tuple<T, Args...> & t)
    {
      return(WriteTuple<std::tuple<T, Args...> >(f, t));
    }

    //------------------------------------------------------------------------
    //!  T must be a tuple.
    //------------------------------------------------------------------------
    template <typename T>
    static std::istream & ReadTuple(std::istream & is, T & t)
    {
      return(TupleIOHelper<T,std::tuple_size<T>::value-1>::Read(is, t));
    }

    //------------------------------------------------------------------------
    //!  T must be a tuple.
    //------------------------------------------------------------------------
    template <typename T>
    static std::ostream & WriteTuple(std::ostream & os, const T & t)
    {
      return(TupleIOHelper<T,std::tuple_size<T>::value-1>::Write(os, t));
    }

    //------------------------------------------------------------------------
    //!  T must be a tuple.
    //------------------------------------------------------------------------
    template <typename T>
    static ssize_t ReadTuple(int fd, T & t)
    {
      return(TupleIOHelper<T,std::tuple_size<T>::value-1>::Read(fd, t));
    }

    //------------------------------------------------------------------------
    //!  T must be a tuple.
    //------------------------------------------------------------------------
    template <typename T>
    static ssize_t WriteTuple(int fd, const T & t)
    {
      return(TupleIOHelper<T,std::tuple_size<T>::value-1>::Write(fd, t));
    }

    //------------------------------------------------------------------------
    //!  T must be a tuple.
    //------------------------------------------------------------------------
    template <typename T>
    static size_t ReadTuple(FILE *f, T & t)
    {
      if (TupleIOHelper<T,std::tuple_size<T>::value-1>::Read(f, t) > 0)
        return(1);
      return(0);
    }

    //------------------------------------------------------------------------
    //!  T must be a tuple.
    //------------------------------------------------------------------------
    template <typename T>
    static size_t WriteTuple(FILE *f, const T & t)
    {
      if (TupleIOHelper<T,std::tuple_size<T>::value-1>::Write(f, t) > 0)
        return(1);
      return(0);
    }

    //------------------------------------------------------------------------
    //!  T must be a tuple.
    //------------------------------------------------------------------------
    template <typename T>
    static uint32_t TupleStreamedLength(const T & t)
    {
      return(TupleIOHelper<T,std::tuple_size<T>::value-1>::StreamedLength(t));
    }

  private:
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static ssize_t Read(int fd, void *buf, size_t buflen);

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    static ssize_t Write(int fd, const void *buf, size_t buflen);
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    template <typename _inputIteratorT>
    static uint32_t StreamedLength(_inputIteratorT f, _inputIteratorT l)
    {
      uint32_t  rc = 0;
      for ( ; f != l; ++f)
        rc += StreamedLength(*f);
      return(rc);
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    template <typename _inputIteratorT>
    static std::ostream & Write(std::ostream & os,
                                _inputIteratorT f, _inputIteratorT l)
    {
      if (os) {
        for ( ; f != l; ++f) {
          if (! Write(os, *f)) {
            break;
          }
        }
      }
      return(os);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    template <typename _inputIteratorT>
    static size_t Write(FILE *file, _inputIteratorT f, _inputIteratorT l)
    {
      size_t  rc = 0;
      if (file) {
        for ( ; f != l; ++f) {
          if (Write(file, *f) <= 0) {
            break;
          }
        }
        if (f == l)
          rc = 1;
      }
      return(rc);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    template <typename _inputIteratorT>
    static ssize_t Write(int fd, _inputIteratorT f, _inputIteratorT l)
    {
      ssize_t  rc = 0;
      if (fd >= 0) {
        for ( ; f != l; ++f) {
          ssize_t  bytesWritten = Write(fd, *f);
          if (bytesWritten > 0) {
            rc += bytesWritten;
          }
          else {
            rc = -1;
            break;
          }
        }
      }
      return(rc);
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    template <typename _containerT>
    static uint32_t ContainerStreamedLength(const _containerT & c)
    {
      uint32_t  rc = sizeof(uint32_t);  // for size()

      rc += StreamedLength<typename _containerT::const_iterator>(c.begin(), 
                                                                 c.end());
      return(rc);
    }
    
    //------------------------------------------------------------------------
    //!  Reads a _containerT from an istream.  Returns the istream.
    //!  We use this for deques, lists, vectors, sets and multisets.
    //------------------------------------------------------------------------
    template <typename _containerT>
    static std::istream & ContainerRead(std::istream & is,
                                        _containerT & c)
    {
      if (! c.empty())
        c.clear();
      if (is) {
        uint32_t  numEntries;
        if (Read(is, numEntries)) {
          for (uint32_t i = 0; i < numEntries; ++i) {
            typename _containerT::value_type  val;
            if (! Read(is, val))
              break;
            c.insert(c.end(), std::move(val));
          }
        }
      }
      return(is);
    }

    //------------------------------------------------------------------------
    //!  Writes a container to an ostream.  Returns the ostream.
    //!  We use this for all containers.
    //------------------------------------------------------------------------
    template <typename _containerT>
    static std::ostream & ContainerWrite(std::ostream & os, 
                                         const _containerT & c)
    {
      if (os) {
        uint32_t  numEntries = c.size();
        if (Write(os, numEntries)) {
          if (numEntries) {
            Write<typename _containerT::const_iterator>(os, 
                                                        c.begin(), 
                                                        c.end());
          }
        }
      }
      return(os);
    }

    //------------------------------------------------------------------------
    //!  Reads a container from a FILE pointer.  Returns 1 on success,
    //!  0 on failure.
    //!  We use this for deques, lists, vectors, sets and multisets.
    //------------------------------------------------------------------------
    template <typename _containerT>
    static size_t ContainerRead(FILE *f, _containerT & c)
    {
      size_t  rc = 0;
      if (! c.empty())
        c.clear();
      if (f) {
        uint32_t  numEntries;
        if (Read(f, numEntries)) {
          uint32_t  i = 0;
          for ( ; i < numEntries; ++i) {
            typename _containerT::value_type  val;
            if (Read(f, val) > 0)
              c.insert(c.end(), std::move(val));
            else
              break;
          }
          if (i == numEntries)
            rc = 1;
        }
      }
      return(rc);
    }
    
    //------------------------------------------------------------------------
    //!  Writes a container to a FILE pointer.  Returns 1 on success, 
    //!  0 on failure.
    //!  We use this for all containers.
    //------------------------------------------------------------------------
    template <typename _containerT>
    static size_t ContainerWrite(FILE *f, const _containerT & c)
    {
      size_t  rc = 0;
      if (f) {
        uint32_t  numEntries = c.size();
        if (Write(f, numEntries)) {
          if (numEntries) {
            rc = Write<typename _containerT::const_iterator>(f, 
                                                             c.begin(), c.end());
          }
          else {
            rc = 1;
          }
        }
      }
      return(rc);
    }
    
    //------------------------------------------------------------------------
    //!  Reads a container from a file descriptor.  Returns the number of
    //!  bytes read on success, -1 on failure.
    //!  We use this for deques, lists, vectors, sets and multisets.
    //------------------------------------------------------------------------
    template <typename _containerT>
    static ssize_t ContainerRead(int fd, _containerT & c)
    {
      if (! c.empty())
        c.clear();
      ssize_t  rc = -1;
      if (fd >= 0) {
        uint32_t  numEntries;
        ssize_t  bytesRead = Read(fd, numEntries);
        if (bytesRead == sizeof(numEntries)) {
          rc = bytesRead;
          for (uint32_t i = 0; i < numEntries; ++i) {
            typename _containerT::value_type  val;
            bytesRead = Read(fd, val);
            if (bytesRead > 0) {
              rc += bytesRead;
              c.insert(c.end(), std::move(val));
            }
            else {
              rc = -1;
              break;
            }
          }
        }
      }
      return(rc);
    }

    //------------------------------------------------------------------------
    //!  Writes a container to a file descriptor.  Returns the number of
    //!  bytes written on success, -1 on failure.
    //!  We use this for all containers.  
    //------------------------------------------------------------------------
    template <typename _containerT>
    static ssize_t ContainerWrite(int fd, const _containerT & c)
    {
      ssize_t  rc = -1;
      if (fd >= 0) {
        uint32_t  numEntries = c.size();
        uint32_t  bytesWritten = Write(fd, numEntries);
        if (bytesWritten == sizeof(numEntries)) {
          rc = bytesWritten;
          if (numEntries) {
            bytesWritten = 
              Write<typename _containerT::const_iterator>(fd,
                                                          c.begin(), 
                                                          c.end());
            if (bytesWritten > 0)
              rc += bytesWritten;
            else
              rc = -1;
          }
        }
      }
      return(rc);
    }

    
    //------------------------------------------------------------------------
    //!  Reads a PairAssociative container from an istream.  Returns the
    //!  istream.
    //!  We use this for map, multimap and hash_map.
    //------------------------------------------------------------------------
    template <typename _containerT>
    static std::istream & 
    PairAssocContRead(std::istream & is, _containerT & m)
    {
      if (! m.empty())
        m.clear();
      if (is) {
        uint32_t  numEntries;
        if (Read(is, numEntries)) {
          for (uint32_t i = 0; i < numEntries; ++i) {
            typename _containerT::key_type  key;
            if (Read(is, key)) {
              typename _containerT::mapped_type  val;
              if (Read(is, val))
                m.insert(typename _containerT::value_type(std::move(key), std::move(val)));
              else
                break;
            }
            else
              break;
          }
        }
      }
      return(is);
    }

    //------------------------------------------------------------------------
    //!  Reads a PairAssociative container from a FILE pointer.  Returns 1
    //!  on success, 0 on failure.
    //!  We use this for map, multimap and hash_map.
    //------------------------------------------------------------------------
    template <typename _containerT>
    static size_t PairAssocContRead(FILE *f, _containerT & m)
    {
      ssize_t  rc = 0;
      if (! m.empty())
        m.clear();
      if (f) {
        uint32_t  numEntries;
        if (Read(f, numEntries)) {
          uint32_t i = 0;
          for ( ; i < numEntries; ++i) {
            typename _containerT::key_type  key;
            if (Read(f, key) > 0) {
              typename _containerT::mapped_type  val;
              if (Read(f, val) > 0) {
                m.insert(typename _containerT::value_type(std::move(key),
                                                          std::move(val)));
              }
              else {
                break;
              }
            }
            else {
              break;
            }
          }
          if (i == numEntries)
            rc = 1;
        }
      }
      return(rc);
    }

    //------------------------------------------------------------------------
    //!  Reads a PairAssociative container from a file descriptor.  Returns
    //!  the number of bytes read on success, -1 on failure.
    //!  We use this for map, multimap and hash_map.
    //------------------------------------------------------------------------
    template <typename _containerT>
    static ssize_t PairAssocContRead(int fd, _containerT & m)
    {
      ssize_t  rc = -1;
      if (! m.empty())
        m.clear();
      if (fd >= 0) {
        uint32_t  numEntries;
        ssize_t  bytesRead = Read(fd, numEntries);
        if (bytesRead == sizeof(numEntries)) {
          rc = bytesRead;
          for (uint32_t i = 0; i < numEntries; ++i) {
            typename _containerT::key_type  key;
            bytesRead = Read(fd, key);
            if (bytesRead > 0) {
              rc += bytesRead;
              typename _containerT::mapped_type  val;
              bytesRead = Read(fd, val);
              if (bytesRead > 0) {
                rc += bytesRead;
                m.insert(typename _containerT::value_type(std::move(key),
                                                          std::move(val)));
              }
              else {
                rc = -1;
                break;
              }
            }
            else {
              rc = -1;
              break;
            }
          }
        }
      }
      return(rc);
    }

    //------------------------------------------------------------------------
    //!  Declare tuple IO helper class template.  elt is the last element
    //!  index (size of the tuple minus 1).
    //------------------------------------------------------------------------
    template <typename T, size_t elt>
    class TupleIOHelper;
    
    //------------------------------------------------------------------------
    //!  Specialization for a tuple with one element.
    //------------------------------------------------------------------------
    template <typename T>
    class TupleIOHelper<T, 0>
    {
    public:
      //----------------------------------------------------------------------
      //!  Read a tuple \c t from an istream \c is.
      //----------------------------------------------------------------------
      static std::istream & Read(std::istream & is, T & t)
      {
        return(IO::Read(is, std::get<0>(t)));
      }
      
      //----------------------------------------------------------------------
      //!  Write a tuple \c t to an ostream \c os.
      //----------------------------------------------------------------------
      static std::ostream & Write(std::ostream & os, const T & t)
      {
        return(IO::Write(os, std::get<0>(t)));
      }
      
      //----------------------------------------------------------------------
      //!  Read a tuple \c t from a file descriptor \c fd.  Returns the
      //!  number of bytes read on success, -1 on failure.
      //----------------------------------------------------------------------
      static ssize_t Read(int fd, T & t)
      {
        return(IO::Read(fd, std::get<0>(t)));
      }
      
      //----------------------------------------------------------------------
      //!  Write a tuple \c t to a file descriptor \c fd.  Returns the
      //!  number of bytes written on success, -1 on failure.
      //----------------------------------------------------------------------
      static ssize_t Write(int fd, const T & t)
      {
        return(IO::Write(fd, std::get<0>(t)));
      }
      
      //----------------------------------------------------------------------
      //!  Read a tuple \c t from a FILE \c f.  Returns 1 on success, 
      //!  0 on failure.
      //----------------------------------------------------------------------
      static size_t Read(FILE *f, T & t)
      {
        return(IO::Read(f, std::get<0>(t)));
      }
      
      //----------------------------------------------------------------------
      //!  Write a tuple \c t to a FILE \c f.  Returns 1 on success,
      //!  0 on failure.
      //----------------------------------------------------------------------
      static size_t Write(FILE *f, const T & t)
      {
        return(IO::Write(f, std::get<0>(t)));
      }
      
      //----------------------------------------------------------------------
      //!  Return the number of bytes that will be written if we write the
      //!  tuple \c t.
      //----------------------------------------------------------------------
      static uint32_t StreamedLength(const T & t)
      {
        return(IO::StreamedLength(std::get<0>(t)));
      }
    };
    
    //------------------------------------------------------------------------
    //!  The recursive tuple IO helper template.
    //------------------------------------------------------------------------
    template <typename T, size_t elt>
    class TupleIOHelper
    {
    public:
      //----------------------------------------------------------------------
      //!  Read a tuple \c t from an istream \c is.
      //----------------------------------------------------------------------
      static std::istream & Read(std::istream & is, T & t)
      {
        if (TupleIOHelper<T,elt-1>::Read(is, t))
          IO::Read(is, std::get<elt>(t));
        return(is);
      }
      
      //----------------------------------------------------------------------
      //!  Write a tuple \c t to an ostream \c os.
      //----------------------------------------------------------------------
      static std::ostream & Write(std::ostream & os, const T & t)
      {
        if (TupleIOHelper<T,elt-1>::Write(os, t))
          IO::Write(os, std::get<elt>(t));
        return(os);
      }
      
      //----------------------------------------------------------------------
      //!  Read a tuple \c t from a file descriptor \c fd.
      //----------------------------------------------------------------------
      static ssize_t Read(int fd, T & t)
      {
        ssize_t  rc = TupleIOHelper<T,elt-1>::Read(fd, t);
        rc += IO::Read(fd, std::get<elt>(t));
        return(rc);
      }
      
      //----------------------------------------------------------------------
      //!  Write a tuple \c t to a file descriptor \c fd.
      //----------------------------------------------------------------------
      static ssize_t Write(int fd, const T & t)
      {
        ssize_t  rc = TupleIOHelper<T,elt-1>::Write(fd, t);
        rc += IO::Write(fd, std::get<elt>(t));
        return(rc);
      }
      
      //----------------------------------------------------------------------
      //!  Read a tuple \c t from a FILE \c f.
      //----------------------------------------------------------------------
      static size_t Read(FILE *f, T & t)
      {
        size_t  rc = TupleIOHelper<T,elt-1>::Read(f, t);
        if (rc > 0)
          rc = IO::Read(f, std::get<elt>(t));
        return(rc);
      }
      
      //----------------------------------------------------------------------
      //!  Write a tuple \c t to a FILE \c f.
      //----------------------------------------------------------------------
      static size_t Write(FILE *f, const T & t)
      {
        size_t  rc = TupleIOHelper<T,elt-1>::Write(f, t);
        if (rc > 0) {
          rc = IO::Write(f, std::get<elt>(t));
        }
        
        return(rc);
      }
      
      //----------------------------------------------------------------------
      //!  Return the number of bytes that will be written if we write the
      //!  tuple \c t.
      //----------------------------------------------------------------------
      static uint32_t StreamedLength(const T & t)
      {
        uint32_t  rc = TupleIOHelper<T,elt-1>::StreamedLength(t);
        rc += IO::StreamedLength(std::get<elt>(t));
        return(rc);
      }
    };

  public:
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Hash, typename _Pred, typename _Alloc>
    static uint32_t
    StreamedLength(const std::unordered_map<_keyT, _valueT, _Hash, _Pred, _Alloc> & m)
    {
      return(ContainerStreamedLength<std::unordered_map<_keyT, _valueT, _Hash, _Pred, _Alloc> >(m));
    }
    
    //------------------------------------------------------------------------
    //!  Reads an unordered_map<_keyT,_valueT> from an istream.  Returns 
    //!  the istream.
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Hash, typename _Pred, typename _Alloc>
    static std::istream & 
    Read(std::istream & is,
         std::unordered_map<_keyT, _valueT, _Hash, _Pred, _Alloc> & m)
    {
      return(PairAssocContRead<std::unordered_map<_keyT, _valueT, _Hash, _Pred, _Alloc> >(is, m));
    }

    //------------------------------------------------------------------------
    //!  Writes a unordered_map<_keyT,_valueT> to an ostream.  Returns the
    //!  ostream.
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Hash, typename _Pred, typename _Alloc>
    static std::ostream & 
    Write(std::ostream & os,
          const std::unordered_map<_keyT, _valueT, _Hash, _Pred, _Alloc> & m)
    {
      return(ContainerWrite<std::unordered_map<_keyT,_valueT,_Hash,_Pred,_Alloc> >(os, m));
    }

    //------------------------------------------------------------------------
    //!  Reads an unordered_map<_keyT,_valueT> from a file descriptor.
    //!  Returns the number of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Hash, typename _Pred, typename _Alloc>
    static ssize_t 
    Read(int fd, std::unordered_map<_keyT, _valueT, _Hash, _Pred, _Alloc> & m)
    {
      return(PairAssocContRead<std::unordered_map<_keyT, _valueT, _Hash, _Pred, _Alloc> >(fd, m));
    }

    //------------------------------------------------------------------------
    //!  Writes an unordered_map<_keyT,_valueT> to a file descriptor. 
    //!  Returns the number of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template<typename _keyT, typename _valueT, 
             typename _Hash, typename _Pred, typename _Alloc>
    static ssize_t 
    Write(int fd, 
          const std::unordered_map<_keyT,_valueT,_Hash,_Pred,_Alloc> & m)
    {
      return(ContainerWrite<std::unordered_map<_keyT,_valueT,_Hash,_Pred,_Alloc> >(fd, m));
    }
    
    //------------------------------------------------------------------------
    //!  Reads an unordered_map from a FILE pointer.  Returns 1 on success,
    //!  0 on failure.
    //------------------------------------------------------------------------
    template<typename _keyT, typename _valueT, typename _Hash,
             typename _Pred, typename _Alloc>
    static size_t
    Read(FILE *f, std::unordered_map<_keyT,_valueT,_Hash,_Pred,_Alloc> & hm)
    {
      return(PairAssocContRead<std::unordered_map<_keyT,_valueT,_Hash,_Pred,_Alloc> >(f, hm));
    }

    //------------------------------------------------------------------------
    //!  Writes an unordered_map to a FILE pointer.  Returns 1 on success,
    //!  0 on failure.
    //------------------------------------------------------------------------
    template<typename _keyT, typename _valueT, typename _Hash,
             typename _Pred, typename _Alloc>
    static size_t  
    Write(FILE *f, const std::unordered_map<_keyT,_valueT,_Hash,_Pred,_Alloc> & hm)
    {
      return(ContainerWrite<std::unordered_map<_keyT,_valueT,_Hash,_Pred,_Alloc> >(f, hm));
    }

    //========================================================================
    //  I/O functions for unordered_multimap
    //========================================================================

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Hash, typename _Pred, typename _Alloc>
    static uint32_t
    StreamedLength(const std::unordered_multimap<_keyT, _valueT, _Hash, _Pred, _Alloc> & m)
    {
      return(ContainerStreamedLength<std::unordered_multimap<_keyT, _valueT, _Hash, _Pred, _Alloc> >(m));
    }
    
    //------------------------------------------------------------------------
    //!  Reads an unordered_multimap<_keyT,_valueT> from an istream.  Returns 
    //!  the istream.
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Hash, typename _Pred, typename _Alloc>
    static std::istream & Read(std::istream & is,
                               std::unordered_multimap<_keyT, _valueT, _Hash, _Pred, _Alloc> & m)
    {
      return(PairAssocContRead<std::unordered_multimap<_keyT, _valueT, _Hash, _Pred, _Alloc> >(is, m));
    }

    //------------------------------------------------------------------------
    //!  Writes a unordered_multimap<_keyT,_valueT> to an ostream.  Returns 
    //!  the ostream.
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Hash, typename _Pred, typename _Alloc>
    static std::ostream & 
    Write(std::ostream & os,
          const std::unordered_multimap<_keyT, _valueT, _Hash, _Pred, _Alloc> & m)
    {
      return(ContainerWrite<std::unordered_multimap<_keyT,_valueT,_Hash,_Pred,_Alloc> >(os, m));
    }

    //------------------------------------------------------------------------
    //!  Reads an unordered_multimap<_keyT,_valueT> from a file descriptor.
    //!  Returns the number of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _keyT, typename _valueT, 
              typename _Hash, typename _Pred, typename _Alloc>
    static ssize_t 
    Read(int fd, std::unordered_multimap<_keyT, _valueT, _Hash, _Pred, _Alloc> & m)
    {
      return(PairAssocContRead<std::unordered_multimap<_keyT, _valueT, _Hash, _Pred, _Alloc> >(fd, m));
    }

    //------------------------------------------------------------------------
    //!  Writes an unordered_multimap<_keyT,_valueT> to a file descriptor. 
    //!  Returns the number of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template<typename _keyT, typename _valueT, 
             typename _Hash, typename _Pred, typename _Alloc>
    static ssize_t 
    Write(int fd, const std::unordered_multimap<_keyT,_valueT,_Hash,_Pred,_Alloc> & m)
    {
      return(ContainerWrite<std::unordered_multimap<_keyT,_valueT,_Hash,_Pred,_Alloc> >(fd, m));
    }
    
    //------------------------------------------------------------------------
    //!  Reads an unordered_multimap from a FILE pointer.  Returns 1 on
    //!  success, 0 on failure.
    //------------------------------------------------------------------------
    template<typename _keyT, typename _valueT, typename _Hash,
             typename _Pred, typename _Alloc>
    static size_t
    Read(FILE *f, std::unordered_multimap<_keyT,_valueT,_Hash,_Pred,_Alloc> & hm)
    {
      return(PairAssocContRead<std::unordered_multimap<_keyT,_valueT,_Hash,_Pred,_Alloc> >(f, hm));
    }

    //------------------------------------------------------------------------
    //!  Writes an unordered_multimap to a FILE pointer.  Returns 1 on success,
    //!  0 on failure.
    //------------------------------------------------------------------------
    template<typename _keyT, typename _valueT, typename _Hash,
             typename _Pred, typename _Alloc>
    static size_t  
    Write(FILE *f, const std::unordered_multimap<_keyT,_valueT,_Hash,_Pred,_Alloc> & hm)
    {
      return(ContainerWrite<std::unordered_multimap<_keyT,_valueT,_Hash,_Pred,_Alloc> >(f, hm));
    }

    //========================================================================
    //  I/O functions for unordered_set
    //========================================================================

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Hash, 
              typename _Pred, typename _Alloc>
    static uint32_t
    StreamedLength(const std::unordered_set<_valueT, _Hash, _Pred, _Alloc> & m)
    {
      return(ContainerStreamedLength<std::unordered_set<_valueT, _Hash, _Pred, _Alloc> >(m));
    }
    
    //------------------------------------------------------------------------
    //!  Reads an unordered_set<_valueT> from an istream.  Returns 
    //!  the istream.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Hash, 
              typename _Pred, typename _Alloc>
    static std::istream & 
    Read(std::istream & is,
         std::unordered_set<_valueT, _Hash, _Pred, _Alloc> & m)
    {
      return(ContainerRead<std::unordered_set<_valueT, _Hash, _Pred, _Alloc> >(is, m));
    }

    //------------------------------------------------------------------------
    //!  Writes a unordered_set<_valueT> to an ostream.  Returns the
    //!  ostream.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Hash,
              typename _Pred, typename _Alloc>
    static std::ostream & 
    Write(std::ostream & os,
          const std::unordered_set<_valueT, _Hash, _Pred, _Alloc> & m)
    {
      return(ContainerWrite<std::unordered_set<_valueT,_Hash,_Pred,_Alloc> >(os, m));
    }

    //------------------------------------------------------------------------
    //!  Reads an unordered_set<_valueT> from a file descriptor.
    //!  Returns the number of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Hash,
              typename _Pred, typename _Alloc>
    static ssize_t 
    Read(int fd, std::unordered_set<_valueT, _Hash, _Pred, _Alloc> & m)
    {
      return(ContainerRead<std::unordered_set<_valueT, _Hash, _Pred, _Alloc> >(fd, m));
    }

    //------------------------------------------------------------------------
    //!  Writes an unordered_set<_valueT> to a file descriptor. 
    //!  Returns the number of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template<typename _valueT, typename _Hash,
             typename _Pred, typename _Alloc>
    static ssize_t 
    Write(int fd, const std::unordered_set<_valueT,_Hash,_Pred,_Alloc> & m)
    {
      return(ContainerWrite<std::unordered_set<_valueT,_Hash,_Pred,_Alloc> >(fd, m));
    }
    
    //------------------------------------------------------------------------
    //!  Reads an unordered_set from a FILE pointer.  Returns 1 on success,
    //!  0 on failure.
    //------------------------------------------------------------------------
    template<typename _valueT, typename _Hash,
             typename _Pred, typename _Alloc>
    static size_t
    Read(FILE *f, std::unordered_set<_valueT,_Hash,_Pred,_Alloc> & hm)
    {
      return(ContainerRead<std::unordered_set<_valueT,_Hash,_Pred,_Alloc> >(f, hm));
    }

    //------------------------------------------------------------------------
    //!  Writes an unordered_set to a FILE pointer.  Returns 1 on success,
    //!  0 on failure.
    //------------------------------------------------------------------------
    template<typename _valueT, typename _Hash,
             typename _Pred, typename _Alloc>
    static size_t  
    Write(FILE *f, const std::unordered_set<_valueT,_Hash,_Pred,_Alloc> & hm)
    {
      return(ContainerWrite<std::unordered_set<_valueT,_Hash,_Pred,_Alloc> >(f, hm));
    }

    //========================================================================
    //  I/O functions for unordered_multiset
    //========================================================================

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Hash, 
              typename _Pred, typename _Alloc>
    static uint32_t
    StreamedLength(const std::unordered_multiset<_valueT, _Hash, _Pred, _Alloc> & m)
    {
      return(ContainerStreamedLength<std::unordered_multiset<_valueT, _Hash, _Pred, _Alloc> >(m));
    }
    
    //------------------------------------------------------------------------
    //!  Reads an unordered_multiset<_valueT> from an istream.  Returns 
    //!  the istream.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Hash, 
              typename _Pred, typename _Alloc>
    static std::istream & 
    Read(std::istream & is,
         std::unordered_multiset<_valueT, _Hash, _Pred, _Alloc> & m)
    {
      return(ContainerRead<std::unordered_multiset<_valueT, _Hash, _Pred, _Alloc> >(is, m));
    }

    //------------------------------------------------------------------------
    //!  Writes a unordered_multiset<_valueT> to an ostream.  Returns the
    //!  ostream.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Hash,
              typename _Pred, typename _Alloc>
    static std::ostream & 
    Write(std::ostream & os,
          const std::unordered_multiset<_valueT, _Hash, _Pred, _Alloc> & m)
    {
      return(ContainerWrite<std::unordered_multiset<_valueT,_Hash,_Pred,_Alloc> >(os, m));
    }

    //------------------------------------------------------------------------
    //!  Reads an unordered_multiset<_valueT> from a file descriptor.
    //!  Returns the number of bytes read on success, -1 on failure.
    //------------------------------------------------------------------------
    template <typename _valueT, typename _Hash,
              typename _Pred, typename _Alloc>
    static ssize_t 
    Read(int fd, std::unordered_multiset<_valueT, _Hash, _Pred, _Alloc> & m)
    {
      return(ContainerRead<std::unordered_multiset<_valueT, _Hash, _Pred, _Alloc> >(fd, m));
    }

    //------------------------------------------------------------------------
    //!  Writes an unordered_multiset<_valueT> to a file descriptor. 
    //!  Returns the number of bytes written on success, -1 on failure.
    //------------------------------------------------------------------------
    template<typename _valueT, typename _Hash,
             typename _Pred, typename _Alloc>
    static ssize_t 
    Write(int fd, const std::unordered_multiset<_valueT,_Hash,_Pred,_Alloc> & m)
    {
      return(ContainerWrite<std::unordered_multiset<_valueT,_Hash,_Pred,_Alloc> >(fd, m));
    }
    
    //------------------------------------------------------------------------
    //!  Reads an unordered_multiset from a FILE pointer.  Returns 1 on
    //!  success, 0 on failure.
    //------------------------------------------------------------------------
    template<typename _valueT, typename _Hash,
             typename _Pred, typename _Alloc>
    static size_t
    Read(FILE *f, std::unordered_multiset<_valueT,_Hash,_Pred,_Alloc> & hm)
    {
      return(ContainerRead<std::unordered_multiset<_valueT,_Hash,_Pred,_Alloc> >(f, hm));
    }

    //------------------------------------------------------------------------
    //!  Writes an unordered_multiset to a FILE pointer.  Returns 1 on 
    //!  success, 0 on failure.
    //------------------------------------------------------------------------
    template<typename _valueT, typename _Hash,
             typename _Pred, typename _Alloc>
    static size_t  
    Write(FILE *f, const std::unordered_multiset<_valueT,_Hash,_Pred,_Alloc> & hm)
    {
      return(ContainerWrite<std::unordered_multiset<_valueT,_Hash,_Pred,_Alloc> >(f, hm));
    }

  };


}  // namespace Dwm

#endif  // _DWMIO_HH_

//---------------------------- emacs settings -----------------------------
//  Local Variables:
//  mode: C++
//  tab-width: 2
//  indent-tabs-mode: nil
//  c-basic-offset: 2
//  End:
//-------------------------------------------------------------------------
