//===========================================================================
// @(#) $Name$
// @(#) $Id: DwmBZ2IO.cc 9111 2017-04-14 22:22:47Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2004-2007
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR
//  PURPOSE, OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY 
//  PATENT, TRADEMARK OR OTHER RIGHTS.
//===========================================================================

extern "C" {
  #include <sys/types.h>
  #ifdef __FreeBSD__
    #include <sys/endian.h>
  #endif
  #include <unistd.h>
  #include <stdlib.h>
  #include <netinet/in.h>
  #include <rpc/rpc.h>
  #include <rpc/xdr.h>

  #ifdef __OpenBSD__
    #define be64toh  betoh64
  #endif
}

#include <cassert>

#include "DwmBZ2IO.hh"
#include "DwmSvnTag.hh"
#include "DwmPortability.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/libDwm/tags/libDwm-0.6.11/src/DwmBZ2IO.cc 9111 $");

namespace Dwm {

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  static uint32_t XdrEncode(float val, char **buf)
  {
    float value = val;
    XDR   xdrs;
    if (*buf)
      free(*buf);
    *buf = (char *)malloc(4);
    xdrmem_create(&xdrs,*buf,4,XDR_ENCODE);
    xdr_float(&xdrs,&value);
    xdr_destroy((&xdrs));
    return(4);
  }
  
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  static uint32_t XdrEncode(double val, char **buf)
  {
    double  value = val;
    XDR   xdrs;
    if (*buf)
      free(*buf);
    *buf = (char *)malloc(8);
    xdrmem_create(&xdrs,*buf,8,XDR_ENCODE);
    xdr_double(&xdrs,&value);
    xdr_destroy((&xdrs));
    return(8);
  }
  
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  static void XdrDecode(char *buf, float & val)
  {
    assert(buf);

    float  value;
    XDR    xdrs;
    xdrmem_create(&xdrs,buf,4,XDR_DECODE);
    xdr_float(&xdrs,&value);
    val = value;
    xdr_destroy((&xdrs));

    return;
  }

  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  static void XdrDecode(char *buf, double & val)
  {
    assert(buf);

    double  value;
    XDR     xdrs;
    xdrmem_create(&xdrs,buf,8,XDR_DECODE);
    xdr_double(&xdrs,&value);
    val = value;
    xdr_destroy((&xdrs));

    return;
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  int BZ2IO::BZWrite(BZFILE *bzf, char c)
  {
    int  rc = -1;
    if (bzf)
      rc = BZ2_bzwrite(bzf,(void *)&c,sizeof(c));
    return(rc);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  int BZ2IO::BZWrite(BZFILE *bzf, uint8_t c)
  {
    int  rc = -1;
    if (bzf)
      rc = BZ2_bzwrite(bzf,(void *)&c,sizeof(c));
    return(rc);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  int BZ2IO::BZWrite(BZFILE *bzf, int16_t val)
  {
    int  rc = -1;
    if (bzf) {
      int16_t v = htons(val);
      rc = BZ2_bzwrite(bzf,(void *)&v,sizeof(v));
    }
    return(rc);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  int BZ2IO::BZWrite(BZFILE *bzf, uint16_t val)
  {
    int  rc = -1;
    if (bzf) {
      uint16_t v = htons(val);
      rc = BZ2_bzwrite(bzf,(void *)&v,sizeof(v));
    }
    return(rc);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  int BZ2IO::BZWrite(BZFILE *bzf, int32_t val)
  {
    int  rc = -1;
    if (bzf) {
      int32_t v = htonl(val);
      rc = BZ2_bzwrite(bzf,(void *)&v,sizeof(v));
    }
    return(rc);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  int BZ2IO::BZWrite(BZFILE *bzf, uint32_t val)
  {
    int  rc = -1;
    if (bzf) {
      uint32_t v = htonl(val);
      rc = BZ2_bzwrite(bzf,(void *)&v,sizeof(v));
    }
    return(rc);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  int BZ2IO::BZWrite(BZFILE *bzf, const int64_t & val)
  {
    int  rc = -1;
    if (bzf) {
      int64_t v = htobe64(val);
      rc = BZ2_bzwrite(bzf,(void *)&v,sizeof(v));
      if (rc != sizeof(v))
        rc = -1;
    }
    return(rc);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  int BZ2IO::BZWrite(BZFILE *bzf, const uint64_t & val)
  {
    int  rc = -1;
    if (bzf) {
      uint64_t v = htobe64(val);
      rc = BZ2_bzwrite(bzf,(void *)&v,sizeof(v));
      if (rc != sizeof(v))
        rc = -1;
    }
    return(rc);
  }
  
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  int BZ2IO::BZWrite(BZFILE *bzf, float val)
  {
    if (! bzf)
      return(-1);
    
    char     *buf = 0;
    uint32_t  xe = XdrEncode(val,&buf);
    int   rc = BZ2_bzwrite(bzf,buf,xe);
    free(buf);
    
    if (rc < 4)
      rc = -1;

    return(rc);
  }
  
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  int BZ2IO::BZWrite(BZFILE *bzf, const double & val)
  {
    if (! bzf)
      return(-1);
    
    char  *buf = 0;
    uint32_t  xe = XdrEncode(val,&buf);
    int   rc = BZ2_bzwrite(bzf,buf,xe);

    if (rc < 8)
      rc = -1;

    return(rc);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  int BZ2IO::BZWrite(BZFILE *bzf, const std::string & s)
  {
    int  rc = -1;
    
    if (bzf) {
      int32_t  len = s.length();
      if (BZWrite(bzf,len) == sizeof(len)) {
        rc = sizeof(len);
        if (BZ2_bzwrite(bzf,(void *)s.c_str(),len) == len)
          rc += len;
        else
          rc = -1;
      }
    }
    return(rc);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  int BZ2IO::BZRead(BZFILE *bzf, char & c)
  {
    int  rc = -1;
    if (bzf)
      rc = BZ2_bzread(bzf,(void *)&c,sizeof(c));
    return(rc);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  int BZ2IO::BZRead(BZFILE *bzf, uint8_t & c)
  {
    int  rc = -1;
    if (bzf)
      rc = BZ2_bzread(bzf,(void *)&c,sizeof(c));
    return(rc);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  int BZ2IO::BZRead(BZFILE *bzf, int16_t & val)
  {
    int  rc = -1;
    if (bzf) {
      int16_t  v;
      rc = BZ2_bzread(bzf,(void *)&v,sizeof(v));
      if (rc == sizeof(v))
        val = ntohs(v);
    }
    return(rc);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  int BZ2IO::BZRead(BZFILE *bzf, uint16_t & val)
  {
    int  rc = -1;
    if (bzf) {
      uint16_t  v;
      rc = BZ2_bzread(bzf,(void *)&v,sizeof(v));
      if (rc == sizeof(v))
        val = ntohs(v);
    }
    return(rc);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  int BZ2IO::BZRead(BZFILE *bzf, int32_t & val)
  {
    int  rc = -1;
    if (bzf) {
      int32_t  v;
      rc = BZ2_bzread(bzf,(void *)&v,sizeof(v));
      if (rc == sizeof(v))
        val = ntohl(v);
    }
    return(rc);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  int BZ2IO::BZRead(BZFILE *bzf, uint32_t & val)
  {
    int  rc = -1;
    if (bzf) {
      uint32_t  v;
      rc = BZ2_bzread(bzf,(void *)&v,sizeof(v));
      if (rc == sizeof(v))
        val = ntohl(v);
    }
    return(rc);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  int BZ2IO::BZRead(BZFILE *bzf, int64_t & val)
  {
    int  rc = -1;
    if (bzf) {
      uint64_t  v;
      if (BZ2_bzread(bzf,(void *)&v,sizeof(v)) == sizeof(v)) {
        val = be64toh(v);
        rc = sizeof(v);
      }
    }
    return(rc);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  int BZ2IO::BZRead(BZFILE *bzf, uint64_t & val)
  {
    int  rc = -1;
    if (bzf) {
      uint64_t  v;
      if (BZ2_bzread(bzf,(void *)&v,sizeof(v)) == sizeof(v)) {
        val = be64toh(v);
        rc = sizeof(v);
      }
    }
    return(rc);
  }
  
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  int BZ2IO::BZRead(BZFILE *bzf, float & val)
  {
    int  rc = -1;
    if (bzf) {
      char     buf[4];
      int  rc = BZ2_bzread(bzf,buf,4);
      if (rc == 4) {
        XdrDecode(buf,val);
      }
    }
    return(rc);
  }
  
  //--------------------------------------------------------------------------
  //!  
  //--------------------------------------------------------------------------
  int BZ2IO::BZRead(BZFILE *bzf, double & val)
  {
    int  rc = -1;
    if (bzf) {
      char     buf[8];
      int  rc = BZ2_bzread(bzf,buf,8);
      if (rc == 8) {
        XdrDecode(buf,val);
      }
    }
    return(rc);
  }
  
  //------------------------------------------------------------------------
  //!  
  //------------------------------------------------------------------------
  int BZ2IO::BZRead(BZFILE *bzf, std::string & s)
  {
    int  rc = -1;
    if (bzf) {
      int32_t  len;
      if (BZRead(bzf,len) == sizeof(len)) {
        rc = sizeof(len);
        char  *buf = (char *)calloc(1,len);
        if (buf) {
          if (BZ2_bzread(bzf,(void *)buf,len) == len) {
            rc += len;
            s.assign(buf,len);
          }
          else {
            rc = -1;
          }
          free(buf);
        }
      }
    }
    return(rc);
  }
  

}  // namespace Dwm
