//===========================================================================
// @(#) $DwmPath: dwm/libDwm/tags/libDwm-0.6.11/include/DwmIpv6Address.hh 8401 $
// @(#) $Id: DwmIpv6Address.hh 8401 2016-04-17 06:44:31Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2004-2006, 2016
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR
//  PURPOSE, OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY
//  PATENT, TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmIpv6Address.hh
//!  \brief Dwm::Ipv6Address class definition
//---------------------------------------------------------------------------

#ifndef _DWMIPV6ADDRESS_HH_
#define _DWMIPV6ADDRESS_HH_

extern "C" {
#include <inttypes.h>
#include <netinet/in.h>
}

#include <string>

#include "DwmIpAddress.hh"

namespace Dwm {

  //--------------------------------------------------------------------------
  //!  This class encapsulates an IPv6 address.
  //--------------------------------------------------------------------------
  class Ipv6Address
    : public IpAddress
  {
  public:
    //------------------------------------------------------------------------
    //!  Constructor.  Sets address to 0::
    //------------------------------------------------------------------------
    Ipv6Address();
    
    //------------------------------------------------------------------------
    //!  Construct from a string.
    //------------------------------------------------------------------------
    Ipv6Address(const std::string & addr);

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    Ipv6Address & operator = (const std::string & addr);
    
    //------------------------------------------------------------------------
    //!  Construct from an in6_addr.
    //------------------------------------------------------------------------
    Ipv6Address(const struct in6_addr & addr);

    //------------------------------------------------------------------------
    //!  Returns a string representation.
    //------------------------------------------------------------------------
    operator std::string () const;

    //------------------------------------------------------------------------
    //!  Returns an in6_addr representation.
    //------------------------------------------------------------------------
    operator struct in6_addr () const;
    
    //------------------------------------------------------------------------
    //!  Less-than operator.
    //------------------------------------------------------------------------
    bool operator < (const Ipv6Address & addr) const;

    //------------------------------------------------------------------------
    //!  Greater-than operator.
    //------------------------------------------------------------------------
    bool operator > (const Ipv6Address & addr) const;
    
    //------------------------------------------------------------------------
    //!  Less-than operator for generic IpAddress.
    //------------------------------------------------------------------------
    bool operator < (const IpAddress & addr) const;
    
    //------------------------------------------------------------------------
    //!  Equal-to operator.
    //------------------------------------------------------------------------
    bool operator == (const Ipv6Address & addr) const;

    //------------------------------------------------------------------------
    //!  Mask operator.
    //------------------------------------------------------------------------
    Ipv6Address & operator &= (const Ipv6Address & netmask);
    
    //------------------------------------------------------------------------
    //!  Mask operator.
    //------------------------------------------------------------------------
    Ipv6Address operator & (const Ipv6Address & netmask) const;
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    inline bool IsV6() const
    {
      return(true);
    }

    //------------------------------------------------------------------------
    //!  Returns a pointer to the memory containing the address.
    //------------------------------------------------------------------------
    inline const uint8_t *RawPointer() const
    {
      return (const uint8_t *)&_addr;
    }
    
    //------------------------------------------------------------------------
    //!  Returns the number of bytes that would be written if we called
    //!  one of the Write() members.
    //------------------------------------------------------------------------
    uint32_t StreamedLength() const;
    
    //------------------------------------------------------------------------
    //!  Reads from an istream.  Returns the istream.
    //------------------------------------------------------------------------
    std::istream & Read(std::istream & is);

    //------------------------------------------------------------------------
    //!  Writes to an ostream.  Returns the ostream.
    //------------------------------------------------------------------------
    std::ostream & Write(std::ostream & os) const;

    //------------------------------------------------------------------------
    //!  Reads from a file descriptor.  Returns the number of bytes read
    //!  (16 on success).
    //------------------------------------------------------------------------
    ssize_t Read(int fd);
    
    //------------------------------------------------------------------------
    //!  Writes to a file descriptor.  Returns the number of bytes written
    //!  (16 on success).
    //------------------------------------------------------------------------
    ssize_t Write(int fd) const;

    //------------------------------------------------------------------------
    //!  Reads from a FILE pointer.  Returns 1 on success, 0 on failure.
    //------------------------------------------------------------------------
    size_t Read(FILE * f);
    
    //------------------------------------------------------------------------
    //!  Writes to a FILE pointer.  Returns 1 on success, 0 on failure.
    //------------------------------------------------------------------------
    size_t Write(FILE * f) const;
    
    //------------------------------------------------------------------------
    //!  Reads from a gzFile.  Returns the number of bytes read
    //!  (16 on success).
    //------------------------------------------------------------------------
    int Read(gzFile gzf);
    
    //------------------------------------------------------------------------
    //!  Writes to a gzFile.  Returns the number of bytes written
    //!  (16 on success).
    //------------------------------------------------------------------------
    int Write(gzFile gzf) const;

    //------------------------------------------------------------------------
    //!  Reads from a BZFILE pointer.  Returns the number of bytes read
    //!  (16 on success).
    //------------------------------------------------------------------------
    int BZRead(BZFILE *bzf);
    
    //------------------------------------------------------------------------
    //!  Writes to a BZFILE pointer.  Returns the number of bytes written
    //!  (16 on success).
    //------------------------------------------------------------------------
    int BZWrite(BZFILE *bzf) const;

  private:
    struct in6_addr  _addr;
  };

}  // namespace Dwm

#endif  // _DWMIPV6ADDRESS_HH_

//---------------------------- emacs settings -----------------------------
//  Local Variables:
//  mode: C++/la
//  tab-width: 2
//  indent-tabs-mode: nil
//  c-basic-offset: 2
//  End:
//-------------------------------------------------------------------------
