//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/trunk/classes/src/DwmDnsRRDataSRV.cc 10133 $
// @(#) $Id: DwmDnsRRDataSRV.cc 10133 2018-01-27 17:41:32Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsRRDataSRV.cc
//!  \brief Dwm::Dns::RRDataSRV class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <arpa/inet.h>
}

#include <cstring>
#include <stdexcept>

#include "DwmSvnTag.hh"
#include "DwmDnsLabelSequence.hh"
#include "DwmDnsRRDataSRV.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/trunk/classes/src/DwmDnsRRDataSRV.cc 10133 $");

using namespace std;

namespace Dwm {

  namespace Dns {

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataSRV::RRDataSRV()
        : _priority(0), _weight(0), _port(0), _target()
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    RRDataSRV::RRDataSRV(uint16_t priority, uint16_t weight, uint16_t port,
                         const string & target)
        : _priority(priority), _weight(weight), _port(port), _target(target)
    {}

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool RRDataSRV::operator == (const RRDataSRV & srv) const
    {
      return ((_priority == srv._priority)
              && (_weight == srv._weight)
              && (_port == srv._port)
              && (_target == srv._target));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataSRV::Priority() const
    {
      return _priority;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataSRV::Priority(uint16_t priority)
    {
      _priority = priority;
      return _priority;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataSRV::Weight() const
    {
      return _weight;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataSRV::Weight(uint16_t weight)
    {
      _weight = weight;
      return _weight;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataSRV::Port() const
    {
      return _port;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t RRDataSRV::Port(uint16_t port)
    {
      _port = port;
      return _port;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataSRV::Target() const
    {
      return _target;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & RRDataSRV::Target(const string & target)
    {
      _target = target;
      return _target;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t *RRDataSRV::Encode(uint8_t *pkt, uint8_t *ptr, uint16_t pktlen,
                               LabelPositions & lps) const
    {
      if ((ptr + 6) <= (pkt + pktlen)) {
        uint16_t  val = htons(_priority);
        memcpy(ptr, &val, sizeof(val));
        ptr += sizeof(val);
        val = htons(_weight);
        memcpy(ptr, &val, sizeof(val));
        ptr += sizeof(val);
        val = htons(_port);
        memcpy(ptr, &val, sizeof(val));
        ptr += sizeof(val);
        LabelSequence  ls(_target);
        ptr = ls.Encode(pkt, ptr, pktlen, lps);
      }
      else {
        throw out_of_range("Dwm::Dns::RRDataSRV will not fit in packet");
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const uint8_t *RRDataSRV::Decode(const uint8_t *pkt, const uint8_t *ptr,
                                     uint16_t pktlen, uint16_t rdlen)
    {
      if ((ptr + 6) <= (pkt + pktlen)) {
        memcpy(&_priority, ptr, sizeof(_priority));
        _priority = ntohs(_priority);
        ptr += sizeof(_priority);
        memcpy(&_weight, ptr, sizeof(_weight));
        _weight = ntohs(_weight);
        ptr += sizeof(_weight);
        memcpy(&_port, ptr, sizeof(_port));
        _port = ntohs(_port);
        ptr += sizeof(_port);
        LabelSequence  ls;
        ptr = ls.Decode(pkt, ptr, pktlen);
        _target = (string)ls;
      }
      else {
        throw out_of_range("packet too short to contain Dwm::Dns::RRDataSRV");
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    ostream & operator << (ostream & os, const RRDataSRV & srv)
    {
      if (os) {
        os << srv._priority << ' ' << srv._weight << ' ' << srv._port << ' '
           << srv._target;
      }
      return os;
    }
    
          
  }  // namespace Dns

}  // namespace Dwm
