//===========================================================================
// @(#) $DwmPath: dwm/DwmDns/trunk/classes/src/DwmDnsMessageQuestion.cc 11433 $
// @(#) $Id: DwmDnsMessageQuestion.cc 11433 2020-12-13 06:35:55Z dwm $
//===========================================================================
//  Copyright (c) Daniel W. McRobb 2000, 2016, 2018
//  All rights reserved.
//
//  Redistribution and use in source and binary forms, with or without
//  modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright
//     notice, this list of conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright
//     notice, this list of conditions and the following disclaimer in the
//     documentation and/or other materials provided with the distribution.
//  3. The names of the authors and copyright holders may not be used to
//     endorse or promote products derived from this software without
//     specific prior written permission.
//
//  IN NO EVENT SHALL DANIEL W. MCROBB BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
//  INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE,
//  EVEN IF DANIEL W. MCROBB HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
//
//  THE SOFTWARE PROVIDED HEREIN IS ON AN "AS IS" BASIS, AND
//  DANIEL W. MCROBB HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
//  UPDATES, ENHANCEMENTS, OR MODIFICATIONS. DANIEL W. MCROBB MAKES NO
//  REPRESENTATIONS AND EXTENDS NO WARRANTIES OF ANY KIND, EITHER
//  IMPLIED OR EXPRESS, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE,
//  OR THAT THE USE OF THIS SOFTWARE WILL NOT INFRINGE ANY PATENT,
//  TRADEMARK OR OTHER RIGHTS.
//===========================================================================

//---------------------------------------------------------------------------
//!  \file DwmDnsMessageQuestion.cc
//!  \brief Dwm::Dns::MessageQuestion class implementation
//---------------------------------------------------------------------------

extern "C" {
  #include <arpa/inet.h>
}

#include <cstring>
#include <stdexcept>

#include "DwmSvnTag.hh"
#include "DwmDnsLabelSequence.hh"
#include "DwmDnsMessageQuestion.hh"

static const Dwm::SvnTag svntag("@(#) $DwmPath: dwm/DwmDns/trunk/classes/src/DwmDnsMessageQuestion.cc 11433 $");

using namespace std;

namespace Dwm {

  namespace Dns {

    const map<uint16_t,string>  MessageQuestion::_qtypeNames = {
      { k_typeA,          "A"          },
      { k_typeNS,         "NS"         },
      { k_typeCNAME,      "CNAME"      },
      { k_typeSOA,        "SOA"        },
      { k_typeMB,         "MB"         },
      { k_typeMG,         "MG"         },
      { k_typeMR,         "MR"         },
      { k_typePTR,        "PTR"        },
      { k_typeHINFO,      "HINFO"      },
      { k_typeMINFO,      "MINFO"      },
      { k_typeMX,         "MX"         },
      { k_typeTXT,        "TXT"        },
      { k_typeRP,         "RP"         },
      { k_typeKEY,        "KEY"        },
      { k_typeAAAA,       "AAAA"       },
      { k_typeLOC,        "LOC"        },
      { k_typeSRV,        "SRV"        },
      { k_typeCERT,       "CERT"       },
      { k_typeDS,         "DS"         },
      { k_typeSSHFP,      "SSHFP"      },
      { k_typeRRSIG,      "RRSIG"      },
      { k_typeNSEC,       "NSEC"       },
      { k_typeDNSKEY,     "DNSKEY"     },
      { k_typeDHCID,      "DHCID"      },
      { k_typeNSEC3,      "NSEC3"      },
      { k_typeNSEC3PARAM, "NSEC3PARAM" },
      { k_typeTLSA,       "TLSA"       },
      { k_typeSMIMEA,     "SMIMEA"     },
      { k_typeCDS,        "CDS"        },
      { k_typeCDNSKEY,    "CDNSKEY"    },
      { k_typeOPENPGPKEY, "OPENPGPKEY" },
      { k_typeURI,        "URI"        },
      { k_typeCAA,        "CAA"        }
    };

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    MessageQuestion::MessageQuestion()
        : _qname(), _qtype(0), _qclass(0)
    {}
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    MessageQuestion::MessageQuestion(const string & qname, uint16_t qtype,
                                     uint16_t qclass)
        : _qname(qname), _qtype(qtype), _qclass(qclass)
    {}

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    bool MessageQuestion::operator == (const MessageQuestion & question) const
    {
      return ((_qname == question._qname)
              && (_qtype == question._qtype)
              && (_qclass == question._qclass));
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & MessageQuestion::QName() const
    {
      return _qname;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const string & MessageQuestion::QName(const string & qname)
    {
      _qname = qname;
      return _qname;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t MessageQuestion::QType() const
    {
      return _qtype;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t MessageQuestion::QType(uint16_t qtype)
    {
      _qtype = qtype;
      return _qtype;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    std::string MessageQuestion::QTypeString() const
    {
      std::string  rc;
      auto  it = _qtypeNames.find(_qtype);
      if (it != _qtypeNames.end()) {
        rc = it->second;
      }
      return rc;
    }

    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t MessageQuestion::QClass() const
    {
      return _qclass;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint16_t MessageQuestion::QClass(uint16_t qclass)
    {
      _qclass = qclass;
      return _qclass;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    uint8_t *MessageQuestion::Encode(uint8_t *pkt, uint8_t *ptr,
                                     uint16_t pktlen,
                                     LabelPositions & lps) const
    {
      LabelSequence  ls(_qname);
      ptr = ls.Encode(pkt, ptr, pktlen, lps);
      if ((ptr + 4) <= (pkt + pktlen)) {
        uint16_t  val = htons(_qtype);
        memcpy(ptr, &val, sizeof(val));
        ptr += sizeof(val);
        val = htons(_qclass);
        memcpy(ptr, &val, sizeof(val));
        ptr += sizeof(val);
      }
      else {
        throw out_of_range("Dwm::Dns::MessageHeader will not fit in packet");
      }
      return ptr;
    }
    
    //------------------------------------------------------------------------
    //!  
    //------------------------------------------------------------------------
    const uint8_t *MessageQuestion::Decode(const uint8_t *pkt,
                                           const uint8_t *ptr,
                                           uint16_t pktlen)
    {
      _qname.clear();
      _qtype = 0;
      _qclass = 0;
      LabelSequence  ls;
      ptr = ls.Decode(pkt, ptr, pktlen);
      _qname = (string)ls;
      if ((ptr + 4) <= (pkt + pktlen)) {
        memcpy(&_qtype, ptr, sizeof(_qtype));
        _qtype = ntohs(_qtype);
        ptr += sizeof(_qtype);
        memcpy(&_qclass, ptr, sizeof(_qclass));
        _qclass = ntohs(_qclass);
        ptr += sizeof(_qclass);
      }
      else {
        throw out_of_range("packet too short to contain"
                           " Dwm::Dns::MessageHeader");
      }
      return ptr;
    }
    
    
  }  // namespace Dns

}  // namespace Dwm
